#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{BuddyModel}.
"""

import ecore

from common.singleton import Singleton
from models.gpsmodel import GPSModel
from models.obdmodel import OBDModel
from main.messagedlgctrl import MessageDlgController


class BuddyModel(Singleton):
    """
    Implements buddy-related Infoshare features (handles connected buddies,
    pending requests and data messages).

    @cvar BUDDY_CONNECTING: Buddy connection status flag (buddy connecting).
    @cvar BUDDY_CONNECTED: Buddy connection status flags (buddy connected).
    @cvar MAX_BUDDIES: Maximum number of connecting/connected buddies.
    @cvar BUDDY_REQUEST_TIMEOUT: Buddy request timeout (I{seconds}).
    @cvar USER_REQUEST_TIMEOUT: User request timeout (I{seconds}).
    """

    # Buddy status flags
    (BUDDY_CONNECTING, BUDDY_CONNECTED) = range(2)

    MAX_BUDDIES = 4
    BUDDY_REQUEST_TIMEOUT = 55.0
    USER_REQUEST_TIMEOUT = 60.0

    def __init__(self):
        Singleton.__init__(self)

        self.__buddies = {}
        self.__active_buddy = ''

        self.__buddy_data = {}
        self.__has_view = []
        self.__buddy_connect_cbs = []
        self.__buddy_disconnect_cbs = []
        self.__data_available_cbs = []

        self.__requests = {}
        self.__request_created_cbs = []
        self.__request_removed_cbs = []

    def __get_active_buddy(self):
        """
        Returns the current active buddy (if any), or an empty string instead.

        @rtype: string
        """
        return self.__active_buddy
    active_buddy = property(__get_active_buddy)

    def __get_requests(self):
        """
        Returns pending requests.

        @rtype: string
        """
        return self.__requests.keys()
    requests = property(__get_requests)

    def __get_size(self):
        """
        Returns the number of connecting/connected buddies.

        @rtype: number
        """
        return len(self.__buddies)
    size = property(__get_size)

    def __request_timeout(self, name, msg=None):
        """
        Called when a received or sent request has expired.

        @type   name:   string
        @param  name:   buddy name.
        @type   msg:    string
        @param  msg:    C{MessageDlgController} containing C{Cancel} callback.
        """
        if self.has_request(name):
            self.remove_request(name)

            if msg:
                msg.cancel_cb()
                msg = MessageDlgController()
                msg.show_message("Invitation from <br>%s<br>has expired" % \
                        name, title="BUDDY REQUEST")
            else:
                msg = MessageDlgController()
                msg.show_message("Request timeout for <br>%s" % \
                        name, title="BUDDY REQUEST")

    def __remove_buddy(self, name):
        """
        Removes a buddy from connecting/connected buddies list.

        @type   name:   string
        @param  name:   Buddy name.
        """
        if self.buddy_exists(name):
            del self.__buddies[name]
            if name == self.__active_buddy:
                self.set_next_active_buddy()

    def add_buddy_connect_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a buddy is
        connected.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__buddy_connect_cbs:
            self.__buddy_connect_cbs.append(cb)

    def del_buddy_connect_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when a
        buddy is connected.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__buddy_connect_cbs:
            self.__buddy_connect_cbs.remove(cb)

    def add_buddy_disconnect_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a buddy is
        disconnected.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__buddy_disconnect_cbs:
            self.__buddy_disconnect_cbs.append(cb)

    def del_buddy_disconnect_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when a
        buddy is disconnected.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__buddy_disconnect_cbs:
            self.__buddy_disconnect_cbs.remove(cb)

    def add_data_available_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when there is
        buddy data available.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__data_available_cbs:
            self.__data_available_cbs.append(cb)

    def del_data_available_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when there
        is buddy data available.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__data_available_cbs:
            self.__data_available_cbs.remove(cb)

    def add_request_created_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a request is
        created.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__request_created_cbs:
            self.__request_created_cbs.append(cb)

    def del_request_created_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when a
        request is created.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__request_created_cbs:
            self.__request_created_cbs.remove(cb)

    def add_request_removed_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a request is
        removed.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__request_removed_cbs:
            self.__request_removed_cbs.append(cb)

    def add_received_request_timeout(self, name, alias, \
            accepted_cb, rejected_cb):
        """
        Adds a buddy request timeout.

        @type   name:   string
        @param  name:   Buddy name.
        @type   alias:  string
        @param  alias:  Buddy alias.
        @type   accepted_cb:    callback
        @param  accepted_cb:    Callback to be called if user accepts request.
        @type   rejected_cb:    callback
        @param  rejected_cb:    Callback to be called if user rejects request.
        """
        def confirm_cb():
            self.remove_request(name)
            if callable(accepted_cb):
                accepted_cb()

        def cancel_cb():
            self.remove_request(name)
            if callable(rejected_cb):
                rejected_cb()

        msg = MessageDlgController(confirm_cb=confirm_cb, cancel_cb=cancel_cb)

        if alias != None:
            msg.show_message("%s<br>wants to share Carman Information." \
                    "<br>Do you accept?" % alias, buttons=2)
        else:
            msg.show_message("%s<br>wants to share Carman Information." \
                    "<br>Do you accept?" % name, buttons=2)

        if not self.has_buddies():
            self.__active_buddy = name
        self.__requests[name] = ecore.Timer(self.BUDDY_REQUEST_TIMEOUT, \
                self.__request_timeout, name, msg)
        self.__buddies[name] = self.BUDDY_CONNECTING

    def add_sent_request_timeout(self, name):
        """
        Adds an user request timeout.

        @type   name:   string
        @param  name:   Buddy name.
        """
        if not self.has_buddies():
            self.__active_buddy = name
        self.__requests[name] = ecore.Timer(self.USER_REQUEST_TIMEOUT, \
                self.__request_timeout, name)
        self.__buddies[name] = self.BUDDY_CONNECTING

        for cb in self.__request_created_cbs:
            cb(name)

    def buddy_has_view(self, name=None):
        """
        Verifies if buddy has view.

        @type   name:   string
        @param  name:   Buddy name (if none, active buddy is used).
        @rtype: boolean
        @return:    C{True} if buddy has view, C{False} otherwise.
        """
        if name == None:
            name = self.__active_buddy
        return name in self.__has_view

    def received_data_msg(self, name, message):
        """
        Callback called when a buddy data message is received.

        @type   name:   string
        @param  name:   Buddy name.
        @type   message:    string
        @param  message:    Data message.
        @rtype: boolean
        @return:    C{True} if message should be used, C{False} otherwise.
        """
        if not self.is_buddy_connected(name):
            # Ignore since buddy is not properly connected
            return True

        gps_status = message.split(";")[8]
        obd_status = message.split(";")[9]

        new_data = [float(item) for item in message.split(";")[2:8]]
        new_data.append(gps_status)
        new_data.append(obd_status)

        if gps_status == GPSModel.DISABLED:
            new_data[0] = float(-1) # lat
            new_data[1] = float(-1) # lon
            new_data[2] = float(-1) # alt
            new_data[4] = float(-1) # track

        if obd_status != OBDModel.CONNECTED:
            new_data[5] = float(-1) # rpm
            if gps_status != GPSModel.FIXED:
                new_data[3] = float(-1) # speed

        self.__buddy_data[name] = new_data

        if name in self.__has_view:
            if gps_status == GPSModel.DISABLED:
                self.__has_view.remove(name)
        elif gps_status == GPSModel.FIXED:
            self.__has_view.append(name)

        for cb in self.__data_available_cbs:
            cb(name)
        return True

    def received_close_conn_msg(self, name):
        """
        Callback called when a close connection message is received.

        @type   name:   string
        @param  name:   Buddy name.
        @rtype: boolean
        @return:    C{True} if message should be used, C{False} otherwise.
        """
        if self.buddy_exists(name):
            self.disconnect_buddy(name, reason="Buddy closed connection")
        return True

    def get_connected_buddy_data(self, name=None):
        """
        Return last received data from connected buddy.

        @type   name:   string
        @param  name:   Buddy name (if none, active buddy is used).
        @rtype: tuple
        @return:    Tuple of buddy data C{(latitude, longitude, altitude,
                    speed, RPM, GPS status, RPM status)}.
        """
        if name == None:
            name = self.__active_buddy
        if name in self.__buddy_data:
            return self.__buddy_data[name]
        else:
            return None

    def get_connected_buddies(self):
        """
        Returns a list of connected buddies.

        @rtype: list
        @return:    List of connected buddies.
        """
        return [buddy for buddy in self.__buddies if \
            self.is_buddy_connected(buddy)]

    def get_connecting_buddies(self):
        """
        Returns a list of connecting buddies.

        @rtype: list
        @return:    List of connecting buddies.
        """
        return [buddy for buddy in self.__buddies if \
            self.is_buddy_connecting(buddy)]

    def is_buddy_connected(self, name):
        """
        Verifies if buddy is connected.

        @type   name:   string
        @param  name:   Buddy name.
        @rtype: boolean
        @return:    C{True} if buddy is connected, C{False} otherwise.
        """
        return self.buddy_exists(name) and \
                self.__buddies[name] == self.BUDDY_CONNECTED

    def is_buddy_connecting(self, name):
        """
        Verifies if buddy is connecting.

        @type   name:   string
        @param  name:   Buddy name.
        @rtype: boolean
        @return:    C{True} if buddy is connecting, C{False} otherwise.
        """
        return self.buddy_exists(name) and \
                self.__buddies[name] == self.BUDDY_CONNECTING

    def buddy_exists(self, name):
        """
        Verifies if buddy exists (appears in connecting/connected buddy list).

        @type   name:   string
        @param  name:   Buddy name.
        @rtype: boolean
        @return:    C{True} if buddy exists, C{False} otherwise.
        """
        return name in self.__buddies

    def has_buddies(self):
        """
        Verifies if the buddy list contains buddies.

        @rtype: boolean
        @return:    C{True} of buddy list lenght is greater than 0, C{False}
                    otherwise.
        """
        return len(self.__buddies) > 0

    def has_request(self, name):
        """
        Verifies if request exists.

        @type   name:   string
        @param  name:   Buddy name.
        @rtype: boolean
        @return:    c{True} if request for buddy exists, C{False} otherwise.
        """
        return name in self.__requests

    def connect_buddy(self, name):
        """
        Adds a buddy to connected buddies list.

        @type   name:   string
        @param  name:   Buddy name.
        """
        if not self.has_buddies():
            self.__active_buddy = name
        self.__buddies[name] = self.BUDDY_CONNECTED
        self.__buddy_data[name] = [float(-1), float(-1), float(-1), \
            float(-1), float(-1), float(-1), \
            GPSModel.DISABLED, OBDModel.DISABLED]
        for cb in self.__buddy_connect_cbs:
            cb(name)

    def disconnect_buddy(self, name, reason=None):
        """
        Removes a buddy from connected buddies list.

        @type   name:   string
        @param  name:   Buddy name.
        @type   reason: string
        @param  reason: Buddy disconnection reason to be displayed to user, if
                        any.
        """
        if self.buddy_exists(name):
            self.__remove_buddy(name)
            del self.__buddy_data[name]
            if name in self.__has_view:
                self.__has_view.remove(name)
            for cb in self.__buddy_disconnect_cbs:
                cb(name, reason, connected = True)

    def disconnect_all_buddies(self):
        """
        Disconnects from all connecting/connected buddies.
        """
        for buddy in self.get_connected_buddies():
            self.disconnect_buddy(buddy)
        for buddy in self.get_connecting_buddies():
            self.remove_request(buddy)

    def remove_request(self, name, accepted=False):
        """
        Removes a pending request.

        @type   name:   string
        @param  name:   Buddy name.
        @type   accepted:   boolean
        @param  accepted:   C{True} if user accepted connection, C{False}
                            otherwise.
        """
        if self.has_request(name):
            if not accepted:
                self.__remove_buddy(name)
            self.__requests[name].delete()
            del self.__requests[name]
            for cb in self.__request_removed_cbs:
                cb(name)

    def set_next_active_buddy(self):
        """
        Select next buddy from list to act as active buddy, if any.
        """
        if self.has_buddies():
            if self.__active_buddy in self.__buddies:
                i = self.__buddies.keys().index(self.__active_buddy)
            else:
                i = 0
            i = (i + 1) % self.size
            self.__active_buddy = self.__buddies.keys()[i]
        else:
            self.__active_buddy = ''
