#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{MapsController}.
"""

import edje

from common.carmanconfig import CarmanConfig
from main.container import Controller
from main.mainview import MainView
from main.messagedlgctrl import MessageDlgController
from maps.connectaccountctrl import ConnectAccountController
from maps.connectbuddyctrl import ConnectBuddyController
from maps.infosharingctrl import InfoSharingController
from maps.maptilectrl import MapTileController
from maps.optionctrl import OptionController
from maps.progressbarctrl import ProgressBarController
from maps.trackctrl import TrackController
#from maps.trackctrl import TrackAuxController
from models.gpsmodel import GPSModel
from models.kbmodel import KeyBoardModel
from models.mapmodel import MapModel
from models.buddymodel import BuddyModel
from models.connmodel import ConnectionModel
from models.infosharingmodel import InfoSharingModel


class MapsController(Controller):
    """
    Controls all map related features (map tile downloads, InfoShare buddy
    information/view, GPS tracking).

    @ivar download_ctrl: If set, has an instance of L{DownloadAreaController}
                         or L{DownloadTrackController} (currently not used).
    """

    def __init__(self, parent, canvas, theme):
        self.canvas = canvas
        self._parent = parent
        self.screen = None
        self.download_ctrl = None
        self.__toggle_fullscreen = False
        self.buddy_model = BuddyModel()
        self.buddy_model.add_buddy_connect_cb(self.__goto_buddy_cb)
        self.buddy_model.add_buddy_disconnect_cb(self.__goto_buddy_cb)
        self.buddy_model.add_request_created_cb(self.__goto_buddy_cb)
        self.buddy_model.add_request_removed_cb(self.__goto_buddy_cb)

        self.main_view = MainView()
        self.map_model = MapModel()
        self.map_model.add_map_service_updated(self.check_map_service_zoom)

        self.gps_model = GPSModel()
        self.gps_model.add_status_changed_cb(self.set_gps_status)

        self.is_model = InfoSharingModel()
        if CarmanConfig().is_service:
            self.is_model.add_status_changed_cb(self.__on_is_status_changed)
            self.is_model.add_network_error_cb(self.__on_reconnect_account)
        self.conn_model = ConnectionModel()
        self.conn_model.add_connection_lost_cb(self.__on_connection_lost)

        self.track_ctrl = TrackController(canvas, self.gps_model)
        self.track_aux_ctrl = None # TrackAuxController(canvas)
        self.is_ctrl = InfoSharingController(self, canvas)
        self.map_ctrl = MapTileController(self, canvas, self.track_ctrl,
            self.is_ctrl, self.track_aux_ctrl)
        self.map_ctrl.set_zoom_changed_cb(self.update_zoom_buttons)
        self.progressbar_ctrl = ProgressBarController(canvas)

        self.set_theme(theme)
        lat, lon, self.zoom = CarmanConfig().get_last_fix()
        self.map_ctrl.set_position(lat, lon, self.zoom)

        self.kb_model = KeyBoardModel()
        self.option_ctrl = OptionController(self)

    def __goto_buddy_cb(self, obj=None, item=None, connected=None):
        """
        Updates C{Buddy} image on right side panel.

        @type   obj: object
        @param  obj: Not used.
        @type   item: string
        @param  item: Not used.
        @type   connected: boolean
        @param  connected: Not used.
        """
        if self.is_model.account_status != InfoSharingModel.CONNECTED:
            # Forcefully disconnected from IS account, but still receiving
            # "buddy disconnected" callbacks from BuddyModel.
            return
        elif self.buddy_model.size <= 1:
            self.screen.signal_emit("enable-goto-buddy", "")
        else:
            self.screen.signal_emit("enable-next-buddy", "")

    def __on_buddy_pressed(self, *data):
        """
        Implements what happens when user presses C{buddy} icon on
        right-side panel.

        Actions are:
            1. If InfoShare plugin is not installed, warn user about it.
            2. If InfoShare plugin is installed but not instantiated,
            instantiate InfoShare service.
            3. If Infoshare plugin is instantiated but account is not
            connected, connect it.
            4. If InfoShare account is connected:
                4.1. If not connected to 0 or 1 buddy, open
                L{ConnectBuddyController}.
                4.2. If connected to 2 or more buddies, selects another buddy
                from the list of connected buddies to be the active buddy.
        """
        if CarmanConfig().is_service:
            if not self.is_model.service:
                self.is_model.instantiate_isservice()

            if self.is_model.account_status == InfoSharingModel.CONNECTED:
                if self.buddy_model.size > 1:
                    self.buddy_model.set_next_active_buddy()
                    self.is_ctrl.update_buddy_information()
                    if self.buddy_model.buddy_has_view():
                        self.set_center_buddy()
                    else:
                        self.set_center_gps()
                else:
                    buddy = ConnectBuddyController(self)
                    if buddy.load_list():
                        buddy.show()
            else:
                account = ConnectAccountController()
                if account.exist_account():
                    account.connect_account()
        else:
            msg = MessageDlgController()
            msg.show_message("Infoshare not installed", title="INFOSHARE")

    def __on_connection_lost(self):
        """
        Disconnects from InfoShare account when network connection is lost.
        """
        if CarmanConfig().is_service:
            account = self.is_model.service.get_account()
            self.is_model.service.account_disconnect()
            self.conn_model.add_connection_enabled_cb( \
                self.__on_reconnect_account)

    def __on_is_status_changed(self, status):
        """
        Updates C{Buddy} image on right side panel according to InfoShare
        account status.

        Infoshare account statues are: (L{InfoSharingModel.CONNECTED},
        L{InfoSharingModel.CONNECTING}, L{InfoSharingModel.DISCONNECTED}).

        @type   status: string
        @param  status: InfoShare account status.
        """
        if status == InfoSharingModel.CONNECTED:
            self.screen.signal_emit("enable-goto-buddy", "")
        elif status == InfoSharingModel.CONNECTING:
            self.screen.signal_emit("init-acc-anime", "")
        elif status == InfoSharingModel.DISCONNECTED:
            self.screen.signal_emit("disable-goto-buddy", "")

    def __on_reconnect_account():
        """
        Tries to reconnect InfoShare account when network status is connected.
        If network status is not connected, displays a message to user warning
        him/her about it.
        """
        if ConnectionModel().Status() == ConnectionModel.CONNECTED:
            reconnect = ConnectAccountController()
            if reconnect.exist_account():
                reconnect.connect_account()
        else:
            msg = MessageDlgController()
            msg.show_message("Wlan Disconnected", title="CONNECTION LOST")
    __on_reconnect_account = staticmethod(__on_reconnect_account)

    def __on_toggle_fullscreen(self):
        """
        Toggles between I{default} and I{fullscreen} window modes.
        """
        controller = self.main_view.controller
        view = self.main_view.view

        if self.__toggle_fullscreen:
            view.signal_callback_add("exit-button-pressed", "*",
                                     controller.exit_button)
            view.signal_callback_add("change-button-pressed", "*",
                                     controller.change_button)
            view.signal_callback_add("option-button-pressed", "*",
                                     controller.option_button)
            view.signal_callback_add("status-button-pressed", "*",
                                     controller.status_button)
            self.screen.signal_callback_add("zoom-up-pressed", "",
                                            self.__on_zoom_up)
            self.screen.signal_callback_add("zoom-down-pressed", "",
                                            self.__on_zoom_down)
            self.screen.signal_callback_add("goto-gps-pressed", "",
                                            self.set_center_gps)
            self.screen.signal_callback_add("goto-buddy-pressed", "",
                                            self.__on_buddy_pressed)
            view.signal_emit("default", "")
            self.screen.signal_emit("default", "")
        else:
            view.signal_callback_del("exit-button-pressed", "*",
                                     controller.exit_button)
            view.signal_callback_del("change-button-pressed", "*",
                                     controller.change_button)
            view.signal_callback_del("option-button-pressed", "*",
                                     controller.option_button)
            view.signal_callback_del("status-button-pressed", "*",
                                     controller.status_button)
            self.screen.signal_callback_del("zoom-up-pressed", "",
                                            self.__on_zoom_up)
            self.screen.signal_callback_del("zoom-down-pressed", "",
                                            self.__on_zoom_down)
            self.screen.signal_callback_del("goto-gps-pressed", "",
                                            self.set_center_gps)
            self.screen.signal_callback_del("goto-buddy-pressed", "",
                                            self.__on_buddy_pressed)
            view.signal_emit("fullscreen", "")
            self.screen.signal_emit("fullscreen", "")
        self.__toggle_fullscreen = not self.__toggle_fullscreen

    def __on_zoom_up(self, *params):
        """
        Calls L{MapTileController} to control zoom up.
        """
        self.map_ctrl.zoom_up()

    def __on_zoom_down(self, *params):
        """
        Calls L{MapTileController} to control zoom down.
        """
        self.map_ctrl.zoom_down()

    def check_map_service_zoom(self, repo):
        """
        Updates current map zoom when map service is updated. If the current
        map zoom is lesser or greater than the map service's zoom range, this
        method adjusts the current zoom to the minimum or maximum zoom,
        accordingly.

        @type   repo: class
        @param  repo: Instance of map service class (eg. L{OSMRepo}).
        """
        zoom = self.map_ctrl.zoom
        if zoom is None:
            zoom = self.zoom
        zoom_range = repo.get_zoom()
        if zoom < zoom_range[0]:
            for x in xrange(zoom, zoom_range[0]):
                self.map_ctrl.zoom_down()

        if zoom > zoom_range[-1]:
            for x in xrange(zoom_range[-1], zoom):
                self.map_ctrl.zoom_up()

    def get_view(self):
        """
        Returns main screen.

        @rtype: class
        @return: Instance of L{Edje} view.
        """
        return self.screen

    def get_icon(self):
        """
        Returns C{Maps} icon to be displayed on top-right area.

        @rtype: tuple
        @return: Data used to capture C{Maps} icon data.
        """
        return (self.theme_file, "images/icon-maps")

    def activate(self):
        """
        Activates C{Maps} view.
        """
        self.screen.show()
        self.map_ctrl.activate()
        self.kb_model.add_key_down_cb(self.key_down_cb)

    def deactivate(self):
        """
        Deactivates C{Maps} view.
        """
        self.kb_model.del_key_down_cb(self.key_down_cb)

    def key_down_cb(self, obj, event):
        """
        Associates key presses with callbacks.

        @type   obj: class
        @param  obj: Not used.
        @type   event: class
        @param  event: Instance of L{ecore.x.EventKeyDown}.
        """
        if self.main_view.transiting:
            return False

        if event.key == "Left":
            self.map_ctrl.move(-32, 0)
        elif event.key == "Right":
            self.map_ctrl.move(32, 0)
        elif event.key == "Up":
            self.map_ctrl.move(0, -32)
        elif event.key == "Down":
            self.map_ctrl.move(0, 32)
        elif event.key == "F4":
            self.show_options()
        elif event.key == "F6":
            self.__on_toggle_fullscreen()
        elif event.key == "F7":
            self.__on_zoom_up()
        elif event.key == "F8":
            self.__on_zoom_down()
        else:
            return False

        return True

    def set_global_unit(self, value):
        """
        Calls L{MainView} to set theme according to the given value.

        @type   value: boolean
        @param  value: C{True} if unit system is metric, C{False} if imperial.
        """
        self._parent.set_unit_system(value)

    def set_unit_system(self, value):
        """
        Updates C{Maps} view with the given unit system.

        @type   value: boolean
        @param  value: C{True} if unit system is metric, C{False} if imperial.
        """
        self.is_ctrl.set_unit_system(value)

    def set_global_theme(self):
        """
        Calls L{MainView} to set theme according to L{CarmanConfig}.
        """
        self._parent.set_theme()

    def set_theme(self, theme):
        """
        Updates C{Maps} view with given theme.

        @type   theme: string
        @param  theme: Theme filename with full path.
        """
        if self.screen:
            self.screen.delete()
        self.theme_file = theme
        first_time = self.screen is None
        self.screen = edje.Edje(self.canvas, file=self.theme_file,
                                group="maps")
        self.screen.size = self.canvas.size

        self.track_ctrl.set_view(theme, self.screen)
        #self.track_aux_ctrl.set_view(theme, self.screen)
        self.is_ctrl.set_view(theme, self.screen)
        self.map_ctrl.set_view(theme, self.screen)
        self.progressbar_ctrl.set_view(theme, self.screen)

        if not first_time:
            self.update_zoom_buttons(*self.map_ctrl.get_zoom_state())
            self.map_ctrl.update_position()
        self.set_gps_status(self.gps_model.Status())

        if CarmanConfig().is_service:
            self.__on_is_status_changed(self.is_model.account_status)

        self.screen.signal_callback_add("zoom-up-pressed", "",
                                        self.__on_zoom_up)
        self.screen.signal_callback_add("zoom-down-pressed", "",
                                        self.__on_zoom_down)
        self.screen.signal_callback_add("goto-gps-pressed", "",
                                        self.set_center_gps)
        self.screen.signal_callback_add("goto-buddy-pressed", "",
                                        self.__on_buddy_pressed)

    def set_gps_status(self, status):
        """
        Updates GPS icon on right side panel according to current GPS status.

        @type   status: string
        @param  status: GPS status (L{GPSModel.DISABLED},
                        L{GPSModel.DISCONNECTED}, C{GPSModel.CONNECTING},
                        C{GPSModel.FIXING}, C{GPSModel.FIXED}).
        """
        if status == GPSModel.FIXED:
            self.screen.signal_emit("enable-goto-gps", "")
        else:
            self.screen.signal_emit("disable-goto-gps", "")

    def update_zoom_buttons(self, up, down):
        """
        Updates C{Zoom Up} and C{Zoom Down} images on right side panel.

        @type   up: boolean
        @param  up: C{True} if C{Zoom Up} image goes enabled, C{False}
                    otherwise.
        @type   down: boolean
        @param  down: C{True} if C{Zoom Down} image goes enabled, C{False}
                      otherwise.
        """
        if up:
            self.screen.signal_emit("enable-zoom-up", "")
        else:
            self.screen.signal_emit("disable-zoom-up", "")
        if down:
            self.screen.signal_emit("enable-zoom-down", "")
        else:
            self.screen.signal_emit("disable-zoom-down", "")

    def set_center_buddy(self):
        """
        Sets move callback on buddy position (L{InfoSharingController}).
        """
        self.track_ctrl.set_move_cb(None)
        self.is_ctrl.set_move_cb(self.map_ctrl.set_position_xy)
        self.is_ctrl.update_buddy_information()

    def set_center_free(self):
        """
        Releases move callback. Map window won't be updated when GPS or buddy
        positions goes outside area view.
        """
        self.track_ctrl.set_move_cb(None)
        self.is_ctrl.set_move_cb(None)

    def set_center_gps(self, *data):
        """
        Sets move callback on GPS position (L{TrackController}).

        @type   data: tuple
        @param  data: Not used.
        """
        if self.gps_model.Status() == GPSModel.FIXED:
            self.is_ctrl.set_move_cb(None)
            self.track_ctrl.set_move_cb(self.map_ctrl.set_position_xy)

    """
    def set_track_aux(self, models, track_file):
        self.track_aux_ctrl.set_track_model(models, track_file)
        # FIXME: Point screen to loaded track
        self.map_ctrl.update_position()

    def get_selected_track(self):
        return self.track_aux_ctrl.get_selected_track()

    def unload_track_aux(self):
        self.track_aux_ctrl.clear()
    """

    def get_map_view_area(self):
        """
        Returns map view area.

        @rtype: tuple
        @return: Map view coordinates (M{x1}, M{y1}, M{x2}, M{y2}).
        """
        return self.map_ctrl.get_view_area()

    def show_options(self):
        """
        Shows L{OptionController}.
        """
        self.option_ctrl.show()

    def show_progressbar(self, ctrl):
        """
        Shows download progress bar.

        @type   ctrl: class
        @param  ctrl: Instance of L{DownloadAreaController} or
                      L{DownloadTrackController}.
        """
        self.download_ctrl = ctrl
        self.progressbar_ctrl.show()

    def hide_progressbar(self):
        """
        Hides download progress bar.
        """
        self.download_ctrl = None
        self.progressbar_ctrl.hide()

    def update_progressbar(self, pos):
        """
        Updates L{ProgressBarController} position.

        @type   pos: number
        @param  pos: Progress bar position to be updated (eg. 3 from 10
                     downloads have finished so far).
        """
        self.progressbar_ctrl.update(pos)

    def is_downloading(self):
        """
        Verifies if L{download_ctrl} is set.

        @rtype: boolean
        @return: C{True} if L{download_ctrl} has an instance of
                 L{DownloadAreaController} or L{DownloadTrackController},
                 C{False} otherwise.
        """
        return self.download_ctrl is not None

    def stop_download(self):
        """
        Stops map tile downloads.
        """
        if self.download_ctrl:
            self.download_ctrl.stop_download()
            self.hide_progressbar()

    def finalize(self):
        """
        Calls all child controllers to finish their jobs. Also saves last map
        position.
        """
        self.is_ctrl.finalize()
        self.track_ctrl.finalize()
        #self.track_aux_ctrl.finalize()
        lat, lon, zoom = self.map_ctrl.get_position()
        CarmanConfig().set_last_fix(lat, lon, zoom)
        self.map_ctrl.finalize()
