#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{InfoSharingController}.
"""

from common.carmanconfig import CarmanConfig, METRIC
from main.messagedlgctrl import MessageDlgController
from maps.infosharingview import BuddyView, BuddyInformationView
from models.buddymodel import BuddyModel
from models.gpsmodel import GPSModel
from models.mapmodel import MapModel
from models.infosharingmodel import InfoSharingModel


class InfoSharingController(object):
    """
    Controls InfoShare-related views (updates buddy information window
    L{BuddyInformationView} and buddy icon L{BuddyView}, both visible at Maps
    window).

    @type   controller: class
    @param  controller: Instance of L{MapsController}.
    @type   canvas: class
    @param  canvas: Main canvas.
    @ivar   __move_cb: When set, this callback moves L{BuddyView}'s icon.
    @ivar   buddy_information: Buddy information view.
    @ivar   buddy_view: Buddy icon view.
    """

    def __init__(self, controller, canvas):
        self.__move_cb = None
        self.__zoom = None
        self.view = None

        self.buddy_model = BuddyModel()
        self.buddy_model.add_buddy_connect_cb(self.__buddy_connect_cb)
        self.buddy_model.add_buddy_disconnect_cb(self.__buddy_disconnect_cb)
        self.buddy_model.add_request_created_cb(self.__buddy_connect_cb)
        self.buddy_model.add_request_removed_cb(self.__buddy_disconnect_cb)

        self.map_model = MapModel()

        self.is_model = InfoSharingModel()

        metric = CarmanConfig().get_unit() == METRIC
        self.buddy_view = BuddyView(controller, canvas)
        self.buddy_information = BuddyInformationView(canvas, metric)

    def __update_buddy_area(self, obj=None, signal=None, param=None):
        """
        Updates buddy area according to I{default} and I{fullscreen} window
        modes.
        @type   obj: class
        @param  obj: Instance of L{Edje} which originated signal.
        @type   signal: string
        @param  signal: Signal originated (C{default} or C{fullscreen}).
        @type   param: object
        @param  param: Not used.
        """
        if signal and signal == "fullscreen":
            x1, y1, x2, y2 = self.view.part_geometry_get("map_view")
            area = (x1 + 40, y1 + 40, x2 - 40, y2 - 40)
        else:
            x1, y1, x2, y2 = self.view.part_geometry_get("area_view")
            area = (x1 + 20, y1 + 20, x2 + x1 - 20, y2 + y1 - 20)
        self.buddy_view.set_buddy_area(area)
        self.view.part_swallow("buddy_view", self.buddy_view)

    def __buddy_data_available(self, name):
        """
        Calls L{update_buddy_information} if available data is from active
        buddy.

        @type   name: string
        @param  name: Buddy name.
        """
        if name == self.buddy_model.active_buddy:
            self.update_buddy_information()

    def __buddy_connect_cb(self, name):
        """
        Updates L{BuddyInformationView} If connected buddy is active buddy.

        @type   name: string
        @param  name: Buddy name.
        """
        if name == self.buddy_model.active_buddy:
            alias = self.is_model.get_buddy_alias(name)
            has_alias = lambda alias: alias if alias else name
            self.buddy_information.set_buddy_name(has_alias(alias))

            if self.buddy_model.is_buddy_connected(name):
                self.buddy_model.add_data_available_cb( \
                    self.__buddy_data_available)
                if not self.buddy_information.is_visible():
                    self.buddy_information.set_gps_status( \
                                                GPSModel.DISCONNECTED)
                    self.buddy_information.set_speed_value(-1)
                    self.buddy_information.set_rpm_value(-1)
                self.buddy_information.show_data_values()
            else:
                self.buddy_information.show_buddy_connecting()
            self.buddy_information.show()

    def __buddy_disconnect_cb(self, name, reason=None, connected=False):
        """
        Hides L{BuddyInformationView} and L{BuddyView} if no more connected
        buddies left. Also displays a message to user showing the buddy
        disconnection reason, if any.

        @type   name: string
        @param  name: Buddy name.
        @type   reason: string
        @param  reason: Buddy disconnection reason.
        @type   connected: boolean
        @param  connected: C{True} if buddy is now connected, C{False}
                           otherwise.
        """
        alias = self.is_model.get_buddy_alias(name)
        has_alias = lambda alias: alias if alias else name

        if self.buddy_model.has_buddies():
            self.update_buddy_information()
        else:
            if connected:
                self.buddy_model.del_data_available_cb( \
                    self.__buddy_data_available)
            self.buddy_view.deactivate()
            self.buddy_information.hide()
            self.set_move_cb(None)

        if connected:
            msg = MessageDlgController()
            if reason:
                msg.show_message("%s<br>disconnected:<br>%s" % \
                    (has_alias(alias), reason), title="BUDDY DISCONNECTED")
            else:
                msg.show_message("%s<br>disconnected" % \
                    (has_alias(alias)), title="BUDDY DISCONNECTED")

    def get_lock(self):
        """
        Verifies if L{__move_cb} is set up.

        @rtype: boolean
        @return: C{True} if L{__move_cb} is callable, C{False} otherwise.
        """
        return callable(self.__move_cb)

    def set_move_cb(self, cb):
        """
        Sets L{BuddyView} move callback, when available.

        @type   cb: callback
        @param  cb: Callback to called when L{__move_cb} is used.
        """
        if callable(cb):
            self.__move_cb = cb
            if self.buddy_model.active_buddy != '':
                x, y = [item >> self.__zoom for item in \
                    self.map_model.latlon_to_xy( \
                        *self.buddy_model.get_connected_buddy_data()[0:2])]
                self.__move_cb(x, y, False)
        else:
            self.__move_cb = None

    def set_unit_system(self, value):
        """
        Sets unit system (I{metric} or I{imperial}).

        @rtype: boolean
        @return: C{True} if unit system is metric, C{False} if imperial.
        """
        self.buddy_information.set_unit_system(value)

    def set_view(self, theme, view):
        """
        Updates L{BuddyInformationView} and L{BuddyView} views
        (generally called when theme is changed).

        @type   theme: string
        @param  theme: Theme filename.
        @type   view: class
        @param  view: Instance of L{Edje}.
        """
        self.view = view
        self.view.signal_callback_add("default", "", \
            self.__update_buddy_area)
        self.view.signal_callback_add("fullscreen", "", \
            self.__update_buddy_area)

        if self.buddy_model.is_buddy_connected( \
                        self.buddy_model.active_buddy):
            gps_status = self.buddy_model.get_connected_buddy_data( \
                                self.buddy_model.active_buddy)[6]
            if gps_status == GPSModel.FIXED:
                self.buddy_view.set_theme(theme, True)
            elif gps_status != GPSModel.DISABLED and \
                    self.buddy_model.buddy_has_view():
                self.buddy_view.set_theme(theme, False)
            else:
                self.buddy_view.set_theme(theme)
        else:
            self.buddy_view.set_theme(theme)
        self.__update_buddy_area()
        self.buddy_information.set_view(theme, view)
        self.update_buddy_information()

    def set_position(self, x, y, zoom):
        """
        Updates L{BuddyView} position.

        @type   x: number
        @param  x: Cartesian map M{X} position.
        @type   y: number
        @param  y: Cartesian map M{Y} position.
        @type   zoom: number
        @param  zoom: Map zoom.
        """
        self.buddy_view.set_position(x, y)
        if self.__zoom != zoom:
            self.__zoom = zoom
            if self.buddy_model.active_buddy != '':
                x, y = [item >> self.__zoom for item in \
                    self.map_model.latlon_to_xy( \
                        *self.buddy_model.get_connected_buddy_data()[0:2])]
                self.buddy_view.set_buddy_position(x, y)

    def update_buddy_information(self):
        """
        Updates both L{BuddyInformationView} and L{BuddyView} according to
        active buddy's data.
        """
        def update_buddy_view(buddy_gps):
            self.buddy_view.set_image(buddy_gps)
            x, y = [item >> self.__zoom for item in \
                        self.map_model.latlon_to_xy(lat, lon)]
            self.buddy_view.activate()
            self.buddy_view.set_buddy_position(x, y)
            if callable(self.__move_cb):
                self.__move_cb(x, y)

        active_buddy = self.buddy_model.active_buddy
        alias = self.is_model.get_buddy_alias(active_buddy)
        has_alias = lambda alias: alias if alias else active_buddy
        self.buddy_information.set_buddy_name(has_alias(alias))

        if self.buddy_model.is_buddy_connected(active_buddy):
            (lat, lon, alt, speed, track, rpm, gps_status, obd_status) = \
                self.buddy_model.get_connected_buddy_data(active_buddy)

            self.buddy_information.set_speed_value(speed)
            self.buddy_information.set_rpm_value(rpm)
            self.buddy_information.set_gps_status(gps_status)
            self.buddy_information.show_data_values()

            if self.__zoom != None:
                if gps_status == GPSModel.FIXED:
                    update_buddy_view(True)
                elif gps_status != GPSModel.DISABLED and \
                        self.buddy_model.buddy_has_view():
                    update_buddy_view(False)
                else:
                    self.buddy_view.deactivate()
        else:
            self.buddy_information.show_buddy_connecting()
            self.buddy_view.deactivate()

    def finalize(self):
        """
        Calls L{BuddyModel.disconnect_all_buddies} in order to disconnect from
        all connected buddies.
        """
        self.buddy_model.disconnect_all_buddies()
