#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{CommonAccountController}, L{BasicInfoController},
L{PasswordInfoController} and L{ServerInfoController}.
"""

import re
from maps.manageaccountdlgview import ManageAccountDlgView
from main.messagedlgctrl import MessageDlgController
from models.kbmodel import KeyBoardModel


def message_error(title, message, cb):
    """
    Wrapper for Message Error dialog.

    @type   title: string
    @param  title: Window title.
    @type   message: string
    @param  message: Message to be displayed.
    @type   cb: callback
    @param  cb: Focus callback.
    """
    msg_error = MessageDlgController(cb)
    msg_error.show_message(message, title=title)


class CommonAccountController(object):
    """
    This class implements methods used by SetupAccount, NewAccount and other
    Controllers which modifies or creates an InfoShare account.

    @type title: string
    @param title: Dialog title.
    @type info: dictionary
    @param info: Dictionary containing all information about an Infoshare account.
    @type save_cb: callback
    @param save_cb: Callback method used to save account information.
    @ivar protocol_id: account protocol ID, it can be C{prpl-jabber, prpl-msn,
    prpl-yahoo}.
    """
    def __init__(self, title, info, save_cb):
        self.__title = title
        self.__info = info
        self.__edited_info = {}
        self.protocol_id = self.__info['protocol_id']
        if 'username' not in self.__info:
            self.__edited_info = self.__info
            self.__info = {}
        self.__save_cb = callable(save_cb) and save_cb or None
        self.selected_protocol = None

    def __basic_info_page(self, check_account_exist):
        """
        Creates and shows basic account info dialog containing account username
        and account password.

        @type check_account_exist: boolean
        @param check_account_exist: C{True} if Infoshare has already a
        configured account, C{False} otherwise.
        """
        self.basic_info = BasicInfoController(self.__title,
                self.protocol_id, self.__password_info_page,
                self.__return_to_protocol, check_account_exist)
        self.basic_info.show()
        self.__set_basic_info()

    def __password_info_page(self, *p):
        """
        Creates and shows password info dialog.
        """
        self.set_specific_server()
        self.password_info = PasswordInfoController(self.__title,
                self.__server_info_page, self.__return_to_basic)
        self.password_info.show()
        self.__set_password_info()

    def __server_info_page(self, *p):
        """
        Creates and shows server info dialog.
        """
        self.server_info = ServerInfoController(self.__title,
                self.__save_info, self.__return_to_password)
        self.server_info.show()
        self.__set_server_info()

    def __return_to_protocol(self):
        """
        Returns to Select Protocol dialog.
        """
        KeyBoardModel().set_focus()
        self.__get_basic_info()

    def __return_to_basic(self):
        """
        Returns to basic information dialog.
        """
        self.basic_info.view.first_entry_focus()
        self.__get_password_info()

    def __return_to_password(self):
        """
        Returns to configure password dialog.
        """
        self.password_info.view.first_entry_focus()
        self.__get_server_info()

    def __set_basic_info(self):
        """
        Saves account username and alias into C{self.__info} dict.
        """
        view = self.basic_info.view
        view.first_entry.text = self.get_dict_value('username').split("/")[0]
        view.second_entry.text = self.get_dict_value('alias')

    def __set_password_info(self):
        """
        Saves accout password and remember password flag into C{self.__info}
        dict.
        """
        view = self.password_info.view
        view.first_entry.text = self.get_dict_value('password')
        view.remember = self.get_dict_value('remember_password')
        if not view.remember:
            view.remember = 0
        else:
            view.signal_emit("enable-remember", "")

    def __set_server_info(self):
        """
        Saves server port and server address into C{self.__info} dict.
        """
        view = self.server_info.view
        view.first_entry.text = self.get_dict_value('server')
        view.second_entry.text = str(self.get_dict_value('port'))

    def __get_basic_info(self):
        """
        Gets account username and alias from Basic Info dialog view.
        """
        view = self.basic_info.view
        self.set_dict_value('username', view.first_entry.text.strip())
        self.set_dict_value('alias', view.second_entry.text.strip())

    def __get_password_info(self):
        """
        Gets password and remember password from Password dialog view.
        """
        view = self.password_info.view
        self.set_dict_value('password', view.first_entry.text.strip())
        self.set_dict_value('remember_password', view.remember)

    def __get_server_info(self):
        """
        Gets server port and server address from Server Configuration dialog
        view.
        """
        view = self.server_info.view
        self.set_dict_value('server', view.first_entry.text.strip())
        self.set_dict_value('port', view.second_entry.text.strip())

    def __save_info(self, *p):
        """
        Send all information filled on previous dialogs to Infosharing Service.
        """
        self.__get_basic_info()
        self.__get_password_info()
        self.__get_server_info()
        self.server_info.hide()
        self.password_info.hide()
        self.basic_info.hide()
        KeyBoardModel().set_focus()

        # Hack to avoid resource problems
        if self.get_dict_value('protocol_id') == 'prpl-jabber':
            self.set_dict_value('username', \
                self.get_dict_value('username').split("/")[0] + "/Carman")

        self.__save_cb(self.__edited_info)
        self.__save_cb = None

    def show(self, check_account_exist):
        """
        Shows Basic Info dialog.
        """
        self.__basic_info_page(check_account_exist)

    def set_specific_server(self):
        """
        This method is a hack to force C{talk.google.com} when C{protocol_id} is
        C{prpl-jabber}.
        """
        domain = self.basic_info.view.first_entry.text.split("@")
        if self.get_dict_value('protocol_id') == 'prpl-jabber' and \
                'server' in self.__edited_info and len(domain) > 1 and \
                domain[1] == "gmail.com":
            self.__edited_info['server'] = "talk.google.com"

    def get_dict_value(self, key):
        """
        Get a dict key value. The dict can be C{self.__info} or
        C{self.__edited_info}.

        @type key: string
        @param key: Dict key.
        """
        if key in self.__edited_info:
            return self.__edited_info[key]
        elif key in self.__info and self.__info[key]:
            return self.__info[key]
        else:
            return ""

    def set_dict_value(self, key, value):
        """
        Sets a value to given dict key. The dict can be C{self.__info} or
        C{self.__edited_info}.

        @type key: string
        @param key: Dict key.
        @type value: object
        @param value: Dict value.
        """
        if key in self.__info and self.__info[key] == value and \
                key in self.__edited_info:
            del self.__edited_info[key]
        elif (key in self.__info and self.__info[key] != value) or \
                key not in self.__info:
            self.__edited_info[key] = value


class BasicInfoController(object):
    """
    Controller for Basic Info dialog.

    @type title: string
    @param title: Dialog window title.
    @type protocol: string
    @param protocol: Protocol ID.
    @type cb: callback
    @param cb: Callback used when OK button is clicked.
    @type r_cb: callback
    @param r_cb: Callback used when CANCEL button is clicked.
    @type check_account_exist: boolean
    @param check_account_exist: Flag to indicate if Infoshare already has a
    configured account or not. C{True} if account exists, C{False} otherwise.
    """
    def __init__(self, title, protocol, cb, r_cb, check_account_exist):
        self.__title = title
        self.__protocol = protocol
        self.cb = callable(cb) and cb or None
        self.r_cb = callable(r_cb) and r_cb or None
        self.__check_account_exist = check_account_exist

    def show(self):
        """
        Instantiates dialog view and shows it.
        """
        self.view = ManageAccountDlgView(self.__title, self.__confirm_cb,
                self.__return, "manageaccountdlg1")
        self.view.show()

    def __confirm_cb(self, *p):
        """
        Callback called when OK button is clicked. It is a wrapper to
        C{self.cb()}.

        @type p: set of arguments
        @param p: Not used.
        @rtype: boolean
        @return: C{True} is username is valid, C{False} otherwise.
        """
        if self.check_username():
            self.cb()
            return True
        return False

    def __return(self, *p):
        """
        Method called when Cancel button is clicked. It is a wrapper to
        C{self.r_cb()}.

        @type p: set of arguments
        @param p: Not used.
        """
        self.r_cb()
        self.cb = self.r_cb = None

    def check_username(self):
        """
        Checks if entered username is valid.
        """
        username = self.view.first_entry.text.strip()
        if username == "":
            message_error(self.__title, "Login cannot be empty.",
                    self.view.first_entry_focus)
            return False
        return True

    def hide(self):
        """
        Sets callbacks to None and hides the dialog.
        """
        self.cb = self.r_cb = None
        self.view.hide(True)


class PasswordInfoController(object):
    """
    Password Info dialog controller.

    @type title: string
    @param title: Dialog title.
    @type cb: callback
    @param cb: Callback used when OK button is clicked.
    @type r_cb: callback
    @param r_cb: Callback used when Cancel buttons is clicked.
    """

    def __init__(self, title, cb, r_cb):
        self.__title = title
        self.cb = callable(cb) and cb or None
        self.r_cb = callable(r_cb) and r_cb or None

    def show(self):
        """
        Creates and shows Password Info dialog.
        """
        self.view = ManageAccountDlgView(self.__title, self.__confirm_cb,
                self.__return, "manageaccountdlg2")
        self.view.show()

    def __confirm_cb(self, *p):
        """
        Method called when OK button is clicked. It is a wrapper to
        C{self.cb()}.

        @type p: set of arguments
        @param p: Not used.
        """
        self.cb()

    def __return(self, *p):
        """
        Method called when Cancel button is clicked. It is a wrapper to
        C{self.r_cb()}.

        @type p: set of arguments
        @param p: Not used.
        """
        self.r_cb()
        self.cb = self.r_cb = None

    def hide(self):
        """
        Sets callbacks to None and hides the dialog.
        """
        self.cb = self.r_cb = None
        self.view.hide(True)


class ServerInfoController(object):
    """
    Server Info dialog controller.

    @type title: string
    @param title: Dialog title.
    @type cb: callback
    @param cb: Callback used when OK button is clicked.
    @type r_cb: callback
    @param r_cb: Callback used when Cancel buttons is clicked.
    """
    def __init__(self, title, cb, r_cb):
        self.__title = title
        self.cb = callable(cb) and cb or None
        self.r_cb = callable(r_cb) and r_cb or None

    def show(self):
        """
        Creates and shows Server Info dialog view.
        """
        self.view = ManageAccountDlgView(self.__title, self.__confirm_cb,
                self.__return,
                "manageaccountdlg1")
        self.view.part_text_set("label", "Server")
        self.view.part_text_set("label2", "Port")
        self.view.part_text_set("cancel_bt_label", "Back")
        self.view.part_text_set("confirm_bt_label", "Save")
        self.view.show()

    def __confirm_cb(self, *p):
        """
        Method called when OK button is clicked. It is a wrapper to
        C{self.cb()}.

        @type p: set of arguments
        @param p: Not used.
        """
        if self.check_server() and self.check_port():
            self.cb()
            return False
        return True

    def __return(self, *p):
        """
        Method called when Cancel button is clicked. It is a wrapper to
        C{self.r_cb()}.

        @type p: set of arguments
        @param p: Not used.
        """
        self.r_cb()
        self.cb = self.r_cb = None

    def check_server(self):
        """
        Checks if entered server address is valid.

        @rtype: boolean
        @return: C{True} if server address is valid, C{False} otherwise.
        """
        if self.view.first_entry.text.strip()  == "":
            message_error(self.__title, "Server cannot be empty.",
                    self.view.first_entry_focus)
            self.view.first_entry.text = ""
            return False
        return True

    def check_port(self):
        """
        Checks if entered server port is valid.

        @rtype: boolean
        @return: C{True} if server port is valid, C{False} otherwise.
        """
        if not re.match("[0-9]*$", self.view.second_entry.text) or \
                self.view.second_entry.text.strip() == "":
            message_error(self.__title, \
                "Port field can only contain numbers.", \
                self.view.second_entry_focus)
            self.view.second_entry.text = ""
            return False
        return True

    def hide(self):
        """
        Sets callbacks to None and hides the dialog.
        """
        self.cb = self.r_cb = None
        self.view.hide(True)
