#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{MainView}.

@var    WIDTH: Default Window Width.
@var    HEIGHT: Default Window Height.
"""

import sys, etk
import ecore.evas, edje, ecore
from common.singleton import Singleton

WIDTH = 800
HEIGHT = 480

class MainView(Singleton):
    """
    Viewer for the L{Main} class. Initialize the Evas canvas.
    """
    def __init__(self):
        Singleton.__init__(self)
        self.view = None
        self.controller = None
        if ecore.evas.engine_type_supported_get("software_16_x11"):
            self.ee = ecore.evas.SoftwareX11_16(w=WIDTH, h=HEIGHT)
        else:
            self.ee = ecore.evas.SoftwareX11(w=WIDTH, h=HEIGHT)
        self.canvas = self.ee.evas
        self.ee.title_set("Carman")
        self.ee.name_class = ("Carman", "carman-evas")
        self.ee.fullscreen_set("-w" not in sys.argv)
        self.ee.show()
        self.icon = self.canvas.Image()
        edje.frametime_set(1.0 / 24)

    def get_evas_canvas(self):
        """
        Returns the Evas canvas.

        @rtype: C{evas.Canvas}
        @return: Evas canvas object.
        """
        return self.canvas

    def __get_transiting(self):
        """
        Returns whether the plugin is changing.

        @rtype: boolean
        @return: C{True} if transiting from plugins (C{Maps}, C{Gauges},
                C{Trips}), C{False} otherwise.
        """
        if self.controller:
            return self.controller.transiting
        else:
            return False
    transiting = property(__get_transiting, doc="C{True} if transiting from \
        plugins (C{Maps}, C{Gauges}, C{Trips}), C{False} otherwise")

    def setup(self, controller, theme):
        """
        Sets the main window theme and callbacks to the buttons(Exit,
        Change Plugin, Options button and Status).

        @type   controller: L{MainController}
        @param  controller: L{MainController} object.
        @type   theme: string
        @param  theme: Current theme filename path.
        """
        if self.view:
            self.view.delete()
        self.view = edje.Edje(self.canvas, file=theme, group="mainctrl")
        self.view.size = self.canvas.size
        self.view.layer_set(15)
        self.view.part_swallow("icon", self.icon)
        self.view.show()
        etk.theme_widget_set_from_path(theme)
        self.controller = controller

        self.view.signal_callback_add("exit-button-pressed", "*",
            self.controller.exit_button)
        self.view.signal_callback_add("change-button-pressed", "*",
            self.controller.change_button)
        self.view.signal_callback_add("option-button-pressed", "*",
            self.controller.option_button)
        self.view.signal_callback_add("status-button-pressed", "*",
            self.controller.status_button)

    def set_icon(self, file):
        """
        Sets the current plugin's icon.

        @type   file: tuple
        @param  file: Tuple of C{(Filename, Icon Images)}
        """
        self.icon.file_set(*file)
        w, h = self.icon.image_size_get()
        self.icon.size_set(w, h)
        self.icon.fill_set(0, 0, w, h)

    def show_splashscreen(self, theme):
        """
        Displays the splash screen.
        
        @type   theme: string
        @param  theme: Current theme filename path.
        """
        self.splashscreen = self.canvas.Image(file=(theme,
            "images/splashscreen"))
        w, h = self.splashscreen.image_size_get()
        self.splashscreen.size_set(*self.canvas.size)
        self.splashscreen.fill_set(0, 0, *self.canvas.size)
        self.splashscreen.show()

    def hide_splashscreen(self):
        """
        Hides the splash screen.
        """
        self.splashscreen.delete()

