#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{GaugeController}.
"""

import evas

class GaugeController(object):
    """
    Controller class for a single Gauge.

    @type   screen: Edje object
    @param  screen: Edje screen object.
    @type   sensor: object
    @param  sensor: Sensor object.
    @type   metric: boolean
    @param  metric: C{True} for Metric or C{False} for Imperial.
    """
    def __init__(self, screen, sensor, metric):
        self.screen = screen
        self.sensor = sensor
        self.metric = metric
        self.__last_value = None
        self.unit_part_name = ""
        self.scale = [1, 1]
        self.offset = 0
        self.frequency = 1
        self.pid = sensor.get_pid()
        self.min = [sensor.get_min(False), sensor.get_min(True)]
        self.max = [sensor.get_max(False), sensor.get_max(True)]

    def destroy(self):
        """
        Called when the L{evas.SmartObject} is closed.
        """
        if isinstance(self.view, evas.SmartObject):
            self.screen.part_unswallow(self.view)
            self.view.delete()

    def set_title(self, part_name):
        """
        Sets the object title.

        @type   part_name: string
        @param  part_name: Object part name.
        """
        if part_name:
            title = self.sensor.get_short_description()
            if title:
                self.screen.part_text_set(part_name, title.encode("utf-8"))

    def __set_unit_text(self):
        """
        Sets the unit's text.
        """
        if self.unit_part_name:
            unit = self.sensor.get_unit(self.metric)
            if unit:
                if self.scale[self.metric] > 1:
                    unit += " x%d" % self.scale[self.metric]
                self.screen.part_text_set(self.unit_part_name,
                    unit.encode("utf-8"))

    def set_unit(self, part_name):
        """
        Sets the object's unit.

        @type   part_name: string
        @param  part_name: Object part name.
        """
        self.unit_part_name = part_name
        self.__set_unit_text()

    @staticmethod
    def __conv_to_int(value, default):
        """
        Convert a string to its integer representation.

        @type   value: string
        @param  value: Value to be converted.
        @type   default: int
        @param  default: Default value if there an error ocurred
                         while converting the value.

        @raise Exception: Error while converting I{value}.

        @rtype: int
        @return: Integer representation of the string.
        """
        try:
            return int(value)
        except:
            WARNING("Error converting value \"%s\"" % value)
            return default

    def set_min(self, metric, imperial):
        """
        Sets the minimum values for Metric and Imperial units.

        @type   metric: string
        @param  metric: Metric value.
        @type   imperial: string
        @param  imperial: Imperial value.
        """
        if metric:
            self.min[True] = self.__conv_to_int(metric, self.min[True])
            if self.min[True] == self.max[True]:
                self.min[True] -= 1
        if imperial:
            self.min[False] = self.__conv_to_int(imperial, self.min[False])
            if self.min[False] == self.max[False]:
                self.min[False] -= 1

    def set_max(self, metric, imperial):
        """
        Set maximum values for metric and imperial units.

        @type   metric: string
        @param  metric: Metric value.
        @type   imperial: string
        @param  imperial: Imperial value.
        """
        if metric:
            self.max[True] = self.__conv_to_int(metric, self.max[True])
            if self.max[True] == self.min[True]:
                self.max[True] += 1
        if imperial:
            self.max[False] = self.__conv_to_int(imperial, self.max[False])
            if self.max[False] == self.min[False]:
                self.max[False] += 1

    def set_scale(self, metric, imperial):
        """
        Sets the scale values for Metric and Imperial units.

        @type   metric: string
        @param  metric: Metric value.
        @type   imperial: string
        @param  imperial: Imperial value
        """
        if metric:
            self.scale[True] = self.__conv_to_int(metric, self.scale[True])
            if self.scale[True] == 0:
                self.scale[True] = 1
        if imperial:
            self.scale[False] = self.__conv_to_int(imperial,
                self.scale[False])
            if self.scale[False] == 0:
                self.scale[False] = 1

    def set_frequency(self, value):
        """
        Sets the OBD command frequency.

        @type   value: string
        @param  value: Command frequency
        """
        if value:
            self.frequency = self.__conv_to_int(value, self.frequency)

    def set_offset(self, value):
        """
        Sets the OBD command offset.

        @type   value: string
        @param  value: Command offset.
        """
        if value:
            self.offset = self.__conv_to_int(value, self.offset)

    def get_obd_command(self):
        """
        Returns OBD command.

        @rtype: tuple
        @return: OBD command C{(Sensor_Pid, Frequency, Offset)}.
        """
        return (self.pid[:2], self.frequency, self.offset)

    def set_unit_system(self, metric):
        """
        Sets the unit system value.

        @type   metric: boolean
        @param  metric: C{True} for Metric or C{False} for Imperial.
        """
        self.metric = metric
        self.__set_unit_text()
        self.view.set_unit_system(self.min[metric], self.max[metric],
            self.scale[metric], self.sensor.get_unit(metric))

    def set_view(self, part_name, view):
        """
        Sets the object view.

        @type   part_name: string
        @param  part_name: Object part name.
        @type   view: C{evas.SmartObject}
        @param  view: Object View.
        """
        self.view = view
        self.set_unit_system(self.metric)
        if isinstance(view, evas.SmartObject):
            self.screen.part_swallow(part_name, view)

    def set_value(self, data):
        """
        Sets the current value.

        @type   data: int
        @param  data: Value absolute.
        """
        if data:
            index = not self.metric and 2 or 0
            if ":" in self.pid:
                index += int(self.pid.split(":")[-1]) - 1
            value = data[index]

            if value != self.__last_value:
                value = max(value, self.min[self.metric])
                value = min(value, self.max[self.metric])
                value_rel = float(value - self.min[self.metric]) / \
                    (self.max[self.metric] - self.min[self.metric])
                self.view.set_value(value, value_rel)
                self.__last_value = value

