# util.py - Utility functions
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
Implements util functions
"""

from common.carlog import WARNING

def merge_dicts(dict1, dict2):
    """
    Merges two dictionaries.

    @type dict1: dictionary
    @param dict1: The first dictionary to be merged.
    @type dict2: dictionary
    @param dict1: The second dictionary to be merged.
    """
    dict = {}
    for key, value in dict1.items():
        dict[key.lower()] = value
    for key, value in dict2.items():
        dict[key.lower()] = value

    return dict

def conv_color(color):
    """
    Converts color to RGB
    @rtype: tuple
    @return: RGB color.
    """
    try:
        return (int(color[:2], 16), int(color[2:4], 16), int(color[4:], 16))
    except:
        WARNING("color '%s' invalid !!!" % color)
        return (255, 255, 255)

def get_param(dict, id, value_default):
    """
    Returns the value from the given key of the dictionary.

    @type dict: dictionary
    @param dict: Dictionary with the values.
    @type id: string
    @param id: Dictionary key.
    @type value_default: string
    @param value_default: Default value to be returned.
    @rtype string
    @return: Value to C{id}
    """
    try:
        return dict[id]
    except:
        WARNING("key '%s' not found !!!" % id)
        return value_default

def get_param_int(dict, id, value_default):
    """
    Returns the value to gave key of the dictionary.

    @type dict: dictionary
    @param dict: Dictionary with the values.
    @type id: string
    @param id: Dictionary key.
    @type value_default: int
    @param value_default: Default value to be returned.
    @rtype int
    @return: Value to C{id}
    """
    try:
        return int(float(dict[id]))
    except ValueError:
        WARNING("error converting %s to integer" % dict[id])
        return value_default
    except KeyError:
        WARNING("key %s not found !!!" % id)
        return value_default

def conv_to_unicode(text):
    """
    Converts a string to unicode text.

    @type text: string
    @param text: String to be converted.
    @rtype string
    @return: Text unicode.
    """
    if not isinstance(text, unicode):
        return unicode(text, "utf-8")

    return text

def rect_intersection(rect, rects):
    """
    Verifies intersection between rectangles.

    @type rect: tuple
    @param rect: Rectangle to verify intersection.
    @type rects: list
    @param rect: List of rectangles to verify intersection.
    @rtype: boolean
    @return: C{True} if L{rect} intersects L{rects}, C{False} otherwise
    """
    w = rect[0] + rect[2]
    h = rect[1] + rect[3]
    for r in rects:
        if r[0] + r[2] > rect[0] and w > r[0] and \
          r[1] + r[3] > rect[1] and h > r[1]:
            return True
    return False
