# -*- coding: utf-8 -*-
#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging
import os
import urllib2
import urlparse
from feedparser import parse as parsefeed
from htmllib import HTMLParser
from time import time

from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.core.plugin_prefs import PluginPrefs
from terra.core.threaded_func import ThreadedFunction
from terra.utils.encoding import to_utf8

from common import STATE_INITIAL, STATE_DOWNLOAD_DIALOG_OPENING, \
    STATE_DOWNLOADING, STATE_PAUSED, STATE_DOWNLOADED, STATE_QUEUED, \
    state_mapping


log = logging.getLogger("plugins.canola-core.ondemand.model.folder")
mger = Manager()
db = mger.canola_db

DownloadManager = mger.get_class("DownloadManager")
download_mger = DownloadManager()
network = mger.get_status_notifier("Network")


def whoami():
    import sys
    return sys._getframe(1).f_code.co_name


class OnDemandFolder(ModelFolder):
    terra_type = "Model/Folder/Media/OnDemand"
    feed_table = None
    item_table = None
    model_cls = None # OnDemandModel

    layout = {
        1: ("(id INTEGER PRIMARY KEY, uri VARCHAR UNIQUE, "
            "title VARCHAR, desc VARCHAR, cover VARCHAR, "
            "epoch INTEGER NOT NULL)"),

        2: ("(id INTEGER PRIMARY KEY, uri VARCHAR UNIQUE, "
            "last_updated VARCHAR, title VARCHAR, desc VARCHAR, "
            "epoch INTEGER NOT NULL)"),

        3: ("(id INTEGER PRIMARY KEY, uri VARCHAR UNIQUE, "
            "last_updated VARCHAR, title VARCHAR, desc VARCHAR, "
            "epoch INTEGER NOT NULL)"),
    }

    statements = {
        'update-from-1':
            "INSERT INTO %s (uri, title, desc, epoch) "
            "SELECT uri, title, desc, epoch FROM %(feed-table)s_backup",
        'update-from-2':
            "INSERT INTO %s (uri, title, desc, epoch) "
            "SELECT uri, title, desc, epoch FROM %(feed-table)s_backup",

        'create':           "CREATE TABLE IF NOT EXISTS "
                            "%(feed-table)s %(layout)s",
        'table-info':       "PRAGMA TABLE_INFO(%(feed-table)s)",
        'insert':           "INSERT INTO %(feed-table)s "
                            "(uri, title, desc, last_updated,epoch) "
                            "values (?, ?, ?, ?, ?)",
        'delete':           "DELETE FROM %(feed-table)s WHERE id = ?",
        'exists':           "SELECT * FROM %(feed-table)s WHERE uri == ?",
        'select':           "SELECT id, uri, title, desc, last_updated "
                            "FROM %(feed-table)s",
        'select-title':     "SELECT title FROM %(feed-table)s WHERE id == ?",
        'select-order-by':  "SELECT id, uri, title, desc, last_updated FROM "
                            "%(feed-table)s ORDER BY %(field)s",
        'entries':          "SELECT COUNT(*) from %(feed-table)s",
        'update-title':     "UPDATE %(feed-table)s SET title = ? "
                            "WHERE id == ?",
        'update-cache-tag': "UPDATE %(feed-table)s SET last_updated = ? "
                            "WHERE id == ?",
    }

    @classmethod
    def preprocess_query(cls, query, **kargs):
        params = { 'feed-table': cls.feed_table,
                   'item-table': cls.item_table }
        params.update(kargs)
        real_query = query % params
        return real_query

    @classmethod
    def execute_stmt(cls, stmt, *args, **kargs):
        query = cls.preprocess_query(cls.statements[stmt], **kargs)
        return db.execute(query, args)

    @classmethod
    def execute_stmt_with_cursor(cls, stmt, cur, *args, **kargs):
        query = cls.preprocess_query(cls.statements[stmt], **kargs)
        return cur.execute(query, args).fetchall()

    def __init__(self, title, uri, parent=None):
        self._id = None

        self._title = None
        self.uri = uri
        self.name = self.title = title

        self.desc = None
        self._cache_tag = None

        self.download_process = None
        self.refresh_finished_callback = None

        ModelFolder.__init__(self, self.title, parent)

    def __get_id(self):
        return self._id

    def __set_id(self, value):
        self._id = int(value)

    id = property(__get_id, __set_id)

    def __get_desc_visible(self):
        # Always get an uptodate settings!
        s = PluginPrefs("settings")
        return s.get("ondemand_description_visible", False)

    def __set_desc_visible(self, value):
        if not isinstance(value, bool):
            return

        # Always get an uptodate settings!
        s = PluginPrefs("settings")
        s['ondemand_description_visible'] = value
        s.save()

    description_visible = property(__get_desc_visible, __set_desc_visible)

    def __get_id(self):
        return self._id

    def __set_id(self, value):
        self._id = int(value)
        cover_path = self.settings.get("cover_path")
        self.cover = os.path.join(cover_path,
            "podcasts/%d/cover.jpg" % int(value))

    id = property(__get_id, __set_id)

    def __get_cache_tag(self):
        return self._cache_tag

    def __set_cache_tag(self, value):
        self._cache_tag = value
        self.execute_stmt('update-cache-tag', value, self.id)

    cache_tag = property(__get_cache_tag, __set_cache_tag)

    # accesses db
    def __set_title(self, value):
        self._title = value

        if self.id is None:
            return

        self.execute_stmt('update-title', value, self.id)

    def __get_title(self):
        return self._title

    title = property(__get_title, __set_title)

    def _insert(self):
        values = (self.uri, self.title, self.desc,
                  self.cache_tag, int(time()))
        cur = db.get_cursor()
        self.execute_stmt_with_cursor('insert', cur, *values)
        self.id = cur.lastrowid
        db.commit()

    def commit(self):
        already_exists = self.execute_stmt('exists', self.uri)
        if not already_exists:
            self._insert()

    def is_acceptable_format(self, uri):
        return True

    def get_uri_from_feed_entry(self, entry):
        try:
            return to_utf8(entry.enclosures[0].href)
        except:
            if entry.has_key("link"):
                uri = to_utf8(entry.link)
            return uri
        return None

    def delete(self):
        """Delete a podcast, its episodes and related files."""
        log.debug("Deleting podcast '%s'" % self.title)
        self.reload_children()

        for child in self.children:
            log.info("Deleting episode '%s'" % child.title)
            if child.uri is not None:
                info, child.downloader = \
                    download_mger.get_info_for_target(child.uri)
            child.purge()
        episodes = None

        self.model_cls.execute_stmt('delete-by-feed', self.id)
        self.execute_stmt('delete', self.id)

        path = os.path.dirname(self.cover)
        if os.path.exists(path):
            import shutil
            shutil.rmtree(path)
        db.commit()

    def child_property_changed(self, child):
        pass

    def reload_children(self):
        self.children.freeze()
        try:
            # unload and delete existing .children
            for c in self.children:
                c.unload()
            del self.children[:]

            rows = self.model_cls.execute_stmt('select', self.id)

            cache = dict()
            for r in rows:
                c = self.child_model_from_db_row(r, cache)
                self.append(c)
            cache = None
        finally:
            self.children.thaw()

    def is_updated(self, uri, cache_tag):
        uri_info = urlparse.urlparse(uri)
        address = uri_info.geturl().split("@")[-1].split("://")[-1]

        password = uri_info.password
        username = uri_info.username

        uri = uri_info.scheme + "://" + address
        server = uri_info.netloc.split("@")[-1]

        log.debug("check if %s is updated (%s)", uri, cache_tag)

        if password is not None and username is not None:
            passman = urllib2.HTTPPasswordMgrWithDefaultRealm()
            passman.add_password(None, server, username, password)
            authhandler = urllib2.HTTPBasicAuthHandler(passman)
            opener = urllib2.build_opener(authhandler)
            urllib2.install_opener(opener)
        else:
            opener = urllib2.build_opener()

        request = urllib2.Request(uri)
        if cache_tag is not None and cache_tag.startswith("ETag:"):
            etag = self.cache_tag[5:].strip()
            request.add_header("if-none-match", etag)

        try:
            request.add_header("allow", "HEAD")
            datastream = opener.open(request)
        except urllib2.HTTPError, e:
            if e.code == 304:
                # not modified
                return cache_tag
            else:
                raise e

        if datastream.headers.dict.has_key("etag"):
            cache_tag = "ETag: " + datastream.headers.dict['etag']
        elif datastream.headers.dict.has_key("last-modified"):
            cache_tag = datastream.headers.dict['last-modified']
        else:
            # None means that we have no info, thus we will try
            # to update the feed.
            cache_tag = None

        return cache_tag

    def on_refresh_finished(self, feed, was_updated):
        if was_updated:
            self.reload_children()
            self.notify_model_changed()
        if self.is_loading:
            self.inform_loaded()
        self.refresh_finished_callback = None # remove ourselves

    def do_load(self):
        self.reload_children()
        self.is_loading = self.refresh(end_callback=self.on_refresh_finished)

    def inform_refreshed(self, updated):
        if self.refresh_finished_callback is not None:
            self.refresh_finished_callback(self, updated)

    def refresh(self, force=False, end_callback=None):
        def cb(*a):
            if end_callback is not None:
                end_callback(*a)
            self.refresh_finished_callback = None # remove itself
        self.refresh_finished_callback = cb
        return self.do_refresh(force)

    def do_refresh(self, force=False):
        self.cancel_refresh = False

        def check_updated(exception, cache_tag):
            if exception is not None:
                log.debug("Exception during checking if feed was modified: %s",
                          exception)
                return

            log.debug("feed %s received cache_tag %s", self.name, cache_tag)

            if cache_tag is None or cache_tag != self.cache_tag:
                self.cache_tag = cache_tag
                log.debug("feed %s will be updated", self.name)
                fetch_feed()
            elif force:
                log.debug("feed %s will be updated", self.name)
                fetch_feed()
            else:
                self.inform_refreshed(False)

        def fetch_feed():
            self.download_process = download_mger.add(self.uri,
                ignore_limit=True)
            self.download_process.on_finished_add(fetching_finished)
            self.download_process.on_cancelled_add(disconnect)
            self.download_process.start(compress=True)

        def disconnect():
            self.download_process.on_finished_remove(fetching_finished)
            self.download_process.on_cancelled_remove(disconnect)
            self.download_process = None

        def remove_feed_file(feed_path):
            if os.path.exists(feed_path):
                try:
                    os.unlink(feed_path)
                except:
                    pass

        def fetching_finished(exception, mimetype):
            feed_path = self.download_process.target
            disconnect()

            if exception is not None:
                remove_feed_file(feed_path)
                if isinstance(exception, urllib2.URLError):
                    args = exception.args[0]
                    if isinstance(args, tuple):
                        log.warning("Download process: [errno: %s] %s" % args)
                    else:
                        log.warning("Download process: %s" % args)
                else:
                    raise exception

            if self.cancel_refresh:
                remove_feed_file(feed_path)
                log.info("%s deleted before loading, cancelling list " \
                         "refresh then." % self.__class__.__name__)
                self.inform_refreshed(False)
                return

            t = ThreadedFunction(parsing_finished, parse_feed, feed_path)
            t.start()

        def parse_feed(feed_path):
            try:
                data = parsefeed(feed_path)
            except Exception, e:
                log.error("Exception during feed parsing %s", e)
                remove_feed_file(feed_path)
                return None
            remove_feed_file(feed_path)

            return data

        def parsing_finished(exception, feed):
            if exception is not None:
                raise Exception("%s() thread raised an exception: %s" \
                    % (whoami(), exception))

            if self.cancel_refresh:
                log.info("%s deleted before loading, cancelling list " \
                         "refresh then." % self.__class__.__name__)

            if feed is None or self.cancel_refresh:
                self.inform_refreshed(False)
                return

            self.fetch_cover(feed)

            cur = db.get_cursor()
            for entry in feed.entries:
                uri = self.get_uri_from_feed_entry(entry)
                if uri is None \
                    or not self.is_acceptable_format(uri) \
                    or self.child_model_exists_in_db(uri, cur=cur):
                    continue

                c = self.child_model_from_feed_entry(entry)
                c._insert(cur)

            db.commit()
            cur.close()

            self.inform_refreshed(True)

        if not network or not network.status > 0.0:
            self.inform_refreshed(False)
            return False

        # let's check if it was modified or not need to refresh.
        s = ThreadedFunction(check_updated, self.is_updated,
                             self.uri, self.cache_tag)
        s.start()

        return True # is still loading

    def fetch_cover(self, contents):
        feed = contents.feed
        cover_path = PluginPrefs("settings").get("cover_path")
        cover = os.path.join(cover_path,
            "podcasts/%d/cover.jpg" % self.id)

        if os.path.exists(cover):
            return

        if feed.has_key("image"):
            path = os.path.dirname(cover)
            if not os.path.exists(path):
                os.makedirs(path)

            cover_process = download_mger.add(feed.image.href, cover,
                                              ignore_limit=True)
            cover_process.start()

    def do_unload(self):
        if self.download_process is not None:
            self.download_process.cancel()

        self.commit_changes()

        self.cancel_refresh = True

        # parent method unloads and deletes .children
        ModelFolder.do_unload(self)

    def sync_states(self, item, downloadmger_state):
        # Note: We change private data_state (which do not inform
        # changes as we only call sync_states in construction face.
        state = downloadmger_state
        item._data_state = STATE_INITIAL

        for model_state in state_mapping.iterkeys():
            if state in state_mapping[model_state]:
                item._data_state = model_state
                break

        if item._data_state == STATE_DOWNLOADING:
            item._add_download_process()
            item.fetch()

        if item._data_state == STATE_PAUSED:
            item._add_download_process()
            item.pause()

        return item

    def _ensure_valid_state(self, item):
        # 1. The most trustworthy state is that of the downloadmanager as
        #    changes might not have been saved.

        info, item.downloader = \
            download_mger.get_info_for_target(item.uri or "")

        if item.downloader is not None:
            # automatically sets the correct state, when cbs are connected
            item.connect_callbacks(item.downloader)
            item.sync_progress()
            return item

        if item.uri is not None and os.path.exists(item.uri) \
            and item.filesize != -1:
            cursize = os.path.getsize(item.uri)
            item._progress = cursize / float(item.filesize)
        else:
            item._progress = 0.0

        # 2. The second most trustworth state is looking at the info

        if info is not None:
            item.uri = info['filename']
            self.sync_states(item, info['state'])
            return item

        # 3. If the file is present it is downloaded, orelse we know nothing

        # Note: We change private data_state (which do not inform
        # changes as we only call sync_states in construction face.
        if item.uri and os.path.exists(item.uri):
            item._progress = 1.0
            item._data_state = STATE_DOWNLOADED
        else:
            item._progress = 0.0
            item._data_state = STATE_INITIAL

        return item

    def child_model_exists_in_db(self, uri, cur=None):
        try:
            c = self.model_cls
            if cur:
                already_exists = c.execute_stmt_with_cursor('exists', cur, uri)
            else:
                already_exists = c.execute_stmt('exists', uri)

            if already_exists:
                return True
            else:
                return False
        except:
            return False

    def child_model_from_feed_entry(self, entry):
        child = self.model_cls(self)
        child.use_data_from_feed_entry(entry)
        child.feed_id = self.id

        return child

    def child_model_from_db_row(self, row, cache=None):
        child = self.model_cls(self)
        child.use_data_from_db_row(row, cache)

        child = self._ensure_valid_state(child)
        return child

    def commit_changes(self):
        mger.canola_db.commit()

    def next(self, circular=False, same_type=False):
        progress = 0.0
        current_bak = self.current
        while progress == 0.0:
            try:
                next_obj = ModelFolder.next(self, circular, same_type)
                progress = next_obj.progress
            except:
                # restore the current pointer in case no next was found
                self.current = current_bak
                raise

        return next_obj

    def prev(self, circular=False, same_type=False):
        progress = 0.0
        current_bak = self.current
        while progress == 0.0:
            try:
                prev_obj = ModelFolder.prev(self, circular, same_type)
                progress = prev_obj.progress
            except:
                # restore the current pointer in case no prev was found
                self.current = current_bak
                raise

        return prev_obj

    def options_model_get(self, controller):
        return None


