#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import logging

from terra.core.terra_object import TerraObject
from terra.core.manager import Manager
from ConfigParser import ConfigParser

__all__ = ("ListPlaylistMedia",)

log = logging.getLogger("canola.audio.playlist")
mger = Manager()

CanolaError = mger.get_class("Model/Notify/Error")


class ListPlaylistMedia(TerraObject):
    """Basic class to create a list of children to a ModelFolder,
    from a database.

    """

    def __init__(self, parent, path, n_items):
        self.parent = parent
        self.path = str(path)
        self.dirname = os.path.dirname(self.path)

        try:
            playlist = open(path)
            if self.path.endswith("m3u"):
                self.lines = self._parse_m3u(playlist)
            elif self.path.endswith("pls"):
                self.lines = self._parse_pls(playlist)
            else:
                raise TypeError("Unsupported file format.")
        except Exception, e:
            log.error("Could not load playlist file: %s", e)

        self.clause = "SELECT * FROM audios,files USING (id) WHERE " \
                      "files.dtime = 0 AND files.path LIKE ?"

        self.n_items = n_items
        self._elements = [None] * self.n_items

    def _parse_pls(self, fp):
        p = ConfigParser()
        p.readfp(fp)

        try:
            num = int(p.get("playlist", "NumberOfEntries")) + 1
        except:
            return contents

        paths = []
        for i in xrange(1, num):
            item = p.get("playlist", "File%d" % i)
            if not os.path.exists(item):
                item = os.path.realpath("%s/%s" % (self.dirname, item))
            paths.append(item)

        return paths

    def _parse_m3u(self, fp):
        lines = fp.readlines()[:]

        if lines[0].startswith("#EXTM3U"):
            lines = lines[1:]

        i = 0
        paths = []
        for line in lines:
            line = line.strip("\n").strip("\r")

            if line.startswith("#EXTINF:") or line.isspace():
                continue
            else:
                if not os.path.exists(line):
                    line = os.path.realpath("%s/%s" % (self.dirname, line))
                paths.append(line)

        return paths

    def _create_item(self, index, item):
        raise NotImplementedError("You must implement this")

    def __setitem__(self, key, item):
        self._elements[key] = item

    def __len__(self):
        return self.n_items

    def _get_slice(self, idx):
        if isinstance(idx, slice):
            start = idx.start or 0
            stop = idx.stop or self.n_items
            idx_slice = slice(start, stop, None)
        else:
            idx_slice = slice(idx, idx + 5, None)

        return idx_slice

    def __getitem__(self, idx):
        idx_slice = self._get_slice(idx)

        if None in self._elements[idx_slice]:
            index = idx_slice.start
            if idx_slice.stop < self.n_items:
                stop = idx_slice.stop
            else:
                stop = self.n_items

            cur = mger.canola_db.get_cursor()
            while index < stop:
                if self._elements[index] is None:
                    item = self.lines[index]
                    cur.execute(self.clause, ("%" + item,))
                    row = cur.fetchone()
                    if row:
                        self._elements[index] = self._create_item(row)
                    else:
                        row = [0, item, 0, 0, 0, 0, 0, "/.mp3"]
                        self._elements[index] = self._create_item(row)
                        notify = CanolaError("File not found.")
                        self._elements[index].state_reason = notify
                index += 1

        return self._elements[idx]

    def __delitem__(self, idx):
        if self._elements is None:
            return

        if isinstance(idx, slice):
            stop = idx.stop or len(self)
            start = idx.start or 0
            size = stop - start
        else:
            size = 1
        del self._elements[idx]
        self.n_items -= size

    def reset(self):
        self._elements = [None] * self.n_items
