# Canola2 Youtube Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import glob
import time
import ecore
import urllib
import urllib2
import socket
import logging
import cPickle as pickle

from terra.core.task import Task
from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction
from downloadmanager.constants import DownloadState, ExceptionType

from client import Client
from utils import get_video_path, get_thumb_path, \
    round_rating, space_available

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite


manager = Manager()
DownloadManager = manager.get_class("DownloadManager")
PluginDefaultIcon = manager.get_class("Icon/Plugin")
CanolaError = manager.get_class("Model/Notify/Error")
ActionModelFolder = manager.get_class("Model/Options/Action")
OptionsModelFolder = manager.get_class("Model/Options/Folder")
BaseVideoLocalModel = manager.get_class("Model/Media/Video/Local")

SystemProps = manager.get_class("SystemProperties")
system_props = SystemProps()

download_manager = DownloadManager()

log = logging.getLogger("plugins.canola-tube.youtube.model")


class Icon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Video/YouTube"
    icon = "icon/main_item/youtube"
    plugin = "youtube"


class VideoLocalModel(BaseVideoLocalModel):
    terra_type = "Model/Media/Video/YouTubeLocal"

    SPARE_SPACE = (1024 ** 2) # 1 mb
    SAVED_STAMP = ".saved"
    DOWNLOADED_STAMP = ".downloaded"

    def __init__(self, parent):
        BaseVideoLocalModel.__init__(self, parent)
        self.id = None
        self.size = 0
        self.rating = 0
        self.visit_time = 0
        self.description = None
        self.remote_uri = None
        self.small_thumbnail = None
        self.large_thumbnail = None
        self.updated_date = None
        self.published_date = None
        self.rounded_rating = 0

        self.callback_notify = None
        self.callback_download_progress = None
        self.callback_download_finished = None

        self.timer = None
        self.down_item = None

    def do_load(self):
        self.unloaded = False
        self.progress = 0.0
        self.bytes_written = 0
        self.play_local = False

        self.remove_temporary_videos()

        log.debug("refreshing history for ID %s" % self.id)
        self.visit_time = int(time.time())
        HistoryModelFolder.refresh_model(self)

        if os.path.exists(self.uri):
            self.progress = 1.0
            self.play_local = True
            log.warning("video local play")
            return

        def resolve_finished(uri):
            log.debug("url resolved for ID %s: %s" % (self.id, uri))

            if not uri:
                if self.callback_notify:
                    error = CanolaError("Unable to resolve video url.<br>"
                                        "Check your connection and try again.")
                    self.callback_notify(error)
                log.error("Unable to resolve video uri")
                return

            self.remote_uri = uri
            self.add_download_process(self.remote_uri, self.uri)

        # youtube video uri changes every time
        log.debug("trying to resolve url for ID %s" % self.id)
        Client.resolve_video_url_async(self.id, resolve_finished)

    def add_download_process(self, uri, local_path):
        if self.unloaded:
            return

        try:
            system_props.prepare_write_path(os.path.dirname(local_path),
                                            "Download folder")
        except Exception, e:
            log.error(e.message)
            self.callback_notify(CanolaError(e.message))
            return

        if not space_available(os.path.dirname(local_path), self.SPARE_SPACE):
            log.error("No space left on device")
            self.callback_notify(CanolaError("No space left on device"))
            return

        self.down_item = download_manager.add(uri, local_path,
                                              ignore_limit=True)
        self.down_item.on_finished_add(self.cb_item_finished)
        self.down_item.on_queued_add(self.cb_item_queued)
        self.down_item.on_cancelled_add(self.cb_item_cancelled)
        self.down_item.on_download_started_add(self.cb_item_download_started)
        self.down_item.start(False)
        log.debug("added download process for uri %s" % uri)

    def remove_download_callbacks(self):
        if self.down_item is None:
            return

        self.down_item.on_finished_remove(self.cb_item_finished)
        self.down_item.on_cancelled_remove(self.cb_item_cancelled)
        self.down_item.on_download_started_remove(self.cb_item_download_started)

    def cb_item_queued(self, *ignored):
        log.warning("item queued")

    def do_unload(self):
        if self.down_item is not None:
            self.down_item.cancel()
        self.remove_temporary_videos()
        self.unloaded = True

    def remove_temporary_videos(self):
        video_path = get_video_path()
        msk = os.path.join(video_path, "*.flv")
        for filepath in glob.glob(msk):
            if os.path.exists(filepath + self.SAVED_STAMP):
                continue

            path, name = os.path.split(filepath)
            if os.path.exists(filepath + self.DOWNLOADED_STAMP):
                if self.id == name[:-4]:
                    continue
                else:
                    os.unlink(filepath + self.DOWNLOADED_STAMP)

            os.unlink(filepath)

    def request_thumbnail(self, end_callback=None):
        def request(*ignored):
            urllib.urlretrieve(self.small_thumbnail, self.thumb)

        def request_finished(exception, retval):
            if end_callback:
                end_callback()

        if not self.small_thumbnail or os.path.exists(self.thumb):
            if end_callback:
                end_callback()
        else:
            ThreadedFunction(request_finished, request).start()

    def _add_progress_timer(self):
        """Add timer to synchronize download progress bar"""
        if not self.timer and self.down_item:
            self.timer = ecore.timer_add(0.5, self.synchronize_progress)
            log.debug("Added a progress timer for '%s'" % self.uri)

    def _del_progress_timer(self):
        """Remove timer that synchronizes download progress bar"""
        if self.timer:
            self.timer.stop()
            self.timer.delete()
            self.timer = None
            log.debug("Removed progress timer for '%s'" % self.uri)

    def synchronize_progress(self, *ignore):
        if not self.down_item:
            return False

        bytes_written, size = self.down_item.get_progress()

        if bytes_written > 0:
            self.bytes_written = bytes_written

        if size > 0:
            self.progress = bytes_written / float(size)
        elif size == 0:
            self.progress = 0.0
        else:
            log.debug("Couldn't get size of '%s'" % self.uri)

        if self.callback_download_progress:
            self.callback_download_progress(self.progress)

        log.debug("Progress download of '%s': %f (%d)" \
                      % (self.id, self.progress, self.bytes_written))

        if not space_available(os.path.dirname(self.uri),
                               (size - bytes_written)  + self.SPARE_SPACE):
            self.down_item.cancel()
            self.down_item = None
            log.error("No space left on device")
            self.callback_notify(CanolaError("No space left on device"))
            return

        return (self.progress < 1.0)

    def cb_item_finished(self, exception, mimetype):
        self.remove_download_callbacks()
        self._del_progress_timer()
        self.synchronize_progress()
        self.progress = 1.0 # enforce precision
        self.down_item = None

        if self.callback_download_finished:
            self.callback_download_finished(exception)

    def cb_item_cancelled(self, *ignored):
        self.remove_download_callbacks()
        self._del_progress_timer()
        self.down_item = None

    def cb_item_download_started(self, *ignored):
        self._add_progress_timer()

    def has_milestone(self):
        return os.path.exists(self.uri + self.SAVED_STAMP) \
            or os.path.exists(self.uri + self.DOWNLOADED_STAMP)

    def has_saved_milestone(self):
        return os.path.exists(self.uri + self.SAVED_STAMP)

    def has_downloaded_milestone(self):
        return os.path.exists(self.uri + self.DOWNLOADED_STAMP)

    def save_milestone(self, final=False):
        info = {}
        info['id'] = self.id
        info['title'] = self.title
        info['artist'] = self.artist
        info['rating'] = self.rating
        info['view_count'] = self.view_count
        info['description'] = self.description

        if final:
            filename = self.uri + self.SAVED_STAMP
        else:
            filename = self.uri + self.DOWNLOADED_STAMP

        fp = open(filename, "wb", pickle.HIGHEST_PROTOCOL)
        pickle.dump(info, fp)
        fp.close()


class ServiceModelFolder(ModelFolder):
    terra_type = "Model/Folder/Task/Video/YouTube/Service"

    db = manager.canola_db
    threaded_search = True
    delete_possible = False
    empty_msg = "No videos found"

    # create table statement used for subclasses
    stmt_create = """CREATE TABLE IF NOT EXISTS %s
                     (
                        id VARCHAR PRIMARY KEY,
                        uri VARCHAR,
                        title VARCHAR,
                        author VARCHAR,
                        rating FLOAT,
                        duration INTEGER,
                        view_count INTEGER,
                        epoch INTEGER,
                        visit_time INTEGER,
                        description VARCHAR,
                        small_thumbnail VARCHAR,
                        large_thumbnail VARCHAR,
                        updated_date VARCHAR,
                        published_date VARCHAR
                     )"""

    # select statement used for subclasses
    stmt_select = """SELECT id, uri, title, author, rating, duration,
                            view_count, small_thumbnail, large_thumbnail,
                            description, updated_date, published_date,
                            visit_time
                     FROM %s WHERE id = ?"""

    # insert statement used for subclasses
    stmt_insert = """INSERT INTO %s (id, uri, title, author, rating, duration,
                                     view_count, epoch, visit_time, description,
                                     small_thumbnail, large_thumbnail,
                                     updated_date, published_date)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"""

    # delete statement used for subclasses
    stmt_delete = """DELETE FROM %s WHERE id = ?"""
    stmt_delete_all = """DELETE FROM %s"""

    # select all statement used for subclasses
    stmt_select_all = """SELECT id, uri, title, author, rating, duration,
                                view_count, small_thumbnail, large_thumbnail,
                                description, updated_date, published_date,
                                visit_time
                         FROM %s"""

    def __init__(self, name, parent):
        ModelFolder.__init__(self, name, parent)
        self.callback_notify = None
        self.client = Client()

    def do_load(self):
        self.search()

    def search(self, end_callback=None):
        del self.children[:]

        if not self.threaded_search:
            for c in self.do_search():
                self.children.append(c)
            return

        def refresh():
            return self.do_search()

        def refresh_finished(exception, retval):
            if not self.is_loading:
                log.info("model is not loading")
                return

            if exception is not None:
                if type(exception) is socket.gaierror or \
                        type(exception) is urllib2.URLError:
                    emsg = "Unable to connect to server.<br>" + \
                        "Check your connection and try again."
                else:
                    emsg = "An unknown error has occured."

                log.error(exception)

                if self.callback_notify:
                    self.callback_notify(CanolaError(emsg))

            if retval is None:
                self.inform_loaded()
                return

            for item in retval:
                self.children.append(item)

            if end_callback:
                end_callback()

            self.inform_loaded()

        self.is_loading = True
        ThreadedFunction(refresh_finished, refresh).start()

    def do_search(self):
        raise NotImplementedError("must be implemented by subclasses")

    def parse_entry_list(self, lst):
        return [self._create_model_from_entry(c) for c in lst]

    @classmethod
    def _create_table(cls):
        cls.db.execute(cls.stmt_create % cls.table)

    @classmethod
    def _select_model(cls, id):
        rows = cls.db.execute(cls.stmt_select % cls.table, (id,))
        if not rows:
            return None
        return cls._create_model_from_row(rows[0])

    @classmethod
    def _insert_model(cls, model):
        cls.db.execute(cls.stmt_insert % cls.table,
                        (model.id, model.remote_uri, model.title,
                         model.artist, model.rating, model.duration,
                         model.view_count, 0, model.visit_time,
                         model.description, model.small_thumbnail,
                         model.large_thumbnail, model.updated_date,
                         model.published_date))

    @classmethod
    def _delete_model(cls, id):
        cls.db.execute(cls.stmt_delete % cls.table, (id,))

    @classmethod
    def _delete_model_all(cls):
        cls.db.execute(cls.stmt_delete_all % cls.table)

    @classmethod
    def _select_model_all(cls, parent=None, order_by=None):
        lst = []
        query = cls.stmt_select_all % cls.table
        if order_by:
            query += " ORDER BY %s " % order_by

        for row in cls.db.execute(query):
            lst.append(cls._create_model_from_row(row, parent))
        return lst

    @classmethod
    def _create_model_from_row(cls, row, parent=None):
        model = VideoLocalModel(parent)

        model.id = to_utf8(row[0])
        model.title = to_utf8(row[2])
        model.artist = to_utf8(row[3])
        model.rating = row[4]
        model.duration = row[5]
        model.view_count = row[6]
        model.small_thumbnail = to_utf8(row[7])
        model.large_thumbnail = to_utf8(row[8])
        model.description = to_utf8(row[9])
        model.updated_date = row[10]
        model.published_date = row[11]
        model.visit_time = row[12]

        model.uri = get_video_path(model.id)
        model.thumb = get_thumb_path(model.id)
        model.remote_uri = None # resolve after
        model.rounded_rating = round_rating(model.rating)

        model.duration_formated = "%02d:%02d" % (model.duration / 60,
                                                 model.duration % 60)

        return model

    def _create_model_from_entry(self, data):
        log.debug("creating model for yid %s" % data.id)
        model = VideoLocalModel(self)

        model.id = data.id
        model.title = data.title
        if data.authors:
            model.artist = data.authors[0].name
        else:
            model.artist = ""
        model.rating = data.rating.avg
        model.duration = data.duration
        model.view_count = data.view_count
        model.description = data.description
        model.updated_date = data.updated
        model.published_date = data.published
        model.small_thumbnail = data.get_small_thumbnail()
        model.large_thumbnail = data.get_large_thumbnail()

        model.uri = get_video_path(model.id)
        model.thumb = get_thumb_path(model.id)
        model.remote_uri = None # resolve after
        model.rounded_rating = round_rating(model.rating)

        model.duration_formated = "%02d:%02d" % (model.duration / 60,
                                                 model.duration % 60)

        return model


##############################################################################
# YouTube Local Service Models
##############################################################################

class BookmarkModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Video/YouTube/Service/Bookmark"

    threaded_search = False
    table = "youtube_bookmarks"
    delete_possible = True
    empty_msg = "No videos bookmarked"

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self._create_table()

    def do_search(self):
        return self.select_model_all(self)

    @classmethod
    def select_model(cls, id):
        return cls._select_model(id)

    @classmethod
    def insert_model(cls, model):
        return cls._insert_model(model)

    @classmethod
    def delete_model(cls, id):
        return cls._delete_model(id)

    @classmethod
    def select_model_all(cls, parent=None):
        return cls._select_model_all(parent)


class HistoryModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Video/YouTube/Service/History"

    threaded_search = False
    table = "youtube_history"
    delete_possible = True
    empty_msg = "No videos in history"

    stmt_update = """UPDATE %s SET visit_time = ?
                     WHERE id = ?""" % table

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self._create_table()

    def do_search(self):
        return self.select_model_all(self)

    @classmethod
    def select_model(cls, id):
        return cls._select_model(id)

    @classmethod
    def insert_model(cls, model):
        return cls._insert_model(model)

    @classmethod
    def delete_model(cls, id):
        return cls._delete_model(id)

    @classmethod
    def delete_model_all(cls):
        return cls._delete_model_all()

    @classmethod
    def select_model_all(cls, parent=None):
        return cls._select_model_all(parent, "visit_time DESC")

    @classmethod
    def refresh_model(cls, model):
        try:
            cls.insert_model(model)
        except sqlite.IntegrityError:
            cls.db.execute(cls.stmt_update,
                           (model.visit_time, model.id))


class MyVideosModelFolder(ServiceModelFolder):
    terra_type = "Model/Folder/Task/Video/YouTube/Service/MyVideos"

    threaded_search = False
    table = "youtube_videos"
    delete_possible = True
    empty_msg = "No videos saved"

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self._create_table()

    def do_search(self):
        return self.select_model_all(self)

    @classmethod
    def select_model(cls, id):
        return cls._select_model(id)

    @classmethod
    def insert_model(cls, model):
        return cls._insert_model(model)

    @classmethod
    def delete_model(cls, id):
        video_path = get_video_path(id)
        if os.path.exists(video_path):
            os.unlink(video_path)
        if os.path.exists(video_path + VideoLocalModel.SAVED_STAMP):
            os.unlink(video_path + VideoLocalModel.SAVED_STAMP)

        return cls._delete_model(id)

    @classmethod
    def select_model_all(cls, parent=None):
        lst = []
        video_path = get_video_path()
        for c in cls._select_model_all(parent):
            if os.path.exists(os.path.join(video_path, c.id + ".flv")):
                lst.append(c)
        return lst


##############################################################################
# YouTube Remote Service Models
##############################################################################

class SearchModelFolder(ServiceModelFolder):
    """This model implements the Search option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/Search"

    def __init__(self, name, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self.query = None

    def do_search(self):
        lst = []
        if self.query:
            lst = self.parse_entry_list(self.client.search(self.query))
            self.query = None
        return lst


class FeaturedModelFolder(ServiceModelFolder):
    """This model implements the Featured option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/Featured"

    def do_search(self):
        return self.parse_entry_list(self.client.recently_featured())


class MostViewedModelFolder(ServiceModelFolder):
    """This model implements the Most Viewed option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/MostViewed"

    def do_search(self):
        return self.parse_entry_list(self.client.most_viewed())


class TopRatedModelFolder(ServiceModelFolder):
    """This model implements the Top Rated option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/TopRated"

    def do_search(self):
        return self.parse_entry_list(self.client.top_rated())


class MostRecentModelFolder(ServiceModelFolder):
    """This model implements the Most Recent option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/MostRecent"

    def do_search(self):
        return self.parse_entry_list(self.client.most_recent())


class CategoryModelFolder(ServiceModelFolder):
    """This model implements a category video list option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Service/Category"

    def __init__(self, name, category_id, parent):
        ServiceModelFolder.__init__(self, name, parent)
        self.category_id = category_id

    def do_search(self):
        lst = self.client.video_by_category(self.category_id)
        return self.parse_entry_list(lst)


class CategoriesModelFolder(ServiceModelFolder):
    """This model implements the Category List option."""
    terra_type = "Model/Folder/Task/Video/YouTube/Categories"

    def do_search(self):
        lst = []
        categories = self.client.category_list()
        for key, value in categories.iteritems():
            item = CategoryModelFolder(to_utf8(value),
                                       to_utf8(key), None)
            lst.append(item)
        return lst


##############################################################################
# Main YouTube Model
##############################################################################

class MainModelFolder(ModelFolder, Task):
    """Main YouTube Model.

    This is the main youtube model. It initializes all other models.
    """
    terra_type = "Model/Folder/Task/Video/YouTube"
    terra_task_type = "Task/Folder/Task/Video/YouTube"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "YouTube", parent)

    def do_load(self):
        """Initialize the base youtube models, like Search, Top rated,
        History, Most viewed and others.
        """
        SearchModelFolder("Search", self)
        FeaturedModelFolder("Featured", self)
        MostViewedModelFolder("Most viewed", self)
        BookmarkModelFolder("Bookmarks", self)
        MostRecentModelFolder("Most recent", self)
        TopRatedModelFolder("Top rated", self)
        CategoriesModelFolder("Categories", self)
        HistoryModelFolder("History", self)
        MyVideosModelFolder("Local videos", self)


##############################################################################
# YouTube Option Models
##############################################################################

class PlayerOptionsModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/YouTube"
    title = "Options"
    children_order = ["/Info", "/Save", "/Bookmark"]


class InfoOptionModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/YouTube/Info"
    title = "Info"


class SaveOptionModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/YouTube/Save"
    title = "Save to disk"

    def execute(self):
        try:
            model = self.screen_controller.model
            MyVideosModelFolder.insert_model(model)
            model.save_milestone(final=True)
            return True
        except:
            return False

    def has_record(self):
        model = self.screen_controller.model
        return bool(MyVideosModelFolder.select_model(model.id))

    def downloaded(self):
        model = self.screen_controller.model
        return (model.progress >= 1.0)


class BookmarkOptionModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Player/YouTube/Bookmark"
    title = "Add to bookmark"

    def execute(self):
        try:
            model = self.screen_controller.model
            BookmarkModelFolder.insert_model(model)
            return True
        except:
            return False

    def has_record(self):
        model = self.screen_controller.model
        return bool(BookmarkModelFolder.select_model(model.id))


class HistoryOptionsModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/History/YouTube"
    title = "Options"

    def __init__(self, parent, screen_controller=None):
        OptionsModelFolder.__init__(self, parent, screen_controller)
        ClearHistoryOptionModel(self)


class ClearHistoryOptionModel(ActionModelFolder):
    terra_type = "Model/Options/Action/History/YouTube/Clear"
    name = "Clear history"

    def execute(self):
        HistoryModelFolder.delete_model_all()
        self.parent.screen_controller.model.do_load()
        self.parent.screen_controller.view.loaded()
