#include <QDebug>

#include <QGeoServiceProvider>
#include <QGraphicsSceneMouseEvent>
#include <QGeoMapOverlay>
#include <QSettings>
#include <math.h>

#include "mapwidget.h"
#include "config.h"

#if(QTM_VERSION < 0x010100)
#error "QtMobility version 1.1 or higher required!"
#endif

void MapWidget::updateZoomButtons() {
  this->m_mapOverlay->enableZoomButtons(
	zoomLevel() < maximumZoomLevel(),
	zoomLevel() > minimumZoomLevel());
  
  update();
}

MapWidget::MapWidget(QGeoMappingManager *mgr) : 
  QGraphicsGeoMap(mgr), m_bubble(NULL), m_posIsValid(false), m_gpsMarker(NULL),
  m_manager(mgr) {
  
  // try to get default from qsettings, otherwise start at berlin's zoo
  QSettings settings;
  settings.beginGroup("Map");
  setCenter(QGeoCoordinate(
		   settings.value("Center/Latitude", 52.514).toFloat(),
		   settings.value("Center/Longitude", 13.3611).toFloat()));
  setZoomLevel(settings.value("ZoomLevel", 15).toInt());
  setMapType((QGraphicsGeoMap::MapType)settings.value("Type", 
			      QGraphicsGeoMap::StreetMap).toInt());
  m_gpsLocked = settings.value("GpsLocked", false).toBool();
  m_hiRez = settings.value("HiRez", false).toBool();
  m_precisionAlpha = settings.value("PrecisionAlpha", 32).toInt();
  settings.endGroup();
  if(m_hiRez) m_iconLoader = new IconLoader(Config::MAP_WIDGET_ICON_SIZE_HIREZ);
  else        m_iconLoader = new IconLoader(Config::MAP_WIDGET_ICON_SIZE);

  // create the control overlays
  this->m_mapOverlay = new MapOverlay(this);
  this->m_mapBanner = new MapBanner(this);
  this->m_mapSpinner = NULL;
  
  // and update it to reflect restored map state
  updateZoomButtons();
  if(m_gpsLocked)
    this->m_mapOverlay->changeGpsButton(true);
  
  // create a group for the cache icons
  m_cacheGroup = new QGeoMapGroupObject();
  m_cacheGroup->setZValue(0);
  addMapObject(m_cacheGroup);
  
  // create a gps marker and precision indicator, but hide them
  QPixmap *markerIcon = m_iconLoader->load("gps_marker");
  if(markerIcon) {
    m_gpsMarker = new QGeoMapPixmapObject(QGeoCoordinate(0,0), 
		  QPoint(-m_iconLoader->size()/2, -m_iconLoader->size()/2),
		  *markerIcon);

    m_gpsMarker->setZValue(3);
    m_gpsMarker->setVisible(false);
    addMapObject(m_gpsMarker);
  }
  
  m_gpsPrecision = new QGeoMapCircleObject();
  m_gpsPrecision->setPen(QPen((QColor(0, 0, 0, m_precisionAlpha))));
  m_gpsPrecision->setBrush(QBrush(QColor(0, 0, 0, m_precisionAlpha/2)));
  m_gpsPrecision->setZValue(2);
  addMapObject(m_gpsPrecision);

  // install timer to delay cache reload requests
  this->m_timer = new QTimer(this);
  this->m_timer->setSingleShot(true);
  QObject::connect(this->m_timer, SIGNAL(timeout()),
		   this, SLOT(reloadTimerExpired()));
  
  // connect to map signals to be able to handle viewport changes
  QObject::connect(this, SIGNAL(centerChanged(const QGeoCoordinate &)),
		   this, SLOT(centerChangedEvent(const QGeoCoordinate &)));
  QObject::connect(this, SIGNAL(zoomLevelChanged(qreal)),
		   this, SLOT(zoomLevelChangedEvent(qreal)));
}

MapWidget::~MapWidget() {
  qDebug() << __FUNCTION__;

  // save all kinds of map settings
  QSettings settings;
  settings.beginGroup("Map");
  settings.setValue("Center/Latitude", center().latitude());
  settings.setValue("Center/Longitude", center().longitude());
  settings.setValue("ZoomLevel", zoomLevel());
  settings.setValue("GpsLocked", m_gpsLocked);  
  settings.setValue("PrecisionAlpha", m_precisionAlpha);
  settings.endGroup();

  delete m_iconLoader;
  delete m_timer;
  delete m_mapOverlay;
  delete m_mapBanner;

  if(this->m_mapSpinner) 
    delete m_mapSpinner;
}

// ------------ slots to handle overlay button clicks --------------
void MapWidget::zoomIn() {
  if(zoomLevel() < maximumZoomLevel()) {
    setZoomLevel(zoomLevel()+1);
    updateZoomButtons();
  } 
}

/* never call this method directly as it is bypassed on */
/* e.g. maemo5. use emit showMessage() instead */
void MapWidget::addBanner(const QString &msg) {
  this->m_mapBanner->message(msg);
}

void MapWidget::zoomOut() {
  if(zoomLevel() > minimumZoomLevel()) {
    setZoomLevel(zoomLevel()-1);
    updateZoomButtons();
  }
}

void MapWidget::gpsFollow() {
  m_gpsLocked = true;
  this->m_mapOverlay->changeGpsButton(true);  
  update();
}

void MapWidget::toggleFullscreen() {
  emit fullscreen();
}

void MapWidget::resizeEvent(QGraphicsSceneResizeEvent *event) {
  QGraphicsGeoMap::resizeEvent(event);
  reload();
}

void MapWidget::showEvent(QShowEvent *) {
  reload();
}

void MapWidget::zoomLevelChangedEvent(qreal) {
  reload();
}

void MapWidget::centerChangedEvent(const QGeoCoordinate &newCenter) {

  if(!m_lastUpdateCenter.isValid()) {
    reload();
    return;
  }

  QPointF lastScreenPos(coordinateToScreenPosition(m_lastUpdateCenter));
  QPointF newScreenPos(coordinateToScreenPosition(newCenter));

  // if either horizontal or vertical moved more than 10% of screen, then
  // update
  if(fabs(lastScreenPos.x() - newScreenPos.x()) > size().width()/10 ||
     fabs(lastScreenPos.y() - newScreenPos.y()) > size().height()/10) {
    reload();
  }
}

void MapWidget::hideBubble() {
  // kill any existing bubble
  if(m_bubble) {
    removeMapObject(m_bubble);
    delete m_bubble;
    m_bubble = NULL;

    this->m_mapOverlay->enableForwardButton(false);
  }
}

void MapWidget::showBubble(const Cache &cache) {
  qDebug() << __FUNCTION__ << cache.name();

  hideBubble();
  m_bubble = new MapBubble(this, size().width() > size().height(), cache);
  addMapObject(m_bubble);

  this->m_mapOverlay->enableForwardButton(cache.url().isValid());
}

// currently connected to the "forward" button
void MapWidget::showDetails() {
  Q_ASSERT(m_bubble);
  emit detailClicked(m_bubble->cache().name());
}

void MapWidget::mousePressEvent(QGraphicsSceneMouseEvent* event) {
  qDebug() << __FUNCTION__;

  if(this->m_mapOverlay->mousePress(event->pos()))
    update();

  if(this->m_mapOverlay->isInside(event->pos()))
    m_downPos = QPointF(0,0);
  else
    m_downPos = event->pos();

  m_dragging = false;
}

void MapWidget::mouseReleaseEvent(QGraphicsSceneMouseEvent* event) {
  if(this->m_mapOverlay->mouseRelease(event->pos()))
    update();

  if(m_dragging && m_gpsLocked) {
    m_gpsLocked = false;
    this->m_mapOverlay->changeGpsButton(false);
    update();
  }

  // this doesn't work on n900 as the n900 port of qt mobility fails
  // at detecting clicks onto bitmaps
  if(m_downPos.isNull() || m_dragging)
    return;

  // check if we clicked a cache
  bool clickedBubble = false;
  QString clickedCache;
  Waypoint clickedWaypoint;

  QList<QGeoMapObject*> objects = mapObjectsAtScreenPosition(m_downPos);
  if (objects.length() > 0) {
    for (int i = objects.length()-1; i >= 0; i--) {
      if ((objects[i]->objectName() == "cache") && clickedCache.isEmpty())
	clickedCache = objects[i]->property("name").toString();
      if ((objects[i]->objectName() == "waypoint") && clickedCache.isEmpty()) {
	clickedCache = objects[i]->property("cache").toString();
	clickedWaypoint = 
	  static_cast<MapWaypointObject *>(objects[i])->waypoint();
      }

      if (objects[i]->objectName() == "bubble") 
	clickedBubble = true;
    }
  }

  if(!clickedBubble) {
    hideBubble();

    if(!clickedCache.isEmpty()) {
      m_selectedCache = clickedCache;
      
      /* update caches to show waypoints of selected cache if present */
      emit mapChanged();
 
      if(!clickedWaypoint.coordinate().isValid()) 
	emit cacheClicked(clickedCache);
      else {
	qDebug() << __FUNCTION__ << "User clicked waypoint" <<
	  clickedWaypoint.name() << "of cache" << clickedCache;

      }
    }
  }
}

void MapWidget::mouseMoveEvent(QGraphicsSceneMouseEvent* event) {
  if(this->m_mapOverlay->mouseMove(event->pos()))
    update();

  if(m_downPos.isNull())
    return;

  if(!m_dragging) {
    if((m_downPos - event->pos()).manhattanLength() < DRAG_FUZZ) 
      return;

    // dragged more than DRAG_FUZZ pixels: start actual drag
    m_dragging = true;
    QPointF dp(m_downPos - event->pos());
    pan(dp.x(), dp.y());
    return;
  }

  QPointF d(event->lastPos() - event->pos());
  pan(d.x(), d.y());
}

MapWidget::MapCacheObject::MapCacheObject(IconLoader *loader, const Cache &cache, bool highlight) :
  QGeoMapPixmapObject(cache.coordinate(), 
		      QPoint(-loader->size()/2, -loader->size()/2),
		      *loader->load(cache)) {

  if(highlight) {
    QPixmap *hlPix = loader->loadHighlight(cache);
    setOffset(QPoint(-hlPix->width()/2, -hlPix->height()/2));
    setPixmap(*hlPix);
  }

  setZValue(0);  // below gps marker and bubble
  setObjectName("cache");
  setProperty("name", cache.name());
};

// the pixmap object seems to actually check for clicks onto the bitmap. thus
// we only check for the click being withing the entire pixmap size. furthermore
// the pixmap precise stuff doesn't seem to work at all on the n900/maemo
bool MapWidget::MapCacheObject::contains(const QGeoCoordinate &coo) const {
  return boundingBox().contains(coo);
}

MapWidget::MapWaypointObject::MapWaypointObject(IconLoader *loader, 
		const QString &cacheName, const Waypoint &waypoint) :
  QGeoMapPixmapObject(waypoint.coordinate(),
		      QPoint(-loader->size()/3, -loader->size()/3),
		      *loader->load(waypoint.iconFile(), 2*loader->size()/3)) {
  
  setZValue(1);  // below gps marker and bubble, but above caches
  setObjectName("waypoint");
  setProperty("cache", cacheName);
  m_wpt = waypoint;
};

Waypoint MapWidget::MapWaypointObject::waypoint() const {
  return m_wpt;
}

void MapWidget::updateCaches(const CacheList &cacheList) {
  static bool tooOld = false;

  // save cache if existing bubble
  Cache bubbleCache;
  if(m_bubble) bubbleCache = m_bubble->cache();
  hideBubble();

  // remove all existing caches
  m_cacheGroup->clearChildObjects();

  // determine cachelist age
  int weekAge = cacheList.date().daysTo(QDate::currentDate())/7;
  if(!cacheList.date().isValid()) weekAge = 0;

  if(!tooOld && weekAge > 0) {
    emit showMessage(tr("Displayed cache data is %1 weeks old!").arg(weekAge));
    tooOld = true;
  } else if(tooOld && weekAge == 0)
    tooOld = false;

  // draw all new caches
  QList<Cache>::const_iterator i;
  for( i = cacheList.begin(); i != cacheList.end(); ++i ) {
    bool isSelected = (i->name() == m_selectedCache);

    // highlight if cache has waypoints
    m_cacheGroup->addChildObject(new MapCacheObject(m_iconLoader, *i, 
		    isSelected && i->waypoints().size() > 0));
				 
    // also draw waypoints of selected cache
    if(isSelected) 
      m_selectedWaypoints = i->waypoints();
  }

  // draw all waypoints from list
  foreach(Waypoint wpt, m_selectedWaypoints)
    m_cacheGroup->addChildObject(new MapWaypointObject(m_iconLoader, m_selectedCache, wpt));


  // restore previous bubble
  if(bubbleCache.coordinate().isValid()) {
    // The ugly part: gc.com is randomizing coordinates and
    // the new coordinates used to draw the icon are likely
    // different from the ones saved from the previous bubble
    // position. So we search for the bubble cache within the
    // new cache list and use it's coordinates instead
    
    bool coordinateUpdated = false;
    for( i = cacheList.begin(); i != cacheList.end(); ++i ) {
      if(i->name() == bubbleCache.name()) {
	bubbleCache.setCoordinate(i->coordinate());
	coordinateUpdated = true;
      }
    }

    // only redraw bubble if the cache is still part of the
    // new cache list and if it's cache is still on-screen
    if(coordinateUpdated && 
       viewport().contains(bubbleCache.coordinate()))
      showBubble(bubbleCache);
  }
}

void MapWidget::reloadTimerExpired() {
  qDebug() << __FUNCTION__;

  // check if viewport actually changed
  if(m_currentViewport != viewport()) {
    m_lastUpdateCenter = center();

    emit mapChanged();
    m_currentViewport = viewport();
  }
}

void MapWidget::reload() {
  // start the timer if it's not already running, re-start from zero otherwise
  // using less than 1000ms here makes sure that requests are being sent for
  // every updated gps position (reported every 1000ms)
  if(!this->m_timer->isActive())
    this->m_timer->start(900);
  else
    this->m_timer->setInterval(900);
}

void MapWidget::positionUpdated(const QGeoPositionInfo &pos) {
  // change state of gps button to reflect gps availability
  if(m_posIsValid != pos.coordinate().isValid()) {
    this->m_mapOverlay->enableGpsButton(pos.coordinate().isValid());
    update();
    m_posIsValid = pos.coordinate().isValid();
  }

  // re-center map if gpsLock is active
  if(pos.coordinate().isValid() && m_gpsLocked && !m_dragging) 
    setCenter(pos.coordinate());

  // draw gps marker
  if(m_gpsMarker) {
    if(pos.coordinate().isValid()) {
      m_gpsMarker->setVisible(true);
      m_gpsMarker->setCoordinate(pos.coordinate());

      // horizontal accuracy, will return -1 if unset
      // and circle will be hidden if radius < 0
      m_gpsPrecision->setRadius(pos.attribute(QGeoPositionInfo::HorizontalAccuracy));
      m_gpsPrecision->setCenter(pos.coordinate());
    } else {
      m_gpsMarker->setVisible(false);
      m_gpsPrecision->setRadius(-1);
    }
  }
}

QString MapWidget::managerName() {
  return m_manager->managerName();
}

void MapWidget::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget * widget) {
  // first call parents paint function
  QGraphicsGeoMap::paint(painter, option, widget);

  // then draw our overlay on top
  this->m_mapOverlay->paint(painter);
  this->m_mapBanner->paint(painter);

  if(this->m_mapSpinner)
    this->m_mapSpinner->paint(painter);
}

void MapWidget::setBusy(bool on) {
  if(on) {
    Q_ASSERT(!m_mapSpinner);
    m_mapSpinner = new MapSpinner(this);
  } else {
    Q_ASSERT(m_mapSpinner);
    delete m_mapSpinner;
    m_mapSpinner = NULL;
  }
}

bool MapWidget::hiRez() const {
  return m_hiRez;
}

void MapWidget::setHiRez(bool hiRez) {
  if(m_hiRez != hiRez) {
    m_hiRez = hiRez;

    delete m_iconLoader;
    if(m_hiRez) 
      m_iconLoader = new IconLoader(Config::MAP_WIDGET_ICON_SIZE_HIREZ);
    else
      m_iconLoader = new IconLoader(Config::MAP_WIDGET_ICON_SIZE);

    // this will also cause the overlay to redraw
    emit mapChanged();
  }
}
