#include "iconloader.h"
#include "filefinder.h"

#include <QDebug>
#include <QPainter>
#include <QLabel>
#include <QImage>

bool IconLoader::Pixmap::load(QSvgRenderer *renderer, const QString &name) {
  QString filename = FileFinder::find("icons", name);
  if(filename.isNull()) return false;

  renderer->load(filename);
  return true;
}

// create empty pixmap for graphic effects
IconLoader::Pixmap::Pixmap(const QStringList &name, int size, State state) :
  m_pixmap(NULL) {

  this->m_pixmap = new QPixmap(size, size);
  // fill with transparency and saved icon name
  this->m_pixmap->fill(Qt::transparent);
  this->m_name = name.join(",");
  this->m_state = state;  
}

// create a Pixmap from svg file
IconLoader::Pixmap::Pixmap(QSvgRenderer *renderer, 
			   const QStringList &name, int size) :
  m_pixmap(NULL) {

  if(load(renderer, name[0] + ".svg")) {
    // if no explicit size is given, then use default size of first image
    if(size > 0) 
      this->m_pixmap = new QPixmap(renderer->defaultSize().width()*size/
				   renderer->defaultSize().height(), size);
    else 
      this->m_pixmap = new QPixmap(renderer->defaultSize());
  } else 
    this->m_pixmap = new QPixmap(size, size);
 
  // fill with transparency and saved icon name
  this->m_pixmap->fill(Qt::transparent);
  this->m_name = name.join(",");
  this->m_state = IconLoader::Pixmap::Normal;
  
  QPainter painter(this->m_pixmap);
  painter.setRenderHint(QPainter::Antialiasing, true);

  // draw all given images on top of each other
  QStringList::const_iterator i;
  for(i=name.begin();i!=name.end();i++) {
    if(load(renderer, *i + ".svg"))
      renderer->render(&painter);
    else {
      // unable to load image, draw some kind of "broken" icon
      painter.setPen(QPen(QBrush(Qt::red), size/5) );
      painter.drawEllipse(QRect(QPoint(size/8,size/8), this->m_pixmap->size() - 
				QSize(size/4, size/4)));
    }
  }
}

IconLoader::Pixmap::~Pixmap() {
  if(this->m_pixmap)
    delete this->m_pixmap;
}

QPixmap* IconLoader::Pixmap::pixmap() {
  return m_pixmap;
}

bool IconLoader::Pixmap::is(const QString &name, State state) {
  return m_name == name && m_state == state;
}

IconLoader::IconLoader(int size) {
  m_renderer = new QSvgRenderer();
  this->m_size = size;
}

IconLoader::~IconLoader() {
  qDeleteAll(pixmapList);
  delete m_renderer;
}

QPixmap *IconLoader::load(const QStringList &name, int size) {
  QString joinedName = name.join(",");

  // check if there's already a matching pixmap in the list
  QList<Pixmap*>::const_iterator i;
  for(i = pixmapList.constBegin(); i != pixmapList.constEnd(); ++i) 
    if( (*i)->is(joinedName, IconLoader::Pixmap::Normal) )
      return (*i)->pixmap();
  
  // nothing matching in pixmap list, create a new entry
  Pixmap *pix = new Pixmap(m_renderer, name, size?size:m_size); 
  pixmapList.append(pix);
  
  return pix->pixmap();
}

// todo: cache resulting pixmap!
QPixmap *IconLoader::loadHighlight(const QStringList &name, int size) {
  QString joinedName = name.join(",");

  // check if there's already a matching pixmap in the list
  QList<Pixmap*>::const_iterator i;
  for(i = pixmapList.constBegin(); i != pixmapList.constEnd(); ++i) 
    if( (*i)->is(joinedName, IconLoader::Pixmap::Highlight) )
      return (*i)->pixmap();

  // load non-highlighted icon
  const int BORDER = 8;
  QPixmap *pix = load(name, size);

  // create temporary pixmap with extra space for border ...
  QPixmap tmpPix(pix->width() + 2*BORDER, pix->height() + 2*BORDER);

  // ... and fill with transparency ...
  tmpPix.fill(Qt::transparent);

  // ... and draw icon into the center
  QPainter tmpPainter(&tmpPix);
  tmpPainter.drawPixmap(QPointF(BORDER, BORDER), *pix);

  // convert source to qimage to be able to access individual pixels
  QImage srcImage = tmpPix.toImage();

  // and create a destination image to paint on
  QImage image(srcImage.size(), QImage::Format_ARGB32);
  image.fill(Qt::transparent);
  
  // create hihglight shadow in "image"
  int w = image.width(), h = image.height();
  for(int y = 0; y < h; y++) {
    for( int x = 0; x < w; x++) {
      QRgb pix = srcImage.pixel(x, y);

      // target pixel is at least partially transparent and will 
      // let the highlight shine through
      if(qAlpha(pix) < 0xff) {
	int sum = 0;

	int sx = x-BORDER;   if(sx<0)   sx = 0;
	int ex = x+BORDER+1; if(ex>w-1) ex = w-1;
	int sy = y-BORDER;   if(sy<0)   sy = 0;
	int ey = y+BORDER+1; if(ey>w-1) ey = h-1;

	for(int ay=sy;ay<ey;ay++)
	  for(int ax=sx;ax<ex;ax++)
	    sum += qAlpha(srcImage.pixel(ax, ay));

	sum /= ((2*BORDER+1)*(2*BORDER+1))/4;

	if(sum > 255) sum = 255;
	QColor color(0xff,0xc0,0,sum);
	image.setPixel(x, y, color.rgba());
      }
    }
  }

  // finally combine everything into a new pixmap
  Pixmap *hlPix = new Pixmap(name, tmpPix.width(), 
			     IconLoader::Pixmap::Highlight);

  QPainter hlPainter(hlPix->pixmap());
  // first the highlight ...
  hlPainter.drawImage(QPointF(0,0), image);
  // ... then the icon itself on top
  hlPainter.drawPixmap(QPointF(BORDER,BORDER), *pix);

  // finally add pixmap to cache
  pixmapList.append(hlPix);

  return hlPix->pixmap();
}

QPixmap *IconLoader::loadHighlight(const Cache &cache, int size) {
  QStringList names;
  names << cache.typeIconFile();
  QString overlay = cache.overlayIconFile();
  if(!overlay.isNull())
    names << overlay;
  
  return loadHighlight(names, size);
}

QPixmap *IconLoader::load(const QString &name, int size) {
  return load(QStringList(name), size);
}

QPixmap *IconLoader::loadHighlight(const QString &name, int size) {
  return loadHighlight(QStringList(name), size);
}

QPixmap *IconLoader::load(const Cache &cache, int size) {
  QStringList names;
  names << cache.typeIconFile();
  QString overlay = cache.overlayIconFile();
  if(!overlay.isNull())
    names << overlay;
  
  return load(names, size);
}

QPixmap *IconLoader::load(const Container &container, int size) {
  return load(container.iconFile(), size);
}

int IconLoader::size() const {
  return m_size;
}

QWidget *IconLoader::newWidget(const QString &name, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(name, size));
  return widget;
}

QWidget *IconLoader::newWidget(const QStringList &names, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(names, size));
  return widget;
}

QWidget *IconLoader::newWidget(const Cache &cache, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(cache, size));
  return widget;
}
