#include <QDebug>

#include <QTextDocument>
#include "gchtmlparser.h"
#include "QSgml.h"

GcHtmlParser::GcHtmlParser() {
  qDebug() << __PRETTY_FUNCTION__;
}

GcHtmlParser::~GcHtmlParser() {
  qDebug() << __PRETTY_FUNCTION__;
}

void GcHtmlParser::dump(const QSgmlTag *tag) {
  qDebug() << __FUNCTION__ << 
    tag->Type << tag->Name << tag->Value;
  
  foreach(QString key, tag->Attributes.keys()) 
    qDebug() << key << tag->Attributes.value(key);

}

void GcHtmlParser::dump(const QSgmlTag::QSgmlTaglist &list, int level) {
  if(!list.size()) return;

  for(int i=0;i<list.size();i++) {
    qDebug() << __FUNCTION__ << level << 
      list[i]->Type << list[i]->Name << list[i]->Value;

    foreach(QString key, list[i]->Attributes.keys()) 
      qDebug() << key << list[i]->Attributes.value(key);

    if(level >= 0)
      dump(list[i]->Children, level+1);
  }
}

static bool isDiv(QSgmlTag *tag, const QString &str) {
  return(!tag->Name.compare("div", Qt::CaseInsensitive) && 
	 tag->Type == QSgmlTag::eStartTag &&
	 tag->hasAttribute("class") && 
	 !tag->getArgValue("class").compare(str, Qt::CaseInsensitive));
}

static bool isTag(QSgmlTag *tag, const QString &str) {
  return(!tag->Name.compare(str, Qt::CaseInsensitive) && 
	 tag->Type == QSgmlTag::eStartTag);
}

/* this is a really ugly way to decode this. But it seems to work */
static QString decodeEntities(const QString &html) {
  QTextDocument textDocument;
  textDocument.setHtml(html);
  return textDocument.toPlainText();
}

// return only the pure text below the current tag
QString GcHtmlParser::getText(const QSgmlTag::QSgmlTaglist &list) {
  QString str;
  foreach(QSgmlTag *tag, list) {
    if(tag->Type == QSgmlTag::eCdata) str += tag->Value;
    else                              str += getText(tag->Children);
  }

  return decodeEntities(str);
}

// return the entire html code below current tag
QString GcHtmlParser::getHtml(const QSgmlTag::QSgmlTaglist &list) {
  QString str;
  foreach(QSgmlTag *tag, list) {
    if(tag->Type == QSgmlTag::eCdata) str += tag->Value;
    else {
      QString subStr =  getHtml(tag->Children);
      str += "<" + tag->Name;

      foreach(QString key, tag->Attributes.keys()) 
	str += " " + key + "=\"" + tag->Attributes.value(key) + "\"";

      if(!subStr.isEmpty())
	str += ">" + getHtml(tag->Children) + "</" + tag->Name + ">";
      else
	str += "/>";
    }
  }
  return str;
}

QString GcHtmlParser::searchForItemHeaderH2(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    if(isTag(tag, "h2"))
      return getText(tag->Children).simplified();
    else {
      QString str = searchForItemHeaderH2(tag->Children);
      if(!str.isEmpty()) return str;
    }
  }
  return NULL;
}

QString GcHtmlParser::searchForItemHeader(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString header = isDiv(tag, "item-header")? 
      searchForItemHeaderH2(tag->Children):
      searchForItemHeader(tag->Children);

    if(!header.isEmpty()) return header;
  }
  return NULL;
}

QString GcHtmlParser::searchForItemContentText(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString content = isDiv(tag, "item-content")?
      getHtml(tag->Children).trimmed():
      searchForItemContentText(tag->Children);
    
    if(!content.isEmpty()) return content;
  }
  return NULL;
}

QString GcHtmlParser::searchForHintEncrypted(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString hint = isDiv(tag, "hint-encrypted")?
      getHtml(tag->Children).trimmed():
      searchForHintEncrypted(tag->Children);
    
    if(!hint.isEmpty()) return hint;
  }
  return NULL;
}

Log::Type::Id GcHtmlParser::parseLogType(const QString &str) {
  int i;
  
  struct { 
    Log::Type::Id type;
    QString str;
  } tags[] = {
    { Log::Type::Found,              "Found it" },
    { Log::Type::NotFound,           "Didn't find it" },
    { Log::Type::Maintenance,        "Owner Maintenance" },
    { Log::Type::WriteNote,          "Write Note" },
    { Log::Type::ReviewerNote,       "Post Reviewer Note" },
    { Log::Type::EnableListing,      "Enable Listing" },
    { Log::Type::PublishListing,     "Publish Listing" },
    { Log::Type::WillAttend,         "Will Attend" },
    { Log::Type::Attended,           "Attended" },
    { Log::Type::Photo,              "Webcam Photo taken" },
    { Log::Type::TempDisable,        "Temporarily Disable Listing" },
    { Log::Type::NeedsMaintenance,   "Needs Maintenance" },
    { Log::Type::UpdatedCoordinates, "Update Coordinates" },
    { Log::Type::Unarchive,          "Unarchive" },
    { Log::Type::NeedsArchived,      "Needs Archived" },
    { Log::Type::Archive,            "Archive" },
    { Log::Type::Unknown,            "<unknown>" }
  };

  for(i=0;(tags[i].type != Log::Type::Unknown) && 
	(tags[i].str.compare(str, Qt::CaseInsensitive));i++);
      
  return tags[i].type;
}

int GcHtmlParser::parseMonth(const QString &str) {
  int i;

  QString names[] = {
    "January", "February", "March", "April", "May", "June",
    "July", "August", "September", "October", "November", "December"
  };

  for(i=0;i<12 && names[i].compare(str, Qt::CaseInsensitive);i++);

  qDebug() << __FUNCTION__ << "parsed" << str << "to" << i+1;

  return i+1;
}


void GcHtmlParser::parseLogDt(Log &log, const QSgmlTag::QSgmlTaglist &list) {
  // dump(list);

  /************** extract type ***************/
  foreach(QSgmlTag *tag, list) {
    if(!tag->Name.compare("img", Qt::CaseInsensitive) && 
       (tag->Type == QSgmlTag::eStartTag || 
	tag->Type == QSgmlTag::eStandalone))  {

      // found image, get contents of "alt" attribute
      foreach(QString key, tag->Attributes.keys()) {
	if(!key.compare("alt", Qt::CaseInsensitive)) 
	  log.setType(parseLogType(tag->Attributes.value(key)));
      }
    }
  }

  /************** extract date ***************/

  QString text = getText(list).simplified();
  QStringList dateParts = text.split(',')[1].trimmed().split(' ');

  log.setDate(QDate(dateParts[2].toInt(), parseMonth(dateParts[1]), 
		    dateParts[0].toInt()));

  /************** extract finder name ***************/

  QStringList words = text.split(' ');

  // find the word "by"
  int by = words.indexOf("by");
  if(by >= 0 && by+1 < words.size())
    log.setFinder(words[by+1]);
}

void GcHtmlParser::parseLogDD(Log &log, const QSgmlTag::QSgmlTaglist &list) {
  Description description;
  description.set(true, getHtml(list).trimmed());

  if(description.isSet())
    log.setDescription(description);
}

void GcHtmlParser::searchForLogs(const QSgmlTag::QSgmlTaglist &list, Cache &cache) {
  foreach(QSgmlTag *tag, list) {
    // all logs are within on <dl></dl>
    if(!tag->Name.compare("dl", Qt::CaseInsensitive) && 
       tag->Type == QSgmlTag::eStartTag) {
      bool expectingDT = true;;
      Log log;

      // directly below we have alternating <dt> and <dd> tags
      foreach(QSgmlTag *subtag, tag->Children) {
	if(subtag->Type == QSgmlTag::eStartTag) {
	  if(!subtag->Name.compare("dt", Qt::CaseInsensitive)) {
	    if(expectingDT) {
	      log = Log();
	      parseLogDt(log, subtag->Children);
	      expectingDT = false;   // expecting DD
	    }
	  } else if(!subtag->Name.compare("dd", Qt::CaseInsensitive)) {
	    if(!expectingDT) {  // expecting DD
	      parseLogDD(log, subtag->Children);
	      cache.appendLog(log);
	      expectingDT = true;
	    }
	  }
	}
      }
    } else
      searchForLogs(tag->Children, cache);
  }
}

Attribute::Id GcHtmlParser::parseAttributeType(const QString &str) {
  int i;
  
  struct { 
    Attribute::Id type;
    QString str;
  } tags[] = {
    { Attribute::Dogs,              "dogs" },
    { Attribute::Fee,               "fee" }, 
    { Attribute::Rappelling,        "rappelling" }, 
    { Attribute::Boat,              "boat" },
    { Attribute::Scuba,             "scuba" }, 
    { Attribute::Kids,              "kids" }, 
    { Attribute::OneHour,           "onehour" }, 
    { Attribute::Scenic,            "scenic" }, 
    { Attribute::Hiking,            "hiking" },
    { Attribute::Climbing,          "climbing" }, 
    { Attribute::Wading,            "wading" }, 
    { Attribute::Swimming,          "swimming" }, 
    { Attribute::Available,         "available" }, 
    { Attribute::Night,             "night" },
    { Attribute::Winter,            "winter" }, 
    { Attribute::PoisonOak,         "poisonoak" }, 
    { Attribute::Snakes,            "snakes" }, 
    { Attribute::Ticks,             "ticks" },
    { Attribute::Mine,              "mine" }, 
    { Attribute::Cliff,             "cliff" }, 
    { Attribute::Hunting,           "hunting" }, 
    { Attribute::Danger,            "danger" }, 
    { Attribute::WheelChair,        "wheelchair" },
    { Attribute::Parking,           "parking" }, 
    { Attribute::Public,            "public" }, 
    { Attribute::Water,             "water" }, 
    { Attribute::Restrooms,         "restrooms" }, 
    { Attribute::Phone,             "phone" },
    { Attribute::Picnic,            "picnic" }, 
    { Attribute::Camping,           "camping" }, 
    { Attribute::Bicycles,          "bicycles" }, 
    { Attribute::Motorcycles,       "motorcycles" }, 
    { Attribute::Quads,             "quads" },
    { Attribute::Jeeps,             "jeeps" }, 
    { Attribute::Snowmobiles,       "snowmobiles" }, 
    { Attribute::Horses,            "horses" }, 
    { Attribute::Campfires,         "campfires" }, 
    { Attribute::Thorn,             "thorn" },
    { Attribute::Stealth,           "stealth" }, 
    { Attribute::Stroller,          "stroller" }, 
    { Attribute::FirstAid,          "firstaid" }, 
    { Attribute::Cow,               "cow" }, 
    { Attribute::Flashlight,        "flashlight" },
    { Attribute::LandF,             "landf" }, 
    { Attribute::RV,                "rv" }, 
    { Attribute::FieldPuzzle,       "field_puzzle" }, 
    { Attribute::UV,                "uv" }, 
    { Attribute::Snowshoes,         "snowshoes" },
    { Attribute::Skiis,             "skiis" }, 
    { Attribute::STool,             "s-tool" },
    { Attribute::NightCache,        "nightcache" },
    { Attribute::ParkNGrab,         "parkngrab" },
    { Attribute::AbandonedBuilding, "abandonedbuilding" },
    { Attribute::HikeShort,         "hike_short" },
    { Attribute::HikeMed,           "hike_med" },
    { Attribute::HikeLong,          "hike_long" },
    { Attribute::Fuel,              "fuel" },
    { Attribute::Food,              "food" },
    { Attribute::Unknown,           "" }
  };
  
  for(i=0;(tags[i].type != Attribute::Unknown) && 
	(tags[i].str.compare(str, Qt::CaseInsensitive));i++);
      
  return tags[i].type;
}

void GcHtmlParser::searchForAttributes(const QSgmlTag::QSgmlTaglist &list, Cache &cache) {
  foreach(QSgmlTag *tag, list) {
    // all logs are within image tags
    if(!tag->Name.compare("img", Qt::CaseInsensitive) && 
       (tag->Type == QSgmlTag::eStartTag || 
	tag->Type == QSgmlTag::eStandalone))  {
      QString attName = tag->Attributes.value("src").simplified();

      //      qDebug() << __FUNCTION__ << attName << tag->Attributes.value("alt").simplified();

      // extract base filename excluding path and suffix
      attName = attName.split('/').last().split('.').first();

      // if we find no "-no" at the end it's a "yes"
      bool yes = true;
      if(attName.endsWith("-no", Qt::CaseInsensitive)) {
	attName.chop(3);
	yes = false;
      }

      if(attName.endsWith("-yes", Qt::CaseInsensitive))
	attName.chop(4);
      
      if(!attName.isEmpty() && 
	 attName.compare("attribute-blank", Qt::CaseInsensitive)) {

	// parse attribute
	Attribute::Id attId = parseAttributeType(attName);
	if(attId != Attribute::Unknown) {
	  Attribute attribute(attId, yes);
	  cache.appendAttribute(attribute);
	}
      }
    } else
      searchForAttributes(tag->Children, cache);
  }
}

void GcHtmlParser::parseWaypointType(Waypoint::Type &type, const QSgmlTag::QSgmlTaglist &list) {
  // dump(list);

  /************** extract type ***************/
  foreach(QSgmlTag *tag, list) {
    if(!tag->Name.compare("img", Qt::CaseInsensitive) && 
       (tag->Type == QSgmlTag::eStartTag || 
	tag->Type == QSgmlTag::eStandalone))  {

      // found image, get contents of "alt" attribute
      foreach(QString key, tag->Attributes.keys()) {
	if(!key.compare("alt", Qt::CaseInsensitive)) {
	  int i;

	  struct { 
	    Waypoint::Type type;
	    QString str;
	  } tags[] = {
	    { Waypoint::Multistage,   "Stages of a Multicache" },
	    { Waypoint::Parking,      "Parking Area" },
	    { Waypoint::Final,        "Final Location" },
	    { Waypoint::Question,     "Question to Answer" },
	    { Waypoint::Trailhead,    "Trailhead" },
	    { Waypoint::Refpoint,     "Reference Point" },
	    { Waypoint::Unknown,      "<unknown>" }
	  };
	  
	  QString typeStr = tag->Attributes.value(key);

	  for(i=0;(tags[i].type != Waypoint::Unknown) && 
		!(tags[i].str.split(",").contains(typeStr, Qt::CaseInsensitive));i++);

	  type = tags[i].type;
	}
      }
    }
  }
}

bool GcHtmlParser::isCheckbox(const QSgmlTag::QSgmlTaglist &list) {
  for(int i=0;i<list.size();i++) {
    // "span" element?
    if(!list[i]->Name.compare("span", Qt::CaseInsensitive)) {
      // with class "Checkbox"?
      if(!list[i]->Attributes.value("class").compare("checkbox", Qt::CaseInsensitive))
	return true;
    }
  }
  return false;
}

void GcHtmlParser::parseWaypointTds(Waypoint &wpt, QString &note, 
		    const QString &id, const QSgmlTag::QSgmlTaglist &list) {

  //  dump(list);

  int col = 0;
  bool isNoteRow = false;
  foreach(QSgmlTag *tag, list) {
    if(!tag->Name.compare("td", Qt::CaseInsensitive) && 
       (tag->Type == QSgmlTag::eStartTag || 
	tag->Type == QSgmlTag::eStandalone))  {

      switch(col) {
      case 0:
	// on owned caches, there's an additional column with a checkbox
	// which we just skip
	if(isCheckbox(tag->Children)) col--;
	break;

      case 1:
	// 1 is img with type in alt tag or the "Note:" part of
	// an extra comment row
	if(!getText(tag->Children).simplified().compare("Note:", Qt::CaseInsensitive))
	  isNoteRow = true;
	else {
	  Waypoint::Type type;
	  parseWaypointType(type, tag->Children);
	  wpt.setType(type);
	}
	break;

      case 2:
	// prefix or comment
	if(!isNoteRow) {
	  QString wptId = getText(tag->Children).simplified() + id.right(id.size()-2);
	  wpt.setName(wptId);
	} else
	  note = getText(tag->Children).simplified();
	break;

      case 3:
	break;

      case 4:
	if(!isNoteRow) {
	  QString desc = getText(tag->Children).simplified();
	  // try to cut off trailing "(type)"
	  int index = desc.lastIndexOf('('); 
	  if(index > 0) 
	    desc.remove(index, desc.size());
	  
	  wpt.setDescription(desc.trimmed());
	}
	break;

      case 5: 
	if(!isNoteRow) {
	  QString coordinates = getText(tag->Children).simplified();
	  if(coordinates.size() > 3) {
	    // remove degree signs and minute ticks
	    coordinates.remove('\260'); coordinates.remove('\'');
	    
	    QStringList parts = coordinates.split(' ', QString::SkipEmptyParts);

	    if(parts.size() > 5) {
	      qreal lat = ((!parts[0].compare("s", Qt::CaseInsensitive))?-1:1) *
		parts[1].toFloat() + parts[2].toFloat()/60.0;
	      qreal lon = ((!parts[3].compare("w", Qt::CaseInsensitive))?-1:1) *
		parts[4].toFloat() + parts[5].toFloat()/60.0;
	    
	      wpt.setCoordinate(QGeoCoordinate(lat, lon));
	    }
	  }
	} 
	break;
      }

      col++;
    }
  }
}

void GcHtmlParser::searchForWaypoints(const QSgmlTag::QSgmlTaglist &list, Cache &cache) {
  // waypoints are placed within a table
  foreach(QSgmlTag *tag, list) {
    // each row my be a waypoint (may also be a header which doesn't)
    if(!tag->Name.compare("tr", Qt::CaseInsensitive) && 
       (tag->Type == QSgmlTag::eStartTag || 
	tag->Type == QSgmlTag::eStandalone))  {
      Waypoint wpt;
      QString note;
      parseWaypointTds(wpt, note, cache.name(), tag->Children);

      if(!note.isEmpty()) {
	// append note to last waypoint
	wpt = cache.waypoints().last();
	wpt.setComment(note);
	cache.updateWaypoint(wpt);
      } else 
	if(!wpt.name().isEmpty())
	  cache.appendWaypoint(wpt);
      
    } else 
      searchForWaypoints(tag->Children, cache);
  }
}

void GcHtmlParser::searchForItem(const QSgmlTag::QSgmlTaglist &list, 
				 Cache &cache) {
  foreach(QSgmlTag *tag, list) {
    if(isDiv(tag, "item")) {
      QString header = searchForItemHeader(tag->Children);

      if(!header.isEmpty()) {
	qDebug() << __FUNCTION__ << "found" << header;
	
	/* now search for matching "item-content" */
	if(!header.compare("Short Description", Qt::CaseInsensitive)) {
	  Description description;
	  description.set(true, searchForItemContentText(tag->Children));
	  if(description.isSet()) cache.setShortDescription(description);

	} else if(!header.compare("Long Description", Qt::CaseInsensitive)) {
	  Description description;
	  description.set(true, searchForItemContentText(tag->Children));
	  if(description.isSet()) cache.setLongDescription(description);

	} else if(!header.compare("Additional Hints", Qt::CaseInsensitive)) {
	  Description hint;
	  hint.set(true, searchForHintEncrypted(tag->Children));
	  if(hint.isSet()) cache.setHint(hint);

	} else if(!header.compare("Logs", Qt::CaseInsensitive)) {
	  cache.clearLogs();
	  searchForLogs(tag->Children, cache);

	} else if(!header.compare("Attributes", Qt::CaseInsensitive)) {
	  cache.clearAttributes();
	  searchForAttributes(tag->Children, cache);

	} else if(!header.compare("Additional Waypoints", Qt::CaseInsensitive)) {
	  cache.clearWaypoints();
	  searchForWaypoints(tag->Children, cache);
	}

      }
    } else
      searchForItem(tag->Children, cache);
  }
}

bool GcHtmlParser::decode(const QString &data, Cache &cache) {
  m_error = QString();
  
  QSgml html(data); 
  QList<QSgmlTag*> body;
  html.getElementsByName("body", &body);
  if(body.size() > 0)
    searchForItem(body[0]->Children, cache);

  return true;
}

QString GcHtmlParser::error() const {
  return m_error;
}
