#include <QDebug>

#include <QGeoServiceProvider>
#include <QGraphicsSceneMouseEvent>
#include <QGeoMapOverlay>
#include <QSettings>
#include <QDesktopServices>

#include "mapwidget.h"

#if(QTM_VERSION < 0x010200)
#error "QtMobility version 1.2 or higher required!"
#endif

void MapWidget::updateZoomButtons() {
  this->m_mapOverlay->enableZoomButtons(
		zoomLevel() < maximumZoomLevel(),
		zoomLevel() > minimumZoomLevel());
  
  update();
}

MapWidget::MapWidget(QGeoMappingManager *mgr) : 
  QGraphicsGeoMap(mgr), m_bubble(NULL), m_posIsValid(false), m_gpsMarker(NULL),
  m_manager(mgr) {

  // try to get default from qsettings, otherwise start at berlin's zoo
  QSettings settings;
  settings.beginGroup("Map");
  setCenter(QGeoCoordinate(
           settings.value("Center/Latitude", 52.514).toFloat(),
	   settings.value("Center/Longitude", 13.3611).toFloat()));
  setZoomLevel(settings.value("ZoomLevel", 15).toInt());
  setMapType((QGraphicsGeoMap::MapType)settings.value("Type", 
	      QGraphicsGeoMap::StreetMap).toInt());
  m_iconSize = settings.value("IconSize", 48).toInt();  
  m_gpsLocked = settings.value("GpsLocked", false).toBool();
  m_precisionAlpha = settings.value("PrecisionAlpha", 32).toInt();
  settings.endGroup();
  m_iconLoader = new IconLoader(m_iconSize);

  // create the control overlays
  this->m_mapOverlay = new MapOverlay(this);
  addMapOverlay( this->m_mapOverlay );

  // and update it to reflect restored map state
  updateZoomButtons();
  if(m_gpsLocked)
    this->m_mapOverlay->changeGpsButton(true);

  // create a group for the cache icons
  m_cacheGroup = new QGeoMapGroupObject();
  m_cacheGroup->setZValue(0);
  addMapObject(m_cacheGroup);

  // create a gps marker and precision indicator, but hide them
  QPixmap *markerIcon = m_iconLoader->load("gps_marker");
  if(markerIcon) {
    m_gpsMarker = new QGeoMapPixmapObject(QGeoCoordinate(0,0), 
		     QPoint(-m_iconLoader->size()/2, -m_iconLoader->size()/2),
		  *markerIcon);

    m_gpsMarker->setZValue(2);
    m_gpsMarker->setVisible(false);
    addMapObject(m_gpsMarker);
  }

  m_gpsPrecision = new QGeoMapCircleObject();
  m_gpsPrecision->setPen(QPen((QColor(0, 0, 0, m_precisionAlpha))));
  m_gpsPrecision->setBrush(QBrush(QColor(0, 0, 0, m_precisionAlpha/2)));
  m_gpsPrecision->setZValue(1);
  addMapObject(m_gpsPrecision);

  // install timer to delay cache reload requests
  this->m_timer = new QTimer(this);
  this->m_timer->setSingleShot(true);
  QObject::connect(this->m_timer, SIGNAL(timeout()),
		   this, SLOT(reloadTimerExpired()));

  // connect to map signals to be able to handle viewport changes
  QObject::connect(this, SIGNAL(centerChanged(const QGeoCoordinate &)),
		   this, SLOT(centerChangedEvent(const QGeoCoordinate &)));
  QObject::connect(this, SIGNAL(zoomLevelChanged(qreal)),
		   this, SLOT(zoomLevelChangedEvent(qreal)));
}

MapWidget::~MapWidget() {
  qDebug() << __FUNCTION__;

  // save all kinds of map settings
  QSettings settings;
  settings.beginGroup("Map");
  settings.setValue("Center/Latitude", center().latitude());
  settings.setValue("Center/Longitude", center().longitude());
  settings.setValue("ZoomLevel", zoomLevel());
  settings.setValue("IconSize", m_iconSize);
  settings.setValue("GpsLocked", m_gpsLocked);  
  settings.setValue("PrecisionAlpha", m_precisionAlpha);
  settings.endGroup();

  delete m_iconLoader;
  delete m_timer;
}

// ------------ slots to handle overlay button clicks --------------
void MapWidget::zoomIn() {
  if(zoomLevel() < maximumZoomLevel()) {
    setZoomLevel(zoomLevel()+1);
    updateZoomButtons();
  } 

  if(zoomLevel() == maximumZoomLevel()) 
    emit showMessage(tr("Maximum zoom level reached"));
}

void MapWidget::zoomOut() {
  if(zoomLevel() > minimumZoomLevel()) {
    setZoomLevel(zoomLevel()-1);
    updateZoomButtons();
  }

  if(zoomLevel() == minimumZoomLevel()) 
    emit showMessage(tr("Minimum zoom level reached"));
}

void MapWidget::gpsFollow() {
  m_gpsLocked = true;
  this->m_mapOverlay->changeGpsButton(true);  
  update();
}

void MapWidget::resizeEvent(QGraphicsSceneResizeEvent *event) {
  QGraphicsGeoMap::resizeEvent(event);
  reload();
}

void MapWidget::showEvent(QShowEvent *) {
  reload();
}

void MapWidget::zoomLevelChangedEvent(qreal) {
  reload();
}

void MapWidget::centerChangedEvent(const QGeoCoordinate &) {
  reload();
}

void MapWidget::hideBubble() {
  // kill any existing bubble
  if(m_bubble) {
    removeMapObject(m_bubble);
    delete m_bubble;
    m_bubble = NULL;

    this->m_mapOverlay->enableForwardButton(false);
  }
}

void MapWidget::showBubble(const Cache &cache) {
  qDebug() << __FUNCTION__ << cache.name();

  hideBubble();
  m_bubble = new MapBubble(this, size().width() > size().height(), cache);
  addMapObject(m_bubble);

  this->m_mapOverlay->enableForwardButton(cache.url().isValid());
}

// currently connected to the "forward" button
void MapWidget::showDetails() {
  Q_ASSERT(m_bubble);

  if(!QDesktopServices::openUrl ( m_bubble->cache().url() ))
    emit showMessage(tr("Error opening cache url ") + m_bubble->cache().url().toString());
  else
    emit showMessage(tr("Opening \"%1\" in browser ...").
		     arg(m_bubble->cache().description()));
}

void MapWidget::mousePressEvent(QGraphicsSceneMouseEvent* event) {
  if(this->m_mapOverlay->mousePress(event->pos()))
    update();

  if(this->m_mapOverlay->isInside(event->pos()))
    m_downPos = QPointF(0,0);
  else
    m_downPos = event->pos();

  m_dragging = false;
}

void MapWidget::mouseReleaseEvent(QGraphicsSceneMouseEvent* event) {
  if(this->m_mapOverlay->mouseRelease(event->pos()))
    update();

  if(m_dragging && m_gpsLocked) {
    m_gpsLocked = false;
    this->m_mapOverlay->changeGpsButton(false);
    update();
  }

  // this doesn't work on n900 as the n900 port of qt mobility fails
  // at detecting clicks onto bitmaps
  if(m_downPos.isNull() || m_dragging)
    return;

  // check if we clicked a cache
  bool clickedBubble = false;
  QString clickedCache;

  QList<QGeoMapObject*> objects = mapObjectsAtScreenPosition(m_downPos);
  if (objects.length() > 0) {
    for (int i = objects.length()-1; i >= 0; i--) {
      if ((objects[i]->objectName() == "cache") && clickedCache.isEmpty())
	clickedCache = objects[i]->property("name").toString();
      
      if (objects[i]->objectName() == "bubble") 
	clickedBubble = true;
    }
  }

  if(!clickedBubble) {
    hideBubble();

    if(!clickedCache.isEmpty()) 
      emit cacheClicked(clickedCache);
  }
}

void MapWidget::mouseMoveEvent(QGraphicsSceneMouseEvent* event) {
  if(this->m_mapOverlay->mouseMove(event->pos()))
    update();

  if(m_downPos.isNull())
    return;

  if(!m_dragging) {
    if((m_downPos - event->pos()).manhattanLength() < DRAG_FUZZ) 
      return;

    // dragged more than DRAG_FUZZ pixels: start actual drag
    m_dragging = true;
    QPointF dp(m_downPos - event->pos());
    pan(dp.x(), dp.y());
    return;
  }

  QPointF d(event->lastPos() - event->pos());
  pan(d.x(), d.y());
}

MapWidget::MapCacheObject::MapCacheObject(IconLoader *loader, const Cache &cache) :
  QGeoMapPixmapObject(cache.coordinate(), 
		      QPoint(-loader->size()/2, -loader->size()/2),
		      *loader->load(cache)) {
  
  setZValue(0);  // below gps marker and bubble
  setObjectName("cache");
  setProperty("name", cache.name());
  setProperty("description", cache.description());
};

// the pixmap object seems to actually check for clicks onto the bitmap. thus
// we only check for the click being withing the entire pixmap size. furthermore
// the pixmap precise stuff doesn't seem to work at all on the n900/maemo
bool MapWidget::MapCacheObject::contains(const QGeoCoordinate &coo) const {
  return boundingBox().contains(coo);
}

void MapWidget::updateCaches(const CacheList &cacheList) {
  // save cache if existing bubble
  Cache bubbleCache;
  if(m_bubble) bubbleCache = m_bubble->cache();
  hideBubble();

  // remove all existing caches
  m_cacheGroup->clearChildObjects();

  // draw all new caches
  QList<Cache>::const_iterator i;
  for( i = cacheList.begin(); i != cacheList.end(); ++i ) 
    m_cacheGroup->addChildObject(new MapCacheObject(m_iconLoader, *i));

  // restore previous bubble
  if(bubbleCache.coordinate().isValid()) {
    // The ugly part: gc.com is randomizing coordinates and
    // the new coordinates used to draw the icon are likely
    // different from the ones saved from the previous bubble
    // position. So we search for the bubble cache within the
    // new cache list and use it's coordinates instead
    
    bool coordinateUpdated = false;
    for( i = cacheList.begin(); i != cacheList.end(); ++i ) {
      if(i->name() == bubbleCache.name()) {
	bubbleCache.setCoordinate(i->coordinate());
	coordinateUpdated = true;
      }
    }

    // only redraw bubble if the cache is still part of the
    // new cache list and if it's cache is still on-screen
    if(coordinateUpdated && 
       viewport().contains(bubbleCache.coordinate()))
      showBubble(bubbleCache);
  }
}

void MapWidget::reloadTimerExpired() {
  qDebug() << __FUNCTION__;

  // check if viewport actually changed
  if(m_currentViewport != viewport()) {
    emit mapChanged();
    m_currentViewport = viewport();
  }
}

void MapWidget::reload() {
  // start the timer if it's not already running, re-start from zero otherwise
  // using less than 1000ms here makes sure that requests are being sent for
  // every updated gps position (reported every 1000ms)
  if(!this->m_timer->isActive())
    this->m_timer->start(900);
  else
    this->m_timer->setInterval(900);
}

void MapWidget::positionUpdated(const QGeoPositionInfo &pos) {
  // change state of gps button to reflect gps availability
  if(m_posIsValid != pos.coordinate().isValid()) {
    this->m_mapOverlay->enableGpsButton(pos.coordinate().isValid());
    update();
    m_posIsValid = pos.coordinate().isValid();
  }

  // re-center map if gpsLock is active
  if(pos.coordinate().isValid() && m_gpsLocked && !m_dragging) 
    setCenter(pos.coordinate());

  // draw gps marker
  if(m_gpsMarker) {
    if(pos.coordinate().isValid()) {
      m_gpsMarker->setVisible(true);
      m_gpsMarker->setCoordinate(pos.coordinate());

      // horizontal accuracy, will return -1 if unset
      // and circle will be hidden if radius < 0
      m_gpsPrecision->setRadius(pos.attribute(QGeoPositionInfo::HorizontalAccuracy));
      m_gpsPrecision->setCenter(pos.coordinate());
    } else {
      m_gpsMarker->setVisible(false);
      m_gpsPrecision->setRadius(-1);
    }
  }
}

QString MapWidget::managerName() {
  return m_manager->managerName();
}
