/*
  gcprovider - online cache provider for cacheme

  the gc provider works asynchronous and keeps a queue of things it has
  to do
*/

#include <QDebug>

#include "gcprovider.h"
#include "gcparser.h"

// The GcUrl class automatically creates urls for geocaching.com
class GcUrl : public QUrl {
public:
  GcUrl(const QString &url) : 
    QUrl("http://www.geocaching.com" + url) {}
};

// the GcNetworkRequest pre-sets some header values
class GcNetworkRequest : public QNetworkRequest {
public:
  GcNetworkRequest() : QNetworkRequest() {
    setRawHeader("User-Agent", "Cacheme 1.0");
  }
};

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcProvider::subString(QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

// ------------ handling of user token ---------------

bool GcProvider::decodeUserToken(const QByteArray &data) {
  QString str(data);
  this->m_userToken = subString(str,  "userToken = '", "';");
  return !this->m_userToken.isNull();
}

void GcProvider::requestUserToken() {
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx"));
  this->m_manager->get(request);
}

GcProvider::GcProvider() : m_initialized(false), m_cacheList("GcProvider") {
  qDebug() << __FUNCTION__;

  // setup network manager and listen for its replies
  this->m_manager = new QNetworkAccessManager(this);

  connect(this->m_manager, SIGNAL(finished(QNetworkReply*)),
	  this, SLOT(replyFinished(QNetworkReply*)));

  // initialize network (e.g. get a user token and log in)
  requestUserToken();
}

GcProvider::~GcProvider() {
  qDebug() << __FUNCTION__;
}

QString GcProvider::name() {
  return "GcProvider";
}

bool GcProvider::initialized() {
  return m_initialized;
}

void GcProvider::replyFinished(QNetworkReply *reply) {
  qDebug() << __FUNCTION__;

  // whatever happened, we are now idle again
  if(reply->error() != QNetworkReply::NoError) {
    qDebug() << __FUNCTION__ << "Reply error:" << reply->errorString();
  } else {

    // invoke appropriate decoder
    if(reply->isFinished()) {

      // as long as the gcprovider isn't initialized all replys
      // be handled internally
      if(!m_initialized) {
	qDebug() << __FUNCTION__ << "initialization reply";
	if(decodeUserToken(reply->readAll())) {
	  qDebug() << __FUNCTION__ << "UserToken is" << this->m_userToken;
	}

	// now everything is initialized
	m_initialized = true;
      } else {
	qDebug() << __FUNCTION__ << "reply";

	// initialization is finished, any reply received now should
	// be a reply to a real request

	// save type of request that just finished
	RequestEntry::RequestType type = m_pending->type();
	m_pending->done();

	GcParser gcParser;

	switch(type) {
	case RequestEntry::Overview: 
	  // start parser (and tell him whether we tried to load a "big" area)
	  if(gcParser.decodeOverview(reply->readAll(), m_cacheList, m_bigArea))
	    emit replyOverview(m_cacheList);
	  else
	    emit replyError(gcParser.error());
	  break;
	  
	case RequestEntry::Info: 
	  if(gcParser.decodeInfo(reply->readAll(), m_cache))
	    emit replyInfo(m_cache);
	  else
	    emit replyError(gcParser.error());
	  break;

	default:
	  qDebug() << __FUNCTION__ << "unknown request type" << type;
	  Q_ASSERT(0);
	  break;
	}
      }
      m_pending->next();
    }
  }
}

void GcProvider::postJson(const QString &post) {
  // build and send request
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx/MapAction"));
  request.setRawHeader("Accept", "application/json");
  request.setRawHeader("Content-Type", "application/json; charset=utf-8");
  m_manager->post(request, post.toUtf8());
}

void GcProvider::processRequestOverview(const QGeoBoundingBox &area) {
  qDebug() << __PRETTY_FUNCTION__;

  // check if bounding box doesn't exceed limits (40km diagonally)
  this->m_bigArea = area.topLeft().distanceTo(area.bottomRight()) > 40000;
  
  // build request string
  postJson("{\"dto\":{\"data\":{\"c\":1,\"m\":\"\",\"d\":\"" + 
	   QString::number(area.topLeft().latitude()) + "|" + 
	   QString::number(area.bottomRight().latitude()) + "|" + 
	   QString::number(area.bottomRight().longitude()) + "|" + 
	   QString::number(area.topLeft().longitude()) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

void GcProvider::processRequestInfo(const QString &name) {
  int id = -1;

  qDebug() << __PRETTY_FUNCTION__ << name;

  // try to find matching cache id
  QList<Cache>::const_iterator i;
  for( i = m_cacheList.begin(); i != m_cacheList.end(); ++i ) {
    if(i->name() == name) {
      id = i->id();
      m_cache = *i;
    }
  }

  if(id < 0) {
    qDebug() << __FUNCTION__ << "unable to resolve cache name to gc.com id";

    // remove current request and continue with the next one
    m_pending->done();
    m_pending->next();

    emit replyError("Unable to determine cache id");
    return;
  }

  qDebug() << __FUNCTION__ << "id" << id;

  postJson("{\"dto\":{\"data\":{\"c\":2,\"m\":\"\",\"d\":\"" + 
	   QString::number(id) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

void GcProvider::requestDetail(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;
}

