// Filename: mainwindow.cpp

#include <QDebug>
#include <QMainWindow>

#include "mainwindow.h"
#include "mapwidget.h"

#ifdef Q_WS_MAEMO_5
#include <QMaemo5InformationBox>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

void grabZoomKeys(QWidget *window) {
  if(!window->winId()) return;
  
  unsigned long val = 1;
  Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
  if(!atom) return;
  
  XChangeProperty (QX11Info::display(), window->winId(), atom, 
		   XA_INTEGER, 32, PropModeReplace,
		   reinterpret_cast<unsigned char *>(&val),  1);
}
#endif

MainWindow::MainWindow(LocationProvider *locationProvider, QWidget *parent) : 
  QMainWindow(parent), m_locationProvider(locationProvider) {

  qDebug() << __FUNCTION__ << "Cache provider " << m_cacheProvider.name();

  setWindowTitle("CacheMe");

#ifdef Q_WS_MAEMO_5
  setAttribute(Qt::WA_Maemo5AutoOrientation, true);
  grabZoomKeys(this);
#endif

  this->m_serviceProvider = new QGeoServiceProvider("nokia");
  this->m_mapWidget = new MapWidget(this->m_serviceProvider->mappingManager());

  // make sure cache updates are processed
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QGeoCoordinate &)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QGeoCoordinate &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyOverview(const CacheList &)), 
		   this, SLOT(updateCaches(const CacheList &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyInfo(const Cache &)), 
		   this->m_mapWidget, SLOT(showBubble(const Cache &)));

  QObject::connect(&this->m_cacheProvider, SIGNAL(replyError(const QString &)), 
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_mapWidget, SIGNAL(mapChanged()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this, SIGNAL(increasePressed()),
		   this->m_mapWidget, SLOT(zoomIn()));

  QObject::connect(this, SIGNAL(decreasePressed()),
		   this->m_mapWidget, SLOT(zoomOut()));

  // connect to map to handle clicks on cache icons
  QObject::connect(this->m_mapWidget, SIGNAL(cacheClicked(const QString &)),
		   this, SLOT(selectCache(const QString &)));

  QGraphicsScene *scene = new QGraphicsScene(this);
  m_view = new QGraphicsView( scene );
  m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVisible(true);
  m_view->setInteractive(true);

  scene->addItem(m_mapWidget);

  setCentralWidget(m_view);

  QSettings settings;
  settings.beginGroup("MainWindow");
  resize(settings.value("size", QSize(640, 480)).toSize());
  move(settings.value("pos", QPoint(200, 200)).toPoint());
  settings.endGroup();
}

MainWindow::~MainWindow() {
  qDebug() << __FUNCTION__;

  QSettings settings;
  settings.beginGroup("MainWindow");
  settings.setValue("size", size());
  settings.setValue("pos", pos());
  settings.endGroup();
  
  delete m_serviceProvider;
}

void MainWindow::resizeEvent(QResizeEvent*) {
  m_view->setSceneRect(QRectF(QPointF(0.0, 0.0), m_view->size()));
  m_mapWidget->resize(m_view->size());
}

void MainWindow::keyPressEvent(QKeyEvent* event) {
  // check if device is in portrait mode
  bool portrait = size().height() > size().width();

  switch (event->key()) {
  case Qt::Key_F7:
    if(portrait) emit decreasePressed();
    else         emit increasePressed();
    event->accept();
    break;
    
  case Qt::Key_F8:
    if(portrait) emit increasePressed();
    else         emit decreasePressed();
    event->accept();
    break;
  }
  QWidget::keyPressEvent(event);
}

void MainWindow::selectCache(const QString &name) {
  // ask CacheProvider for details about this cache
  this->m_cacheProvider.requestInfo(name);
}

void MainWindow::reloadCaches() {
  // request new caches for the maps new viewport
  m_cacheProvider.requestOverview(this->m_mapWidget->viewport());
}

void MainWindow::updateCaches(const CacheList &cacheList) {
  m_message = "";
  this->m_mapWidget->updateCaches(cacheList);
}

void MainWindow::displayError(const QString &message) {
  if(m_message != message) {
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, message);
#endif
    m_message = message;
  }
}
