/*
 * btc-watc-movie.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include <string.h>

#include "btc-watc-movie.h"

G_DEFINE_TYPE (BtcWatcMovie, btc_watc_movie, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_NAME,
        PROP_STINGERS,
        PROP_YEAR,
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_WATC_MOVIE, BtcWatcMoviePrivate))

struct _BtcWatcMoviePrivate {
        gchar *name;
        int stingers;
        gchar *year;
};

static void
btc_watc_movie_get_property (GObject *object, guint property_id,
                         GValue *value, GParamSpec *pspec)
{
        BtcWatcMovie *self = BTC_WATC_MOVIE (object);

        switch (property_id) {
        case PROP_NAME:
                g_value_set_string (value, self->priv->name);
                break;
        case PROP_STINGERS:
                g_value_set_enum (value, self->priv->stingers);
                break;
        case PROP_YEAR:
                g_value_set_string (value, self->priv->year);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_watc_movie_set_property (GObject *object, guint property_id,
                         const GValue *value, GParamSpec *pspec)
{
        BtcWatcMovie *self = BTC_WATC_MOVIE (object);

        switch (property_id) {
        case PROP_NAME:
                btc_watc_movie_set_name (self, g_value_get_string (value));
                break;
        case PROP_STINGERS:
                btc_watc_movie_set_stingers (self, g_value_get_int (value));
                break;
        case PROP_YEAR:
                btc_watc_movie_set_year (self, g_value_get_string (value));
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_watc_movie_finalize (GObject *object)
{
        BtcWatcMovie *self = BTC_WATC_MOVIE (object);

        g_free (self->priv->name);
        g_free (self->priv->year);

        G_OBJECT_CLASS (btc_watc_movie_parent_class)->finalize (object);
}

static void
btc_watc_movie_class_init (BtcWatcMovieClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcWatcMoviePrivate));

        object_class->get_property = btc_watc_movie_get_property;
        object_class->set_property = btc_watc_movie_set_property;
        object_class->finalize = btc_watc_movie_finalize;

        g_object_class_install_property
                (object_class, PROP_NAME,
                 g_param_spec_string ("name", "Movie name",
                                 "The movie name",
                                 NULL,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_STINGERS,
                 g_param_spec_int ("stingers", "Movie stingers",
                                 "Tells if movie has stingers",
                                 0,
                                 BTC_STINGERS_LAST,
                                 BTC_STINGERS_UNKNOWN,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_YEAR,
                 g_param_spec_string ("year", "Movie year",
                                 "The movie year",
                                 NULL,
                                 G_PARAM_READWRITE));
}

static void
btc_watc_movie_init (BtcWatcMovie *self)
{
        self->priv = GET_PRIVATE (self);

        self->priv->name = NULL;
        self->priv->stingers = BTC_STINGERS_UNKNOWN;
        self->priv->year = NULL;
}

BtcWatcMovie*
btc_watc_movie_new (const gchar *raw_name)
{
        BtcWatcMovie *instance = g_object_new (BTC_TYPE_WATC_MOVIE, NULL);
        int len = strlen (raw_name);

        /* The raw name comes with the format
         * $(MOVIE_NAME) $(YEAR)$(STINGERS)
         *
         * If stingers are present, actual length is len-1.
         * If date is present, actual length is len-6
         * */

        if (g_str_has_suffix (raw_name, "*")) {
                instance->priv->stingers = BTC_STINGERS_YES;
                len--;
        }
        else if (g_str_has_suffix (raw_name, "?")) {
                instance->priv->stingers = BTC_STINGERS_UNKNOWN;
                len--;
        }
        else
                instance->priv->stingers = BTC_STINGERS_NO;

        if (raw_name[len-1] == ')') {
                len = len -6;

                gchar *year = g_strndup (raw_name + len +1, 4);
                btc_watc_movie_set_year (instance, year);
                g_free (year);
        }

        gchar *name = g_strndup (raw_name, len);

        btc_watc_movie_set_name (instance, name);

        g_free (name);
        return instance;
}

gboolean
btc_watc_movie_set_name (BtcWatcMovie *self, const gchar *name)
{
        g_return_val_if_fail (BTC_IS_WATC_MOVIE (self), FALSE);

        g_free (self->priv->name);

        self->priv->name = g_strdup (name);

        return TRUE;
}

gboolean
btc_watc_movie_set_stingers (BtcWatcMovie *self, int stingers)
{
        g_return_val_if_fail (BTC_IS_WATC_MOVIE (self), FALSE);
        g_return_val_if_fail (stingers == BTC_STINGERS_UNKNOWN ||
                              stingers == BTC_STINGERS_YES ||
                              stingers == BTC_STINGERS_NO, FALSE);

        self->priv->stingers = stingers;

        return TRUE;
}

const gchar*
btc_watc_movie_get_name (BtcWatcMovie *self)
{
        g_return_val_if_fail (BTC_IS_WATC_MOVIE (self), NULL);

        return self->priv->name;
}

int
btc_watc_movie_get_stingers (BtcWatcMovie *self)
{
        g_return_val_if_fail (BTC_IS_WATC_MOVIE (self), FALSE);

        return self->priv->stingers;
}

void
btc_watc_movie_print (BtcWatcMovie *self)
{
        const gchar *has_stingers = NULL;
        g_return_if_fail (BTC_IS_WATC_MOVIE (self));

        if (self->priv->stingers == BTC_STINGERS_NO)
                has_stingers = "No";
        else if (self->priv->stingers == BTC_STINGERS_YES)
                has_stingers = "Yes";
        else
                has_stingers = "Unknown";

        g_print ("[Name]: %s\n", self->priv->name);
        g_print ("[Stingers]: %s\n", has_stingers);
        g_print ("[Year]: %s\n",
                 self->priv->year ? self->priv->year : " - ");
}

gboolean
btc_watc_movie_set_year (BtcWatcMovie *self, const gchar *year)
{
        g_return_val_if_fail (BTC_IS_WATC_MOVIE (self), FALSE);

        g_free (self->priv->year);

        self->priv->year = g_strdup (year);

        return self->priv->year != NULL;
}

const gchar*
btc_watc_movie_get_year (BtcWatcMovie *self)
{
        return self->priv->year;
}
