/*
 * btc-tmdb-movie-service.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include <dbus/dbus-glib-bindings.h>

#include "btc-tmdb-movie-service.h"
#include "btc-tmdb-image.h"

#define TMDB_MOVIE_SERVICE_OBJECT_PATH "/TMDBMovie"
#define TMDB_MOVIE_SERVICE_NAME "com.simonpena.butaca.tmdbmovie"

G_DEFINE_TYPE (BtcTmdbMovieService, btc_tmdb_movie_service, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_ID,
        PROP_DBUSGCONN,
};

#define GET_PRIVATE(o) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_TMDB_MOVIE_SERVICE, BtcTmdbMovieServicePrivate))

struct _BtcTmdbMovieServicePrivate {
        BtcTmdbMovie *movie;
        DBusGConnection *connection;
        gchar *suffix;
};

gboolean
btc_tmdb_movie_service_get_title (BtcTmdbMovieService *self, gchar **title,
                                  GError **error)
{
        *title = g_strdup (btc_tmdb_movie_get_name (self->priv->movie));
        return *title != NULL;
}

gboolean
btc_tmdb_movie_service_get_popularity (BtcTmdbMovieService *self, gchar **popularity,
                                       GError **error)
{
        *popularity = g_strdup (btc_tmdb_movie_get_popularity (self->priv->movie));
        return *popularity != NULL;
}

gboolean
btc_tmdb_movie_service_get_rating (BtcTmdbMovieService *self, gchar **rating,
                                   GError **error)
{
        *rating = g_strdup (btc_tmdb_movie_get_rating (self->priv->movie));
        return *rating != NULL;
}

gboolean
btc_tmdb_movie_service_get_released (BtcTmdbMovieService *self, gchar **released,
                                     GError **error)
{
        *released = g_strdup (btc_tmdb_movie_get_released (self->priv->movie));
        return *released != NULL;
}

gboolean
btc_tmdb_movie_service_get_overview (BtcTmdbMovieService *self, gchar **overview,
                                     GError **error)
{
        *overview = g_strdup (btc_tmdb_movie_get_overview (self->priv->movie));
        return *overview != NULL;
}

gboolean
btc_tmdb_movie_service_get_images (BtcTmdbMovieService *self, GPtrArray **image_array,
                                   GError **error)
{
        GList *iter = NULL;
        *image_array = g_ptr_array_new ();

        GList *movie_images = btc_tmdb_movie_get_images (self->priv->movie);
        for (iter = movie_images; iter; iter = iter->next) {

                GValueArray *image_properties = g_value_array_new (4);
                BtcTmdbImage *movie_image = BTC_TMDB_IMAGE (iter->data);

                GValue value = { 0 };
                g_value_init (&value, G_TYPE_STRING);

                g_value_set_string (&value,
                                    btc_tmdb_image_get_imagetype (movie_image));

                g_value_array_append (image_properties,
                                      &value);

                g_value_set_string (&value,
                                    btc_tmdb_image_get_url (movie_image));

                g_value_array_append (image_properties,
                                      &value);

                g_value_set_string (&value,
                                    btc_tmdb_image_get_size (movie_image));

                g_value_array_append (image_properties,
                                      &value);

                g_value_set_string (&value,
                                    btc_tmdb_image_get_id (movie_image));

                g_value_array_append (image_properties,
                                      &value);

                g_ptr_array_add (*image_array, image_properties);
        }

	return image_array != NULL;
}

#include "btc-tmdb-movie-service-glue.h"

static void
setup_dbus (BtcTmdbMovieService *self)
{
        DBusGProxy *proxy;
        guint request_name_result;
        GError *error = NULL;
        gchar *object_path = NULL;

        proxy = dbus_g_proxy_new_for_name (self->priv->connection,
                                           DBUS_SERVICE_DBUS,
                                           DBUS_PATH_DBUS,
                                           DBUS_INTERFACE_DBUS);

        if (!org_freedesktop_DBus_request_name (proxy,
                                                TMDB_MOVIE_SERVICE_NAME,
                                                0, &request_name_result,
                                                &error)) {
                g_warning ("Unable to register service: %s", error->message);
                g_error_free (error);
        }

        object_path = g_strdup_printf (TMDB_MOVIE_SERVICE_OBJECT_PATH "/%s",
                        self->priv->suffix);

        dbus_g_connection_register_g_object (self->priv->connection,
                                             object_path,
                                             G_OBJECT (self));

        g_free (object_path);
        g_object_unref (proxy);
}

static void
btc_tmdb_movie_service_set_property (GObject *object, guint property_id,
                                  const GValue *value, GParamSpec *pspec)
{
        BtcTmdbMovieService *self = BTC_TMDB_MOVIE_SERVICE (object);

        switch (property_id) {
        case PROP_DBUSGCONN:
                if (!self->priv->connection) {
                        DBusGConnection *tmp = g_value_get_pointer (value);
                        if (tmp) {
                                self->priv->connection =
                                        dbus_g_connection_ref (tmp);
                                setup_dbus (self);
                        }
                }
                g_assert (self->priv->connection);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_tmdb_movie_service_finalize (GObject *object)
{
        BtcTmdbMovieService *self = BTC_TMDB_MOVIE_SERVICE (object);

        if (self->priv->connection) {
                dbus_g_connection_unref (self->priv->connection);
        }
        g_free (self->priv->suffix);
        g_object_unref (self->priv->movie);
        G_OBJECT_CLASS (btc_tmdb_movie_service_parent_class)->finalize (object);
}

static void
btc_tmdb_movie_service_class_init (BtcTmdbMovieServiceClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcTmdbMovieServicePrivate));

        object_class->set_property = btc_tmdb_movie_service_set_property;
        object_class->finalize = btc_tmdb_movie_service_finalize;

        g_object_class_install_property
                (object_class, PROP_DBUSGCONN,
                 g_param_spec_pointer ("connection", "DBusGConnection",
                                       "DBus GConnection",
                                       G_PARAM_WRITABLE));

         dbus_g_object_type_install_info (BTC_TYPE_TMDB_MOVIE_SERVICE,
                                          &dbus_glib_btc_tmdb_movie_service_object_info);
}

static void
btc_tmdb_movie_service_init (BtcTmdbMovieService *self)
{
        self->priv = GET_PRIVATE (self);
        self->priv->movie = NULL;
        self->priv->connection = NULL;
        self->priv->suffix = NULL;
}

BtcTmdbMovieService*
btc_tmdb_movie_service_new (DBusGConnection *connection,
                BtcTmdbMovie *movie, const gchar *suffix)
{
        BtcTmdbMovieService *instance = g_object_new (BTC_TYPE_TMDB_MOVIE_SERVICE, NULL);
        instance->priv->suffix = g_strdup(suffix);
        g_object_set (instance, "connection", connection, NULL);
        instance->priv->movie = movie;
        return instance;
}
