#!/bin/sh
# A script to replace /bin/busybox and creates symlinks to new functions.
# The default locations of busybox' functions (applets) are defined in the file $INSTALLDIR/functions
# It keeps track of the installed symlinks by writing them to $INSTALLDIR/installed-symlinks in
# a similiar fashion as locations are defined in the "functions" file.
#
# The scripts check whether symlinks/binaries of the utilities already exist, and if not,
# it checks whether the new busybox binary supports it. If so, it creates a symlink to /bin/busybox.
#
# NB The busybox binary needs to support the install applet

# By Dennis Groenen <dennis_groenen@hotmail.com>
# GPLv3 licensed

# Version 0.2 05-16-2011 (MM-DD-YYYY)
# 0.1: Initial release
# 0.2: Use $EXECPWR to not have to rely on /bin/busybox' functions
#      Minor clean-ups and be quieter

INSTALLDIR="/opt/busybox-power"
EXECPWR="$INSTALLDIR/busybox.power"
VERBOSE="0"

if test "`$EXECPWR id -u`" -ne 0; then
  echo "Error: you're not running me as root, aborting"
  echo "  Also, DO NOT run me as a stand-alone application"
  echo "  I will be called automatically upon installation"
  echo "  of busybox-power"
  exit 1
fi

#if test -n "`pgrep dpkg`" -o "`pgrep apt`"
if ! lsof /var/lib/dpkg/lock >> /dev/null
  then 
    echo "Error: you're running me as a stand-alone application"
    echo "  Do not do this, I will be called automatically upon"
    echo "  installation of busybox-power"
    exit 1
fi

####### Sanity checks before continuing installation
if test ! -e $INSTALLDIR/functions
  then
    echo "Error: cannot find list of defined functions"
    exit 1
fi

if test -e $INSTALLDIR/installed-symlinks
  then
    echo "Error: symlinks already seem to be made?"
    echo "  This script is not supposed to be ran twice"
    exit 1
fi
####### End of sanity checklist

if test $VERBOSE == 1; then echo "Binary: $EXECPWR" \ 
echo "Version string: `$EXECPWR | $EXECPWR head -n 1`"; fi

md5sum $INSTALLDIR/busybox.power | $EXECPWR awk '{ print $1 }' > $INSTALLDIR/busybox.power.md5
md5sum /bin/busybox | $EXECPWR awk '{ print $1 }' > $INSTALLDIR/busybox.original.md5

# Check whether busybox-power isn't installed already
INSTBINARY_MD5=`$EXECPWR cat $INSTALLDIR/busybox.power.md5`
ORIGBINARY_MD5=`$EXECPWR cat $INSTALLDIR/busybox.original.md5`
if test ! "$INSTBINARY_MD5" == "$ORIGBINARY_MD5"
  then
    $EXECPWR cp /bin/busybox $INSTALLDIR/busybox.original
    $EXECPWR cp -f $INSTALLDIR/busybox.power /bin/busybox
fi

# Load defined BusyBox functions
source $INSTALLDIR/functions

# Get a list of supported functions by the new binary
if test -d /tmp/busybox-power; then $EXECPWR rm -Rf /tmp/busybox-power; fi
$EXECPWR mkdir -p /tmp/busybox-power
$INSTALLDIR/busybox.power --install -s /tmp/busybox-power
$EXECPWR ls /tmp/busybox-power/ > $INSTALLDIR/functions_supported
$EXECPWR rm -Rf /tmp/busybox-power

# Prepare file which keeps track of installed symlinks by busybox-power
echo "# Automatically generated by busybox-power. DO NOT EDIT" > $INSTALLDIR/installed-symlinks
echo -e "\nDESTINATIONS=\"$DESTINATIONS\"" >> $INSTALLDIR/installed-symlinks
echo -e "\n# Installed symlinks" >> $INSTALLDIR/installed-symlinks

# Walk through all possible destinations
for DESTDIR in $DESTINATIONS
  do 
    # Enable us to see all entries in $DESTINATION as variables
    eval "APPLICATIONS=\$$DESTDIR"

    # Set destination directory accordingly
    case $DESTDIR in
      DEST_BIN)
	DIR="/bin"
      ;;
      DEST_SBIN)
	DIR="/sbin"
      ;;
      DEST_USRBIN)
	DIR="/usr/bin"
      ;;
      DEST_USRSBIN)
	DIR="/usr/sbin"
      ;;
    esac

  # Keep track of installed symlinks per destination
  SYMLINKS="$DESTDIR=\""

  if test $VERBOSE == 1; then echo -e "\nSymlinking functions in $DIR"; fi
  # Walk through all applications from the current destination
  for APP in $APPLICATIONS
    do
      # The following code is executed for every application in the current destination
      if test ! -e $DIR/$APP
	then
	  if `$EXECPWR grep -Fq "$APP" $INSTALLDIR/functions_supported` # Check whether the function is supported by the busybox binary
	    then
	      if test $VERBOSE == 1; then echo Symlinking: $DIR/$APP; fi
	      $EXECPWR ln -s /bin/busybox $DIR/$APP
	      SYMLINKS="$SYMLINKS $APP" 
	  fi
      fi
  done

  # Write out installed symlinks
  echo "$SYMLINKS\"" >> $INSTALLDIR/installed-symlinks
done

$EXECPWR rm $INSTALLDIR/functions_supported
echo "do not delete me" > $INSTALLDIR/installed
