#include "mapwidget.h"

MapWidget::MapWidget(const QGeoCoordinate& station, const QGeoCoordinate& user, QWidget* parent)
    : QWidget(parent)
{

    // When the network request finishes, call the function which handles
    // the received data
    QObject::connect(&networkAccessManager,
                     SIGNAL(finished(QNetworkReply*)),
                     this,
                     SLOT(handleNetworkData(QNetworkReply*)));

    setMinimumSize(400,400);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    fetchMap(size(), station, user);
}

void MapWidget::fetchMap(const QSize& size, const QGeoCoordinate& station, const QGeoCoordinate& user)
{

    qreal latitude = station.latitude();
    qreal longitude = station.longitude();

    qreal user_latitude = user.latitude();
    qreal user_longitude = user.longitude();

    qreal center_latitude = (latitude + user_latitude)/2;
    qreal center_longitude = (longitude + user_longitude)/2;

    qreal distance_to_fit = station.distanceTo(user);
    qreal logpartup = (6.28318531*6372.795*1000)/(distance_to_fit); // 2*PI*EARTH_SPHERE_RADIUS

    qreal logpartupval = log(logpartup);

    qreal logpartdownresult = log(2.0);
    int zoom = logpartupval/logpartdownresult;

    if (zoom > 17) zoom = 17;

    const QString GOOGLE_MAPS_URL_TEMPLATE =
            "http://maps.google.com/maps/api/staticmap?center=%1,%2&zoom=%7&size=%3x%4&maptype=mobile&markers=color:red|label:B|%8,%9&markers=color:blue|label:Y|%5,%6&sensor=false";

    QUrl url = QUrl(GOOGLE_MAPS_URL_TEMPLATE.arg(
            QString::number(center_latitude), QString::number(center_longitude),
            QString::number(size.width()), QString::number(size.height()),
            QString::number(user_latitude), QString::number(user_longitude),
            QString::number(zoom),
            QString::number(latitude), QString::number(longitude)
            ));

    QNetworkRequest request;
    request.setUrl(url);
    networkAccessManager.get(request);
}

void MapWidget::handleNetworkData(QNetworkReply* reply)
{
    QImage image;
    if (reply->error() == QNetworkReply::NoError) {
        image.load(reply, 0);
        if (!image.isNull()) {
            // Convert QImage into QPixmap and store it into a member
            // variable
            mapPixmap = QPixmap::fromImage(image);
        }
    }
    reply->deleteLater();

    // Update the display
    update();
}

void MapWidget::paintEvent(QPaintEvent* paintEvent)
{
    // Draw the map tile onto the display
    QPainter painter(this);
    painter.drawPixmap(0, 0, mapPixmap);
}
