/* This file is part of Beifahrer.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * Beifahrer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beifahrer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beifahrer. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class LiftListWindow : StackableWindow {
	AdacMitfahrclub adac;
	ListStore store;
	TreeView tree;
	TreeViewColumn route_column;
	Label no_lifts;

	public LiftListWindow (AdacMitfahrclub _adac) {
		adac = _adac;
	}

	construct {
		set_title ("Beifahrer");

		store = new ListStore (6, typeof (string), typeof (string), typeof (string), typeof (string), typeof (string), typeof (Lift));

		tree = (TreeView) Hildon.gtk_tree_view_new_with_model (UIMode.NORMAL, store);
		Hildon.gtk_widget_set_theme_size (tree, SizeType.FINGER_HEIGHT);

		tree.set_headers_visible (false);
		tree.set_rules_hint (true);

		// Tree selection object
		var selection = tree.get_selection ();
		selection.set_mode (SelectionMode.SINGLE);

		// Source and destination column
		route_column = new TreeViewColumn.with_attributes ("Route", new CellRendererText (), "text", 0);
		route_column.set_reorderable (false);
		route_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		route_column.set_expand (true);
		tree.append_column (route_column);

		// Date and time column
		var datetime_column = new TreeViewColumn.with_attributes ("DateTime", new CellRendererText (), "text", 1);
		datetime_column.set_reorderable (false);
		datetime_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		datetime_column.set_expand (true);
		tree.append_column (datetime_column);

		// Free places column
		var places_column = new TreeViewColumn.with_attributes ("Places", new CellRendererText (), "text", 2);
		places_column.set_reorderable (false);
		places_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		places_column.set_expand (true);
		tree.append_column (places_column);

		// Price column
		var price_column = new TreeViewColumn.with_attributes ("Price", new CellRendererText (), "text", 3);
		price_column.set_reorderable (false);
		price_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		price_column.set_expand (true);
		tree.append_column (price_column);

		// Smoker/Non-smoker column
		var smoke_column = new TreeViewColumn.with_attributes ("Smoker", new CellRendererPixbuf (), "icon-name", 4);
		smoke_column.set_reorderable (false);
		smoke_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		smoke_column.set_expand (false);
		tree.append_column (smoke_column);

		var pannable = new PannableArea ();
		pannable.add (tree);

		no_lifts = new Label (_("No lifts"));
		Hildon.helper_set_logical_font (no_lifts, "LargeSystemFont");
		Hildon.helper_set_logical_color (no_lifts, RcFlags.FG, StateType.NORMAL, "SecondaryTextColor");
		no_lifts.set_size_request (-1, 6 * 70);
		no_lifts.set_alignment ((float) 0.5, (float) 0.42);

		var vbox = new VBox (false, 0);
		vbox.pack_start (pannable, true, true, 0);
		vbox.pack_start (no_lifts, false, false, 0);

		add (vbox);

		tree.row_activated.connect (this.on_row_activated);
		BeifahrerProgram.orientation.changed.connect (this.on_orientation_changed);
	}

	public void find_lifts (string city_from, string city_to, Date date) {
		set_title ("%s - %s".printf (city_from, city_to));

		var lift_list = adac.get_lift_list (city_from, city_to, date);
		foreach (Lift lift in lift_list) {
			TreeIter iter;
			string icon_name = null;
			if (LiftFlags.SMOKER in lift.flags)
				icon_name = "beifahrer_smoker";
			else if (LiftFlags.NON_SMOKER in lift.flags)
				icon_name = "beifahrer_non_smoker";
			store.insert_with_values (out iter, -1, 0, lift.city_from + " - " + lift.city_to,
			                                        1, (lift.time != null) ? (lift.date + ", " + lift.time) : lift.date,
			                                        2, _("%d pl.").printf (lift.places),
			                                        3, lift.price,
								4, icon_name,
			                                        5, lift);
			// (LiftFlags.WOMEN_ONLY,ADAC_MEMBER in lift.flags)
		}
		if (lift_list.length () > 0)
			no_lifts.hide ();
	}

	private void on_row_activated (TreeView tree, TreePath /*_*/path, TreeViewColumn column) {
		TreeModel model = tree.model;
		TreeIter iter;
		Lift lift;

		if (model.get_iter (out iter, path)) {
			model.get (iter, 5, out lift);

			if (lift.description == null)
				Hildon.gtk_window_set_progress_indicator (this, 1);
			var window = new LiftDetailWindow (adac, lift);
			window.show_all ();

			// TODO - this should be async
			if (lift.description == null) {
				adac.update_lift_details (lift);
				window.update_lift_details ();
				Hildon.gtk_window_set_progress_indicator (this, 0);
			}
		}
	}

	private void on_orientation_changed () {
		if (BeifahrerProgram.orientation.portrait) {
			route_column.set_visible (false);
		} else {
			route_column.set_visible (true);
		}
	}
}
