#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <hal/libhal.h>
#include <dbus/dbus.h>

#include <stdio.h>
 
#include <libhildondesktop/libhildondesktop.h>

#include "battery.h"

HD_DEFINE_PLUGIN_MODULE(BatteryPlugin, battery_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

static gboolean battery_plugin_on_timeout(gpointer data)
{
	DBusConnection *connection;
	DBusError error;
	LibHalContext *ctx;
	const char *udi = "/org/freedesktop/Hal/devices/bme";
	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SYSTEM,&error);
	
	if ( dbus_error_is_set(&error) )
	{
		printf("Unable to connect to DBus: %s\n",error.message);
		dbus_error_free(&error);
		return 1;
	}
	
	ctx = libhal_ctx_new();
	
	if ( !libhal_ctx_set_dbus_connection(ctx, connection) )
	{
		printf("Error %s\n",error.message);
		dbus_error_free(&error);
		return 1;
	}
	
	if ( !libhal_ctx_init(ctx, &error) )
	{
		printf("Hal context initializing failure %s\n",error.message);
		return 1;
	}
	
	GtkWidget *widget = GTK_WIDGET(data);

	if (!widget->window)
		return TRUE;
	
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(widget);
	
	battery->percent = libhal_device_get_property_int(ctx, udi, "battery.charge_level.percentage", NULL);

	GdkRegion *region = gdk_drawable_get_clip_region(widget->window);
	if (!region)
		return TRUE;
	    
	gdk_window_invalidate_region(widget->window, region, TRUE);
	gdk_window_process_updates(widget->window, TRUE);
	gdk_region_destroy(region);

	return TRUE;
}


static void battery_plugin_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	requisition->width = 170;
	requisition->height = 102;
}


static void battery_plugin_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED(widget))
	{
		gdk_window_move_resize(
			widget->window,
			allocation->x, allocation->y,
			allocation->width, allocation->height
		);
	}
}

static void battery_plugin_init(BatteryPlugin* battery)
{
	battery->image_base   = cairo_image_surface_create_from_png("/usr/share/battery/batterybase.png");
	battery->image_green  = cairo_image_surface_create_from_png("/usr/share/battery/batteryhigh.png");
	battery->image_yellow = cairo_image_surface_create_from_png("/usr/share/battery/batterymed.png");
	battery->image_red    = cairo_image_surface_create_from_png("/usr/share/battery/batterylow.png");
	
	battery->timeout_handler = g_timeout_add(60000, battery_plugin_on_timeout, battery);
	battery_plugin_on_timeout(battery);
}

static void battery_plugin_realize(GtkWidget *widget)
{
	GdkScreen* screen = gtk_widget_get_screen(widget);
	gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
	gtk_widget_set_app_paintable(widget, TRUE);
	
	/*widget->allocation.width  = 170;
	widget->allocation.height = 102;*/

	GTK_WIDGET_CLASS(battery_plugin_parent_class)->realize(widget);
}

static gboolean battery_plugin_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(widget);

	cairo_t *cr;

	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));
	gdk_cairo_region(cr, event->region);
	cairo_clip(cr);

	cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
	cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.0);
	cairo_paint(cr);
	
	cairo_set_source_surface(cr, battery->image_base, 0, 0);
	cairo_paint(cr);

	float width = (134.0f / 100.0f) * battery->percent;
	
	if (battery->percent > 66)
		cairo_set_source_surface(cr, battery->image_green, 12, 12);
	else if (battery->percent < 33)
		cairo_set_source_surface(cr, battery->image_red, 12, 12);
	else
		cairo_set_source_surface(cr, battery->image_yellow, 12, 12);
	
	cairo_rectangle(cr, 12, 12, width, 74);
	cairo_clip(cr);
	cairo_paint(cr);

	/* Draw percent */
	cairo_reset_clip(cr);
	
	const char* percent_str = g_strdup_printf("%d %%", battery->percent);
	cairo_text_extents_t te;
	cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
	cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size (cr, 32.0f);
	cairo_text_extents (cr, percent_str, &te);
	cairo_move_to (cr, widget->allocation.width / 2 - te.width / 2 - te.x_bearing, widget->allocation.height / 2 - te.height / 2 - te.y_bearing);
	cairo_show_text (cr, percent_str);
	g_free(percent_str);

	cairo_destroy (cr);

	return GTK_WIDGET_CLASS(battery_plugin_parent_class)->expose_event(widget, event);
}

static void battery_plugin_dispose (GObject *object)
{
	G_OBJECT_CLASS(battery_plugin_parent_class)->dispose(object);
}

static void battery_plugin_finalize(GObject *object)
{
	//g_print("removed... destroying images...");
	
	BatteryPlugin* battery = BATTERY_HOME_PLUGIN(object);

	cairo_surface_destroy(battery->image_base);
	cairo_surface_destroy(battery->image_green);
	cairo_surface_destroy(battery->image_yellow);
	cairo_surface_destroy(battery->image_red);
	
	if (battery->timeout_handler)
	{
		g_source_remove(battery->timeout_handler);
	}

	G_OBJECT_CLASS(battery_plugin_parent_class)->finalize(object);
}

static void battery_plugin_class_init (BatteryPluginClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = battery_plugin_dispose;
	object_class->finalize = battery_plugin_finalize;

	widget_class->realize = battery_plugin_realize;
	widget_class->size_request = battery_plugin_size_request;
	widget_class->size_allocate = battery_plugin_size_allocate;
	widget_class->expose_event = battery_plugin_expose_event;
}

static void battery_plugin_class_finalize(BatteryPluginClass* klass)
{

}
