#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm


import re
import subprocess

import time

class DataSource(object):
    def __init__(self):
        self.providedValues = self.getProvidedValues()
        self.lastValues = {}
        for key in self.providedValues:
            self.markUpdated(key, None, 0)

    def markUpdated(self, key, value, timestamp):
        self.lastValues[key] = (value, timestamp)

    def timestampAdjust(self, type):
        return 0

class DataSourceInternal(DataSource):
    GRAPH_BREAK = 'internal.graph.break'

    def getProvidedValues(self):
        return {self.GRAPH_BREAK: 'none'}

    def read(self, type):
        """
        Type can be: 'periodical' or 'startup' 
        """
        if type == 'startup':
            return {self.GRAPH_BREAK: None}
        return {}

    def decideStoreableValues(self, available, changed):
        if available.has_key(self.GRAPH_BREAK):
            return {self.GRAPH_BREAK: None}
        return {}

    def timestampAdjust(self, type):
        if type == self.GRAPH_BREAK:
            return -1
        return 0

class DataSourceHal(DataSource):
    PERCENTAGE = 'hal.battery.charge_level.percentage'
    CHARGE = 'hal.battery.reporting.current'
    CHARGE_DESIGN = 'hal.battery.reporting.design'
    VOLTAGE = 'hal.battery.voltage.current'
    VOLTAGE_DESIGN = 'hal.battery.voltage.design'
    CHARGING = 'hal.battery.rechargeable.is_charging'
    DISCHARGING = 'hal.battery.rechargeable.is_discharging'

    def getProvidedValues(self):
        return {self.PERCENTAGE: '%',
                self.CHARGE: 'mAh',
                self.CHARGE_DESIGN: 'mAh',
                self.VOLTAGE: 'mV',
                self.VOLTAGE_DESIGN: 'mV',
                self.CHARGING: 'bool',
                self.DISCHARGING: 'bool'}

    def read(self, type):
        """
        Type can be: 'periodical' or 'startup' 
        """
        process = subprocess.Popen(['lshal', '-u',
                                    '/org/freedesktop/Hal/devices/bme'],
                                   stdout=subprocess.PIPE)

        output = process.stdout.readlines()
        process.wait()
        process.stdout.close()

        output = self.dictionarize(output)

        infos = ['battery.charge_level.percentage',
                 'battery.reporting.current',
                 'battery.voltage.current',
                 'battery.rechargeable.is_charging',
                 'battery.rechargeable.is_discharging']
        if type == 'periodical':
            pass
        elif type == 'startup':
            infos.append('battery.reporting.design')
            infos.append('battery.voltage.design')
        else:
            raise 'Unknown type'

        ret = {}

        for info in infos:
            if output.has_key(info):
                infoType = self.providedValues['hal.' + info]
                if infoType in ('%', 'mAh', 'mV'):
                    processedValue = int(output[info])
                elif infoType == 'bool':
                    if output[info] == 'true':
                        processedValue = True
                    else:
                        processedValue = False
                else:
                    processedValue = output[info]
                ret['hal.' + info] = processedValue
                    
        return ret

    # Decides which values should be stored.
    # May include more data than what is actually
    # reported as changed, if values are old.
    def decideStoreableValues(self, available, changed):
        changedAndAvailable = [k for k in changed if available.has_key(k)]

        ret = {}
        for k in changedAndAvailable:
            if self.lastValues[k][0] != available[k]:
                # We still only want to record the value if it has really
                # changed since our last observation. Because of our
                # 'greedy' data collection method, we sometimes have already
                # recorded the changed value before it is announced by a
                # DBUS signal.
                # Also note that most of the rest of this method handles
                # exceptions to the rule stated above.
                ret[k] = available[k]

        now = int(time.time())

        additional = []
        for k in available.keys():
            if k in (self.PERCENTAGE, self.CHARGE, self.VOLTAGE,
                     self.CHARGE_DESIGN, self.VOLTAGE_DESIGN):
                if self.lastValues[k][1] < now - 900 \
                        and self.lastValues[k][0] != available[k]:
                    additional.append(k)
            if k in (self.CHARGING, self.DISCHARGING):
                 if self.lastValues[k][0] != available[k]:
                    additional.append(k)

        for k in additional:
            ret[k] = available[k]

        try:
            if available[self.CHARGING] and not available[self.DISCHARGING]:
                # Don't return percentage and energy, they're not being
                # updated during charge...
                if (self.lastValues[self.DISCHARGING][0] == True):
                    # ..except if we just started charging. Then we want to record
                    # the last known values.
                    # (This should fix the charge that appears to begin too early)
                    for key in (self.PERCENTAGE, self.CHARGE):
                        if not ret.has_key(key) and available.has_key(key):
                            ret[key] = available[key]
                else:
                    # Don't record
                    for key in (self.PERCENTAGE, self.CHARGE):
                        # Remove from returned dict
                        ret.pop(key, None)
                        # Also mark it as updated with it's latest value, so we
                        # don't record it until it's properly recalculated at
                        # end of charge [bug 5235]
                        if available.has_key(key):
                            self.markUpdated(key, available[key], now)
        except KeyError:
            # For some reason, some of the keys wasn't available
            # (This should not happen)
            pass
        return ret 

    def dictionarize(self, lines):
        ret = {}
        regex = re.compile('(\S+) = (\S+)')
        for line in lines:
            match = regex.search(line)
            if match:
                ret[match.group(1)] = match.group(2)
        return ret

