#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm

import os

import gtk
import hildon

import pango

import time
import datetime

import gobject

import beye

import dbus
from dbus.mainloop.glib import DBusGMainLoop
DBusGMainLoop(set_as_default=True)

from beye.sliceset import SliceSet as SliceSet

PERCENTAGE = beye.DataSourceHal.PERCENTAGE
CHARGE = beye.DataSourceHal.CHARGE
VOLTAGE = beye.DataSourceHal.VOLTAGE

class MainWindow(hildon.StackableWindow):
    def __init__(self, dataStorage):
        hildon.StackableWindow.__init__(self)
        self.program = hildon.Program.get_instance()
        self.program.set_can_hibernate(True)

        self.dataStorage = dataStorage

        self.updateRequired = False

        self.connect("delete_event", lambda a, b: self.quit())
        self.connect("notify::is-topmost", self.focusChanged)
        self.program.add_window(self)

        self.layout = gtk.Fixed()
        self.add(self.layout)

        self.graph = Graph(self.dataStorage)
        self.layout.put(self.graph, 0, 0)

        gtk.set_application_name("battery-eye")
        self.setupMenu()

    def createLegendImage(self, color, size):
        p = gtk.gdk.Pixmap(None, size, size, 16)
        gc = p.new_gc(foreground=color)
        p.draw_rectangle(gc, True, 0, 0, size, size)
        return gtk.image_new_from_pixmap(p, None)      

    def setupMenu(self):
        self.menu = hildon.AppMenu()
        
        for type, label, default in ((PERCENTAGE, '%', 1), (CHARGE, 'mAh', 0), (VOLTAGE, 'mV', 0)):
            f = hildon.GtkToggleButton(gtk.HILDON_SIZE_AUTO)
            f.set_label(label)
            f.set_mode(False)
            f.set_active(bool(int(self.dataStorage.getConfig('display:' + type, default))))
            f.set_image(self.createLegendImage(self.graph.dataColors[type], 24))
            #f.connect("clicked", lambda button: self.changeDataset(button, type))
            f.connect("clicked", self.changeDataset, type)
            self.menu.add_filter(f)

        self.zoomButton = hildon.PickerButton(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.zoomButton.set_title('Zoom')
        self.zoomSelector = hildon.TouchSelector(text = True)
        for text in ('1 day', '2 days', '3 days', '5 days', '7 days'):
            self.zoomSelector.append_text(text)
        self.zoomSelector.set_active(0, {25:0, 25*2:1, 25*3:2, 25*5:3, 25*7:4}[int(self.dataStorage.getConfig('display:visibleHours', 25))])
        self.zoomSelector.connect("changed", self.changeZoom)
        self.zoomButton.set_selector(self.zoomSelector)

        self.menu.append(self.zoomButton)

        self.deleteButton = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.deleteButton.set_label('Delete data...')
        self.deleteButton.connect("clicked", self.deleteData)

        self.menu.append(self.deleteButton)

        #self.incUpd = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        #self.incUpd.set_label('Incremental upd.')
        #self.incUpd.connect("clicked", lambda x: self.graph.updateIncremental())

        #self.menu.append(self.incUpd)

        self.aboutButton = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.aboutButton.set_label('About')
        self.aboutButton.connect("clicked", lambda x: AboutDialog(self))

        self.menu.append(self.aboutButton)

        self.menu.show_all()
        self.set_app_menu(self.menu)

    def changeDataset(self, button, dataset):
        if button.get_active():
            self.graph.enableDataset(dataset)
        else:
            self.graph.disableDataset(dataset)
        return True

    def focusChanged(self, window, property):
        if self.get_is_topmost():
            print "\n%s Got focus" % str(datetime.datetime.now())
            if self.updateRequired:
                self.queueUpdate()
        else:
            print "\n%s Lost focus" % str(datetime.datetime.now())
        return False

    def deleteData(self, widget):
        dlg = gtk.Dialog()
        dlg.set_title("Delete all graph data older than")

        selector = hildon.TouchSelector(text = True)
        selector.append_text('3 days')
        selector.append_text('7 days')
        selector.append_text('14 days')
        selector.append_text('28 days')
        selector.append_text('Delete all')
        selector.connect("changed", self.doDelete, dlg)
        dlg.vbox.pack_start(selector, True, True)
        dlg.vbox.set_size_request(300, 300)
        dlg.set_transient_for(self)

        dlg.show_all()
        dlg.run()
        dlg.destroy()

    def doDelete(self, selector, user_data, parent):
        if selector.get_current_text() == 'Delete all':
            msg = 'All graph data will be deleted. Continue?'
        else:
            msg = 'Graph data older than %s will be deleted. Continue?' % selector.get_current_text()
        note = hildon.hildon_note_new_confirmation(parent, msg)
        response = note.run()
        if response == gtk.RESPONSE_OK:
            # Delete
            parent.response(gtk.RESPONSE_OK)
            now = int(time.time())
            values = {'3 days': now - 3 * 86400,
                      '7 days': now - 7 * 86400,
                      '14 days': now - 14 * 86400,
                      '28 days': now - 28 * 86400,
                      'Delete all': now + 1}
            self.dataStorage.deleteObservations(values[selector.get_current_text()])
            self.graph.update()
        note.destroy()
       
    def changeZoom(self, selector, user_data):
        if selector.get_current_text() == None:
            return
        values = {'1 day': 25, '2 days': 25*2, '3 days': 25*3, '5 days': 25*5, '7 days': 25*7}
        self.graph.changeVisibleHours(values[selector.get_current_text()])

    def run(self):
        self.show_all()
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        gobject.idle_add(self.firstIdleCb)
        bus = dbus.SystemBus()
        bus.add_signal_receiver(self.dbusHandler, path='/org/freedesktop/Hal/devices/bme')
        try:
            gtk.main()
        except KeyboardInterrupt:
            print ""
            print "Ctrl-C"

    def firstIdleCb(self):
        self.doUpdate()
        return False

    def queueUpdate(self):
        self.updateRequired = False
        try:
            graphEnd = self.graph.xMax
        except:
            print "Not initialized yet..."
            return
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        now = int(time.time())
        if graphEnd < now + 3600:
            print "Queuing full update"
            gobject.idle_add(self.doUpdate)
        else:
            print "Queuing incremental update"
            gobject.idle_add(self.doUpdateInc)
        
    def doUpdateInc(self):
        self.graph.updateIncremental()
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        return False

    def doUpdate(self):
        self.graph.update()
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        return False

    def dbusHandler(self, *args, **kwargs):
        print "\n%s Values changed" % str(datetime.datetime.now())

        interesting = (PERCENTAGE, VOLTAGE, CHARGE,
                       beye.DataSourceHal.CHARGING,
                       beye.DataSourceHal.DISCHARGING)

        interested = False
        for change in args[1]:
            if 'hal.' + str(change[0]) in interesting:
                interested = True
                break

        if not interested:
            print "Not interested"
            return

        if self.get_is_topmost():
            print "I'm topmost, updating now"
            self.queueUpdate()
        else:
            print "I'm not topmost, updating later"
            self.updateRequired = True
        
    def quit(self):
        # set_app_menu(None) gets rid of glib assertion errors on exit
        self.set_app_menu(None)
        gtk.main_quit()



class DrawAreaInfo(object):
    def __init__(self, canvasSize, drawAreaSize, offset, windowSize):
        self.canvasSize = canvasSize
        self.drawAreaSize = drawAreaSize
        self.offset = offset
        self.windowSize = windowSize

    def canvasAsRect(self):
        return gtk.gdk.Rectangle(0, 0, self.canvasSize[0], self.canvasSize[1])

    def drawAreaAsRect(self):
        return gtk.gdk.Rectangle(self.offset[0], self.offset[1], self.drawAreaSize[0], self.drawAreaSize[1])

class EraseableBuffer(object):
    def __init__(self, pixmap):
        self.pixmap = pixmap
        self.backbuffer = gtk.gdk.Pixmap(None, pixmap.get_size()[0], pixmap.get_size()[1], pixmap.get_depth())
        self.lastDrawParams = None # (x, y, w, h)

    def draw(self, gc, dst, x, y, w, h):
        if w == -1:
            w = self.pixmap.get_size()[0]
        if h == -1:
            h = self.pixmap.get_size()[1]
        self.backbuffer.draw_drawable(gc, dst,
                                      x, y,
                                      0, 0,
                                      w, h)
        dst.draw_drawable(gc, self.pixmap,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = (x, y, w, h)
        return gtk.gdk.Rectangle(x, y, w, h)

    def undraw(self, gc, dst):
        if self.lastDrawParams == None:
            return
        x, y, w, h = self.lastDrawParams
        dst.draw_drawable(gc, self.backbuffer,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = None
        return gtk.gdk.Rectangle(x, y, w, h)

    def get_size(self):
        return self.pixmap.get_size()

class Graph(hildon.PannableArea):
    def __init__(self, dataStorage):
        hildon.PannableArea.__init__(self)
        self.dataStorage = dataStorage
        self.graphWidth = 800
        self.graphHeight = 424
        self.set_property('mov-mode', hildon.MOVEMENT_MODE_HORIZ)
        self.set_property('vscrollbar-policy', gtk.POLICY_NEVER)
        self.set_size_request(self.graphWidth, self.graphHeight)
        self.area = gtk.DrawingArea()
        self.area.set_double_buffered(False)
        self.set_double_buffered(False)
        self.add_with_viewport(self.area)
        self.pixmap = None

        self.area.connect("expose-event", self.redrawGraph)

        self.get_hadjustment().connect("value-changed", self.moveLabels)

        self.bgColor = self.area.get_colormap().alloc_color(0, 0, 0)
        self.gridColor = self.area.get_colormap().alloc_color(40*256, 40*256, 45*256)
        self.labelsColor = self.area.get_colormap().alloc_color(120*256, 120*256, 120*256)

        self.dataColors = {PERCENTAGE: self.area.get_colormap().alloc_color(0, 255*256, 0),
                           CHARGE: self.area.get_colormap().alloc_color(0, 100*256, 20*256),
                           VOLTAGE: self.area.get_colormap().alloc_color(60*256, 60*256, 100*256)}

        self.chargingColor = self.area.get_colormap().alloc_color(35*256, 35*256, 0)
        self.fullColor = self.area.get_colormap().alloc_color(0, 50*256, 0)

        self.datasets = {PERCENTAGE:
                            bool(int(self.dataStorage.getConfig('display:' + PERCENTAGE, 1))),
                         CHARGE:
                            bool(int(self.dataStorage.getConfig('display:' + CHARGE, 0))),
                         VOLTAGE:
                            bool(int(self.dataStorage.getConfig('display:' + VOLTAGE, 0)))}
        
        self.dataRenderers = {VOLTAGE: [],
                              CHARGE: [],
                              PERCENTAGE: []}

        self.unrenderedArea = SliceSet()

        self.chargeInfo = []

        self.visibleHours = int(self.dataStorage.getConfig('display:visibleHours', 25))
        self.maxVisibleHours = 30*24

        self.gridType = self.dataStorage.getConfig('display:gridType', PERCENTAGE)

        self.requestJump = None

    def update(self):
        self.dataRenderers = {VOLTAGE: [],
                              CHARGE: [],
                              PERCENTAGE: []}
        self.chargeInfo = []
        self.updateGrid()
        self.updateData()
        #self.renderData(self.drawAreaInfo.canvasAsRect())
        self.requestJump = self.drawAreaInfo.canvasSize[0]-1
        self.area.queue_draw()
        #gobject.idle_add(self.moveLabels)

    def updateIncremental(self):
        self.updateDataIncremental()
        self.unrenderedArea = SliceSet([(0, self.drawAreaInfo.canvasSize[0] - 1)])
        self.area.queue_draw()
        #self.renderData(self.drawAreaInfo.canvasAsRect())
        #gobject.idle_add(self.moveLabels)

    def enableDataset(self, dataset):
        #print "Enable: %s" % dataset
        self.datasets[dataset] = True
        self.dataStorage.setConfig('display:' + dataset, 1)
        self.updateData()
        self.changeYLabels(dataset)
        self.unrenderedArea = SliceSet([(0, self.drawAreaInfo.canvasSize[0] - 1)])
        self.area.queue_draw()
        #self.renderData(self.drawAreaInfo.canvasAsRect())
        #gobject.idle_add(self.moveLabels)

    def disableDataset(self, dataset):
        #print "Disable: %s" % dataset
        self.datasets[dataset] = False
        self.dataStorage.setConfig('display:' + dataset, 0)
        if self.gridType == dataset:
            self.changeYLabels(self.getBestYLabelType())
        self.unrenderedArea = SliceSet([(0, self.drawAreaInfo.canvasSize[0] - 1)])
        self.area.queue_draw()
        #self.renderData(self.drawAreaInfo.canvasAsRect())
        #gobject.idle_add(self.moveLabels)

    def changeVisibleHours(self, visibleHours):
        #print "Visible Hours: %d" % visibleHours
        self.dataStorage.setConfig('display:visibleHours', visibleHours)
        self.visibleHours = visibleHours
        self.updateGrid()
        timeBounds = (self.xMin, self.xMax)
        for t,v in self.dataRenderers.iteritems():
            for r in v:
                r.changeScale(self.drawAreaInfo, timeBounds)
        for r in self.chargeInfo:
            r.changeScale(self.drawAreaInfo, timeBounds)
        #self.renderData(self.drawAreaInfo.canvasAsRect())
        self.requestJump = self.drawAreaInfo.canvasSize[0]-1
        self.area.queue_draw()
        #gobject.idle_add(self.moveLabels)
        

    def getBestYLabelType(self):
        for s in (PERCENTAGE, CHARGE, VOLTAGE):
            if self.datasets[s]:
                return s
        return PERCENTAGE

    def getYLabelInfo(self, type):
        if type == CHARGE:
            yMin = 0
            yMax = self.dataStorage.getMaxCharge()
            if yMax == None:
                yMax = 1320
            yLabelsInterval = 200
            unit = 'mAh'
        elif type == VOLTAGE:
            yMin = 3500
            yMax = 4200
            yLabelsInterval = 100
            unit = 'mV'
        elif PERCENTAGE:
            yMin = 0
            yMax = 100
            yLabelsInterval = 20
            unit = '%'

        yLabels = [str(v) + unit for v in range(yMin, yMax, yLabelsInterval)] + [str(yMax) + unit]
        return ((yMin, yMax), yLabelsInterval, yLabels)
     
    def changeYLabels(self, type):
        self.grid.changeYGrid(*(self.getYLabelInfo(type) + (self.dataColors[type], )))
        self.gridType = type
        self.dataStorage.setConfig('display:gridType', type)

    def updateGrid(self):
        try:
            if not self.datasets[self.gridType]:
                labelsType = self.getBestYLabelType()
            else:
                labelsType = self.gridType
        except KeyError:
            labelsType = self.getBestYLabelType()
        valueBounds, yLabelsInterval, yLabels = self.getYLabelInfo(labelsType)

        if self.visibleHours < 48:
            self.xGridInterval = 1
            self.xLabelInterval = 2
        elif self.visibleHours < 100:
            self.xGridInterval = 2
            self.xLabelInterval = 6
        else:
            self.xGridInterval = 6
            self.xLabelInterval = 24

        now = int(time.time())

        self.xMax = now + 4*3600

        self.xMin = self.dataStorage.getOldestObservationTime()
        if self.xMin == None:
            self.xMin = self.xMax - self.visibleHours*3600
        else:
            self.xMin = max(self.xMin, self.xMax - 3600 * self.maxVisibleHours)
            self.xMin = min(self.xMin, self.xMax - 3600 * self.visibleHours)
        areaWidth = max(int((self.xMax - self.xMin) / 3600 * (self.graphWidth/float(self.visibleHours))), self.graphWidth)


        self.drawAreaInfo = DrawAreaInfo((areaWidth, self.graphHeight),
                                         (areaWidth, self.graphHeight-40),
                                         (0, 14),
                                         (self.graphWidth, self.graphHeight-10))

        timeBounds = (self.xMin, self.xMax)
        self.grid = GridRenderer(self.drawAreaInfo,
                                 timeBounds, self.xGridInterval,
                                 valueBounds, yLabelsInterval,
                                 self.xLabelInterval, self.labelsColor,
                                 yLabels, self.dataColors[labelsType],
                                 self.area.create_pango_context())

        self.gridType = labelsType
        
        self.pixmap = gtk.gdk.Pixmap(None, areaWidth, self.graphHeight, 16)
        self.area.set_size_request(areaWidth, self.graphHeight)
        self.unrenderedArea = SliceSet([(0, self.drawAreaInfo.canvasSize[0] - 1)])
        
    def updateData(self):
        timeBounds = (self.xMin, self.xMax)

        if not len(self.chargeInfo):
            self.chargeInfo = [DataRendererStatus(segment,
                                                  self.drawAreaInfo,
                                                  timeBounds) for segment in self.getChargeInfo(self.xMin, self.xMax)]

        for type in [t for t in self.datasets if self.datasets[t] and not len(self.dataRenderers[t])]:
            valueBounds = self.getYLabelInfo(type)[0]
            #print "Get data %s" % type
            rawData = self.dataStorage.getObservationsWithBreaks(type, self.xMin, self.xMax)
            self.dataRenderers[type] = [DataRendererValue(segment,
                                                          self.drawAreaInfo,
                                                          timeBounds,
                                                          valueBounds)
                                        for segment in rawData if len(rawData)]

    def updateDataIncremental(self):
        timeBounds = (self.xMin, self.xMax)

        try:
            intervalStart = self.chargeInfo[-1].getFirstObservationTime()
        except IndexError:
            intervalStart = None
        if intervalStart == None:
            intervalStart = self.xMin

        data = self.getChargeInfo(intervalStart, self.xMax)

        if len(self.chargeInfo) and len(data):
            self.chargeInfo[-1] = DataRendererStatus(data[0], self.drawAreaInfo, timeBounds)
            data = data[1:]

        for segment in data:
            self.chargeInfo.append(DataRendererStatus(segment, self.drawAreaInfo, timeBounds))

        for type in [t for t in self.datasets if self.datasets[t]]:
            try:
                intervalStart = self.dataRenderers[type][-1].getLastObservationTime() + 1
            except:
                intervalStart = None
            if intervalStart == None:
                intervalStart = self.xMin

            valueBounds = self.getYLabelInfo(type)[0]
            rawData = self.dataStorage.getObservationsWithBreaks(type, intervalStart, self.xMax)
            
            if len(self.dataRenderers[type]) and len(rawData):
                self.dataRenderers[type][-1].addPoints(rawData[0])
                rawData = rawData[1:]

            for segment in [s for s in rawData if len(rawData)]:
                self.dataRenderers[type].append(DataRendererValue(segment, self.drawAreaInfo, timeBounds, valueBounds))

    def renderData(self, cliprect):
        gc = self.pixmap.new_gc()
        gc.set_clip_rectangle(cliprect)
        gc.foreground = self.bgColor

        area = self.grid.clearYLabels(self.pixmap)
        if area:
            self.area.queue_draw_area(area.x, area.y, area.width, area.height)

        self.pixmap.draw_rectangle(gc, True,
                                   0, 0,
                                   self.drawAreaInfo.canvasSize[0],
                                   self.drawAreaInfo.canvasSize[1])
        
        for d in self.chargeInfo:
            d.render(self.pixmap, {(True, True): self.fullColor, (True, False): self.chargingColor},
                     cliprect)

        self.grid.render(self.pixmap, self.gridColor, cliprect)

        for type, renderers in ((t, self.dataRenderers[t])
                                for t in (VOLTAGE,
                                          CHARGE,
                                          PERCENTAGE) if self.datasets[t]):
            #print "%s: %d sets" % (type, len(renderers))
            c = self.dataColors[type]
            for d in renderers:
                d.render(self.pixmap, c, cliprect)

        self.area.queue_draw()
        self.moveLabels(False)
            
    def redrawGraph(self, widget, event):
        if not self.pixmap:
            # We're not currently initialized, ignore.
            return
            
        if (self.requestJump != None):
            self.jump_to(self.requestJump, 0)
            self.requestJump = None
            self.area.queue_draw()
            return
        
        self.gc = widget.style.fg_gc[gtk.STATE_NORMAL]
            
        area = event.area

#        print "Redraw (%d, %d)->(%d, %d)" % (area.x, area.y, area.x + area.width, area.y + area.height)

        edges = ()
        if self.unrenderedArea.intersection((area.x, area.x + area.width - 1)):
            for i in self.unrenderedArea.intersection((area.x - self.drawAreaInfo.windowSize[0],
                                                       area.x + area.width + self.drawAreaInfo.windowSize[0])):
                edges += i

        if len(edges):
            x = min(edges)
            w = max(edges) - min(edges) + 1
            r = gtk.gdk.Rectangle(x, 0, w, self.drawAreaInfo.canvasSize[1])
            #print "Render (%d, %d)->(%d, %d)" % (r.x, r.y, r.x + r.width, r.y + r.height)
            self.unrenderedArea.subtract((x, max(edges)))
            gobject.idle_add(self.renderData, r)
        else:
            widget.window.draw_drawable(self.gc, self.pixmap,
                                        area.x, area.y,
                                        area.x, area.y,
                                        area.width, area.height)

    def moveLabels(self, adjustment=False):
        #print "Draw Y labels: %d" % int(self.get_hadjustment().get_value())
        area = self.grid.drawYLabels(self.pixmap, int(self.get_hadjustment().get_value())) 
        self.area.queue_draw_area(area.x, area.y, area.width, area.height)

    def getChargeInfo(self, intervalStart, intervalEnd):
        rawData = self.dataStorage.getObservations((beye.DataSourceHal.CHARGING,
                                                    beye.DataSourceHal.DISCHARGING,
                                                    beye.DataSourceInternal.GRAPH_BREAK),
                                                   intervalStart, intervalEnd)
        if len(rawData) == 0:
            return []
        segment = []
        segments = [segment]
        for point in rawData:
            segment.append(point)
            if point[2] == beye.DataSourceInternal.GRAPH_BREAK:
                segment = []
                segments.append(segment)

        # Beware: while this assumption is close enough of
        # truth for now, it might not always be.
        now = intervalEnd

        # Some serious list processing voodoo follows.

        statuses = [] #[[((charging, discharging), start, end), (...), ...], [...], ...]
        for segment in (x for x in segments if len(x) > 0):
            intervalStart = segment[0][0]
            if segment[-1][2] == beye.DataSourceInternal.GRAPH_BREAK:
                intervalEnd = self.dataStorage.getPreviousObservationTime(segment[-1][0])
            else:
                intervalEnd = self.dataStorage.getPreviousObservationTime(now)
            if intervalEnd == None or intervalStart >= intervalEnd:
                continue

            statusList = []
            lastStatus = ((False, True), intervalStart, intervalStart)
            for point in segment:
                if point[2] == beye.DataSourceHal.CHARGING:
                    if point[1] != lastStatus[0][0]:
                        if point[0] > lastStatus[2]:
                            statusList.append((lastStatus[0], lastStatus[1], point[0]))
                        lastStatus = ((bool(point[1]), lastStatus[0][1]), point[0], point[0])
                    else:
                        lastStatus = (lastStatus[0], lastStatus[1], point[0])
                elif point[2] == beye.DataSourceHal.DISCHARGING:
                    if point[1] != lastStatus[0][1]:
                        if point[0] > lastStatus[2]:
                            statusList.append((lastStatus[0], lastStatus[1], point[0]))
                        lastStatus = ((lastStatus[0][0], bool(point[1])), point[0], point[0])
                    else:
                        lastStatus = (lastStatus[0], lastStatus[1], point[0])
            statusList.append((lastStatus[0], lastStatus[1], intervalEnd))
            statuses.append([x for x in statusList if x[1] < x[2]])
        return [x for x in statuses if len(x) > 0]

#        for segment in statuses:
#            print ""
#            for status in segment:
#                if status[0] == (True, False):
#                    descr = "Charging"
#                elif status[0] == (True, True):
#                    descr = "Full"
#                elif status[0] == (False, True):
#                    descr = "Discharging"
#                else:
#                    descr = "W00t? No battery?"
#                print "%s (%d minutes) %s - %s" % (descr, (status[2] - status[1]) / 60,
#                                                   datetime.datetime.fromtimestamp(status[1], beye.utc).astimezone(beye.local),
#                                                   datetime.datetime.fromtimestamp(status[2], beye.utc).astimezone(beye.local))

  
class AboutDialog(gtk.AboutDialog):
    def __init__(self, parent):
        gtk.Dialog.__init__(self)
        self.set_name("battery-eye")
        self.set_logo_icon_name("battery-eye")

        self.set_copyright(beye.__copyright__)
        self.set_version(beye.__version__)
        self.set_website(beye.__url__)

        stat = os.stat(beye.dbFilePath)
        self.set_comments('Database size: %dkB, %d records' % (stat[6] / 1024, parent.dataStorage.countObservations()))

        #gtk.about_dialog_set_url_hook(lambda dialog, link, user_data: webbrowser.open(link), None)
        self.set_license('''The battery-eye software is distributed under GPL version 3. For more information, see
<http://www.gnu.org/licenses/>

The battery-eye icon is a derivative based on original work by
rutty <http://www.flickr.com/photos/rutty/438995413>, and is distributed under a Creative Commons license:
<http://creativecommons.org/licenses/by-nc-sa/2.0/>

battery-eye is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
''')
        self.set_wrap_license(True)
        
        self.show_all()
        self.run()
        self.destroy()

class GridRenderer(object):
    def __init__(self, areaInfo, timeBounds, xInterval, valueBounds, yInterval,
                 xLabelInterval, xLabelsColor, yLabels, yLabelsColor, pangoContext):
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.valueBounds = valueBounds
        self.xInterval = xInterval
        self.yInterval = yInterval
        self.xLabelInterval = xLabelInterval
        self.hours = getHoursInInterval(timeBounds[0], timeBounds[1], xInterval)
        self.yGrid = self.yGridSegments(areaInfo, valueBounds, yInterval)
        self.xGrid = self.xGridSegments(areaInfo, timeBounds)
        self.xLabelsColor = xLabelsColor
        self.yLabelsColor = yLabelsColor
        self.xLabelPixmaps = {}
        self.pangoContext = pangoContext
        self.yLabels = self.createYLabels(yLabels, yLabelsColor)
        self.xLabels = self.createXLabels(timeBounds, xLabelInterval, xLabelsColor)

    def changeYGrid(self, bounds, interval, labels, color):
        self.valueBounds = bounds
        self.yInterval = interval
        self.yLabelsColor = color
        self.yGrid = self.yGridSegments(self.areaInfo, bounds, interval)
        self.yLabels = self.createYLabels(labels, color)

    def yGridSegments(self, areaInfo, yBounds, yInterval):
        y = []
        j = yBounds[0]
        while j < yBounds[1]:
            y.append(j)
            j += yInterval
        y.append(yBounds[1])
        return self.createSegments('y', y, areaInfo, yBounds)

    def xGridSegments(self, areaInfo, xBounds):
        points = [time.mktime(h.timetuple()) for h in self.hours]
        return self.createSegments('x', points, areaInfo, xBounds)

    def createXLabels(self, xBounds, modulo, color):
        pContext = self.pangoContext
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        
        timedata = zip(self.hours, self.xGrid)
        
        xLabels = []
        for hour in ((h[0], h[1][0]) for h in timedata if h[0].hour % modulo == 0):
            if hour[0].hour == 0:
                text = hour[0].strftime('%d. %b')
            else:
                text = hour[0].strftime('%H:%M')
            if self.xLabelPixmaps.has_key(text):
                pixmap = self.xLabelPixmaps[text]
            else:
                layout = pango.Layout(pContext)
                layout.set_text(text)
                pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                              layout.get_pixel_size()[1], 16)
                gc = pixmap.new_gc()
                pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
                gc.foreground = color
                pixmap.draw_layout(gc, 0, 0, layout)
                self.xLabelPixmaps[text] = pixmap
            xLabels.append((hour[1], pixmap))
        return xLabels

    def drawXLabels(self, gc, drawable, labels):
        y = self.areaInfo.offset[1] + self.areaInfo.drawAreaSize[1] + 1
        for x, pixmap, in labels:
            drawable.draw_drawable(gc, pixmap,
                                   0, 0,
                                   x - int(pixmap.get_size()[0]/2), y,
                                   -1, -1)

    def createYLabels(self, labels, color):
        pContext = self.pangoContext
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        labelPixmaps = []
        for label in labels:
            layout = pango.Layout(pContext)
            layout.set_text(label)
            pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                          layout.get_pixel_size()[1], 16)
            gc = pixmap.new_gc()
            pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
            gc.foreground = color
            pixmap.draw_layout(gc, 0, 0, layout)
            labelPixmaps.append(EraseableBuffer(pixmap))
        return labelPixmaps

    def unionRects(self, rects):
        r = None
        for rect in [r1 for r1 in rects if r1 != None]:
            if r == None:
                r = rect
            else:
                r = r.union(rect)
        return r

    def clearYLabels(self, drawable):
        invalidated = []
        gc = drawable.new_gc()
        for pixmap in self.yLabels:
            invalidated.append(pixmap.undraw(gc, drawable))
        return self.unionRects(invalidated)

    # Returns a gtk.gdk.Rectangle describing the area that is invalided
    # by draws.
    def drawYLabels(self, drawable, xPos):
        invalidated = []
        gc = drawable.new_gc()
        for pixmap, position in zip(self.yLabels, self.yGrid):
            invalidated.append(pixmap.undraw(gc, drawable))
            invalidated.append(pixmap.draw(gc, drawable,
                               self.areaInfo.windowSize[0] - pixmap.get_size()[0] + xPos, position[1]-15,
                               -1, 15))
        return self.unionRects(invalidated)
            
    def createSegments(self, variable, points, areaInfo, bounds):
        if len(points) == 0:
            return []
        distance = bounds[1] - bounds[0]
        if distance <= 0:
            # Let's not divide by zero
            return []

        if variable == 'x':
            scale = float(areaInfo.drawAreaSize[0]-1) / distance
            offX = self.areaInfo.offset[0]
            y0 = areaInfo.offset[1] + 1
            y1 = areaInfo.offset[1] + areaInfo.drawAreaSize[1] - 1
            return [(p, y0, p, y1) for p in
                        (int(scale * (p - bounds[0])) + offX for p in points)]
        else:
            scale = -float(areaInfo.drawAreaSize[1]-1) / distance
            dispY = self.areaInfo.offset[1] + areaInfo.drawAreaSize[1]
            x0 = areaInfo.offset[0]
            x1 = areaInfo.offset[1] + areaInfo.drawAreaSize[0] - 1
            return [(x0, p, x1, p) for p in
                        (int(scale * (p - bounds[0])) + dispY for p in points)]

    def render(self, drawable, gridColor, cliprect):
        gc = drawable.new_gc(foreground=gridColor, line_width=1, function=gtk.gdk.XOR)
        gc.set_clip_rectangle(cliprect)
        drawable.draw_segments(gc, self.yGrid)
        drawable.draw_segments(gc, self.xGrid)
        self.drawXLabels(gc, drawable, self.xLabels)
                            
class DataRendererValue(object):
    def __init__(self, rawData, areaInfo, timeBounds, valueBounds):
        self.rawData = rawData
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.valueBounds = valueBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds, self.valueBounds)
        self.coordBounds = (self.scaledData[0][0]-1, self.scaledData[-1][0]+1)

    def getLastObservationTime(self):
        if len(self.rawData):
            return self.rawData[-1][0]
        return None

    def addPoints(self, points):
        self.rawData[len(self.rawData):] = points
        self.scaledData[len(self.scaledData):] = self.rescaleData(points, self.areaInfo, self.timeBounds, self.valueBounds)
        self.coordBounds = (self.scaledData[0][0]-1, self.scaledData[-1][0]+1)

    def changeScale(self, areaInfo, timeBounds, valueBounds=None):
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        if valueBounds:
            self.valueBounds = valueBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds, self.valueBounds)

    def rescaleData(self, points, areaInfo, xBounds, yBounds):
        if len(points) == 0:
            return []
        xDistance = xBounds[1] - xBounds[0]
        yDistance = yBounds[1] - yBounds[0]
        if xDistance <= 0 or yDistance <= 0:
            # Let's not divide by zero
            return []

        scaleX = float(areaInfo.drawAreaSize[0]-1) / xDistance
        scaleY = -float(areaInfo.drawAreaSize[1]-1) / yDistance

        offX = self.areaInfo.offset[0]
        dispY = self.areaInfo.offset[1] + areaInfo.drawAreaSize[1]

        return [(int(scaleX * (p[0] - xBounds[0])) + offX,
                 int(scaleY * (p[1] - yBounds[0])) + dispY) for p in points]

    def render(self, drawable, color, cliprect):
        if max(cliprect.x, self.coordBounds[0]) > min(cliprect.x + cliprect.width - 1, self.coordBounds[1]):
            # Nothing to see here, move along.
            return
        gc = drawable.new_gc(foreground=color, line_width=1, join_style=gtk.gdk.JOIN_ROUND)
        gc.set_clip_rectangle(cliprect)
        color2 = drawable.get_colormap().alloc_color(int(color.red*0.5),
                                                     int(color.green*0.5),
                                                     int(color.blue*0.5))
        gc2 = drawable.new_gc(foreground=color2, line_width=2, join_style=gtk.gdk.JOIN_ROUND)
        gc2.set_clip_rectangle(cliprect)

        if len(self.scaledData) == 1:
            drawable.draw_points(gc2, self.scaledData)
            drawable.draw_points(gc, self.scaledData)
        elif len(self.scaledData) > 1:
            drawable.draw_lines(gc2, self.scaledData)
            drawable.draw_lines(gc, self.scaledData)

class DataRendererStatus(object):
    def __init__(self, rawData, areaInfo, timeBounds):
        self.rawData = rawData
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds)
        self.coordBounds = (self.scaledData[0][1], self.scaledData[-1][2])

    def getFirstObservationTime(self):
        if not len(self.rawData):
            return None
        return self.rawData[0][1]

    def changeScale(self, areaInfo, timeBounds):
        self.areaInfo = areaInfo
        self.timeBounds = timeBounds
        self.scaledData = self.rescaleData(self.rawData, self.areaInfo, self.timeBounds)

    def rescaleData(self, statuses, areaInfo, xBounds):
        if len(statuses) == 0:
            return []
        xDistance = xBounds[1] - xBounds[0]
        if xDistance <= 0:
            # Let's not divide by zero
            return []

        scaleX = float(areaInfo.drawAreaSize[0]-1) / xDistance

        offX = self.areaInfo.offset[0]

        return [(s[0],
                 int(scaleX * (s[1] - xBounds[0])) + offX,
                 int(scaleX * (s[2] - xBounds[0])) + offX) for s in statuses]

    def render(self, drawable, colorMap, cliprect):
        if max(cliprect.x, self.coordBounds[0]) > min(cliprect.x + cliprect.width - 1, self.coordBounds[1]):
            # Nothing to see here, move along.
            return
        #print "render -> %d - %d" % self.coordBounds
        for status in self.scaledData:
            if (not colorMap.has_key(status[0])) or colorMap[status[0]] == None:
                continue
            gc = drawable.new_gc(foreground=colorMap[status[0]])
            gc.set_clip_rectangle(cliprect)
            drawable.draw_rectangle(gc, True,
                                    status[1], self.areaInfo.offset[1],
                                    status[2] - status[1], self.areaInfo.offset[1] + self.areaInfo.drawAreaSize[1] - 14)


def getHoursInInterval(start, end, modulo):
    hour = datetime.timedelta(0, 0, 0, 0, 0, 1)
    interval = datetime.timedelta(0, 0, 0, 0, 0, modulo)

    endDt = nextHour(datetime.datetime.fromtimestamp(end, beye.utc).astimezone(beye.local))
    startDt = nextHour(datetime.datetime.fromtimestamp(start, beye.utc).astimezone(beye.local))
    while startDt.hour % modulo != 0:
        startDt += hour
        if endDt < startDt:
            return []
   
    current = startDt
    ret = []
    while current < endDt:
        ret.append(current)
        current += interval
    return ret

def nextHour(dt):
    deltaBack = datetime.timedelta(0, dt.second, dt.microsecond, 0, dt.minute)
    deltaForward = datetime.timedelta(0, 0, 0, 0, 0, 1)
    return dt + deltaForward - deltaBack


