#include "ohms.h"
#include "utilities.h"

#include "ui_ohms.h"

#include <QtCore/QCoreApplication>
#include <QShortcut>
#include <QString>
#include <QLineEdit>
#include <QObject>
#include <QShortcut>

#include <math.h>
#include <algorithm>  // std::swap

//Static QStrings
static const QString micro(QChar(0x00B5));   // Unicode 'micro' character
static const QString Volt("V");
static const QString Amp("A");
static const QString Watt("W");
static const QString Ohm(QChar(0x2126));     // Unicode 'omega' character
static const QString squared(QChar(0x00B2)); // Unicode 'superscript two' character
static const QString root(QChar(0x221A));    // Unicode 'square root' character
static const QString inf(QChar(0x221E));     // Unicode for Infinity

Ohms::Ohms(QWidget *parent) :
        QDialog(parent),
        ui(new Ui::Ohms)
{
    ui->setupUi(this);

    //Window Text
    this->setWindowTitle("Ohm's Law");
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    this->setWindowFlags(Qt::Window);

    ui->comboUnit1->setCurrentIndex(0);


    // For Connecting Keyboard Buttons
          connectSignals();
}

Ohms::~Ohms()
{
    delete ui;
}

static void initMultipliersVoltage(QComboBox *qcb)
{
    qcb->clear();
    qcb->addItem(micro + Volt, 1e-6);
    qcb->addItem("mV", 1e-3);
    qcb->addItem("V", 1e0);
    qcb->addItem("kV", 1e3);
    qcb->addItem("MV", 1e3);
    qcb->setCurrentIndex(2);
}

static void initMultipliersCurrent(QComboBox *qcb)
{
    qcb->clear();
    qcb->addItem("pA", 1e-12);
    qcb->addItem("nA", 1e-9);
    qcb->addItem(micro + Amp, 1e-6);
    qcb->addItem("mA", 1e-3);
    qcb->addItem("A", 1e0);
    qcb->addItem("kA", 1e3);
    qcb->setCurrentIndex(4);
}

static void initMultipliersPower(QComboBox *qcb)
{
    qcb->clear();
    qcb->addItem("pW", 1e-12);
    qcb->addItem("nW", 1e-9);
    qcb->addItem(micro + Watt, 1e-6);
    qcb->addItem("mW", 1e-3);
    qcb->addItem("W", 1e0);
    qcb->addItem("kW", 1e3);
    qcb->addItem("MW", 1e6);
    qcb->addItem("GW", 1e9);
    qcb->setCurrentIndex(4);
}

static void initMultipliersResistance(QComboBox *qcb)
{
    qcb->clear();
    qcb->addItem("m" + Ohm, 1e-3);
    qcb->addItem(Ohm, 1e0);
    qcb->addItem("k" + Ohm, 1e3);
    qcb->addItem("M" + Ohm, 1e6);
    qcb->addItem("G" + Ohm, 1e9);
    qcb->setCurrentIndex(1);
}

// Keyboard Button Signals - remember to include QShortcut
void Ohms::connectSignals()
{
     connect(new QShortcut(QKeySequence(Qt::Key_Backspace), this), SIGNAL(activated()), this, SLOT(close()));
}


void Ohms::on_comboUnit1_currentIndexChanged(int index)
{
    ui->comboUnit2->clear();

    for (int i = 0; i < ui->comboUnit1->count(); ++i)
        if (i != index)
            ui->comboUnit2->addItem(ui->comboUnit1->itemText(i));

    switch (index)
    {
    case 0: // Voltage
        initMultipliersVoltage(ui->comboMultiplier1);
        break;
    case 1: // Current
        initMultipliersCurrent(ui->comboMultiplier1);
        break;
    case 2: // Power
        initMultipliersPower(ui->comboMultiplier1);
        break;
    case 3: // Resistance
        initMultipliersResistance(ui->comboMultiplier1);
        break;
    }

    ui->comboUnit2->setCurrentIndex(0);
}

void Ohms::on_comboUnit2_currentIndexChanged(int index)
{
    switch (3 * ui->comboUnit1->currentIndex() + index)
    {
    case UNIT_VI:
    case UNIT_PI:
    case UNIT_RI:
        initMultipliersCurrent(ui->comboMultiplier2);
        break;
    case UNIT_VP:
    case UNIT_IP:
    case UNIT_RP:
        initMultipliersPower(ui->comboMultiplier2);
        break;
    case UNIT_VR:
    case UNIT_IR:
    case UNIT_PR:
        initMultipliersResistance(ui->comboMultiplier2);
        break;
    case UNIT_IV:
    case UNIT_PV:
    case UNIT_RV:
        initMultipliersVoltage(ui->comboMultiplier2);
        break;
    }
}

void Ohms::on_spinValue1_valueChanged(double)
{
    recalculate();
}

void Ohms::on_spinValue2_valueChanged(double)
{
    recalculate();
}

void Ohms::on_comboMultiplier1_currentIndexChanged(int)
{
    recalculate();
}

void Ohms::on_comboMultiplier2_currentIndexChanged(int)
{
    recalculate();
}

static QString toQString(double d, const QString &unit)
{
    QString result(QString::number(d) + ' ' + unit);

    if (d < 0.001 || d >= 1000.0)
    {
        result.append(" (");
        result.append(Utils::normalize(d, unit));
        result.append(')');
    }

    return result;
}

void Ohms::recalculate()
{
    int x = ui->comboMultiplier1->currentIndex();
    double value1 = ui->spinValue1->value()
                  * ui->comboMultiplier1->itemData(x).toDouble();

    x = ui->comboMultiplier2->currentIndex();
    double value2 = ui->spinValue2->value()
                  * ui->comboMultiplier2->itemData(x).toDouble();

    const int unit1 = ui->comboUnit1->currentIndex();
    const int unit2 = ui->comboUnit2->currentIndex();

    switch (3 * unit1 + unit2)
    {
    case UNIT_IV:
        std::swap(value1, value2);
    case UNIT_VI:
        ui->labelResultT1->setText("Power (P = V * I) = ");
        ui->labelResultT2->setText("Resistance (R = V / I) = ");
        ui->labelResult1->setText(toQString(value1 * value2, Watt));
        ui->labelResult2->setText((value2 ? toQString(value1 / value2, Ohm) : inf));
        break;
    case UNIT_PV:
        std::swap(value1, value2);
    case UNIT_VP:
        ui->labelResultT1->setText("Current (I = P / V) = ");
        ui->labelResultT2->setText("Resistance (R = V" + squared + " / P) = ");
        ui->labelResult1->setText((value1 ? toQString(value2 / value1, Amp) : inf));
        ui->labelResult2->setText((value2 ? toQString(value1 * value1 / value2, Ohm) : inf));
        break;
    case UNIT_RV:
        std::swap(value1, value2);
    case UNIT_VR:
        ui->labelResultT1->setText("Current (I = V / R) = ");
        ui->labelResultT2->setText("Power (P = V" + squared + " / R) = ");
        ui->labelResult1->setText((value2 ? toQString(value1 / value2, Amp) : inf));
        ui->labelResult2->setText((value2 ? toQString(value1 * value1 / value2, Watt) : inf));
        break;
    case UNIT_IP:
        std::swap(value1, value2);
    case UNIT_PI:
        ui->labelResultT1->setText("Voltage (V = P / I) = ");
        ui->labelResultT2->setText("Resistance (R = P / I" + squared + ") = ");
        ui->labelResult1->setText((value2 ? toQString(value1 / value2, Volt) : inf));
        ui->labelResult2->setText((value2 ? toQString(value1 / (value2 * value2), Ohm) : inf));
        break;
    case UNIT_RI:
        std::swap(value1, value2);
    case UNIT_IR:
        ui->labelResultT1->setText("Voltage (V = I * R) = ");
        ui->labelResultT2->setText("Power (P = I" + squared + " * R) = ");
        ui->labelResult1->setText(toQString(value1 * value2, Volt));
        ui->labelResult2->setText(toQString(value1 * value1 * value2, Watt));
        break;
    case UNIT_RP:
        std::swap(value1, value2);
    case UNIT_PR:
        ui->labelResultT1->setText("Voltage (V = " + root + "(P * R)) = ");
        ui->labelResultT2->setText("Current (I = " + root + "(P / R)) = ");
        ui->labelResult1->setText(toQString(sqrt(value1 * value2), Volt));
        ui->labelResult2->setText((value2 ? toQString(sqrt(value1 / value2), Amp) : inf));
        break;
    }
}
