/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDebug>
#include <QMessageBox>


/*!
  The constructor sets up the main user interface and instantiates the 
  AudioMonitor, the UserNotifier as well as the application Settings. It 
  connects the signals from the AudioMonitor and UserNotifier to internal 
  methods. As a last step it starts the audio monitoring.
*/
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    // load settings
    itsSettings = new Settings();

    // setup audio monitor
    itsAudioMonitor = new AudioMonitor(itsSettings, this);
    connect(itsAudioMonitor, SIGNAL(update(int, int)), this, SLOT(refreshAudioData(int, int)));
    itsMonitorActive = false;
    itsNotificationPending = false;

    // setup GUI
    ui->setupUi(this);
    itsActivationDelayTimer.setSingleShot(true);
    connect(&itsActivationDelayTimer, SIGNAL(timeout()), this, SLOT(activationTimerExpired()));

    // setup default UI values
    ui->lineEdit_phone->setText(itsSettings->itsPhonenumber);

    // setup graphs
    itsAudioLevelGraphics = new AudioLevelGraphicsScene(ui->graphicsViewLevel, itsSettings, this);
    itsAudioCounterGraphics = new AudioLevelGraphicsScene(ui->graphicsViewCounter, itsSettings, this);

    // connect UI dialogs
    connect(ui->actionSettings, SIGNAL(triggered()), itsSettings, SLOT(ShowDialog()));

    // start audio capturing
    startAudio();
}


/*!
  The destructor initiates a save of the application settings. Then it frees
  class instances as needed.
*/
MainWindow::~MainWindow()
{
    // get graphics dimensions
    qDebug() << ui->graphicsViewLevel->geometry().width() << ui->graphicsViewLevel->geometry().height();

    // save settings
    itsSettings->Save();

    delete ui;
}


/*!
  changeEvent reflects the Qt default implementation to switch the user 
  interface language.
*/
void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}


/*!
  This user interface event trigger displays an about message box.
*/
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox msgBox;
    msgBox.setWindowTitle("Babyphone");
    msgBox.setText(tr("The babyphone is a baby monitoring application that measures the audio noise level.\nIf the volume exceeds a configurable threshold the phone initiates a call to a predefined phone number (the parents).\n\n(c) 2010, Roman Morawek"));
    msgBox.exec();
}


/*!
  This user interface event trigger updates the value of the parent's phone 
  number.
*/
void MainWindow::on_lineEdit_phone_textChanged()
{
    // update settings
    itsSettings->itsPhonenumber = ui->lineEdit_phone->text();
}


/*!
  This user interface event trigger reflects the main application button that
  switches the application to the active mode or disables it respectively.
  Before switching to active mode it checks the phone number for a non-empty 
  entry and aborts with en error pop-up if empty.
  on_pushButton_clicked also changes the enabled attribute of several GUI 
  controls depending on the application state.
*/
void MainWindow::on_pushButton_clicked()
{
    // stop the potentially running activation delay counter
    itsActivationDelayTimer.stop();

    // did we switch ON or OFF?
    if (itsMonitorActive) {
        // switch OFF
        deactivateMonitor();
    }
    else {
        // switch ON

        // check whether we have a valid phone number
        if (itsSettings->itsPhonenumber.size() > 0) {
            // activate it
            activateMonitor();
        }
        else {
            // incomplete settings: no valid phone number
            // open dialog to warn user
            QMessageBox::warning(this, tr("Babyphone"),
                  tr("No valid parent's phone number set. Please adjust the application settings."));

            // reset the button color again
            ui->pushButton->setChecked(false);
        }
    }
}


/*!
  activateMonitor instantiates the UserNotifier and starts the activation
  timer. Also it updates the user interface items properties to the active
  status.
*/
void MainWindow::activateMonitor()
{
    // update GUI
    ui->pushButton->setText(tr("Active - Waiting for activation delay"));
    ui->menuSettings->setEnabled(false);
    ui->lineEdit_phone->setEnabled(false);
    if (itsSettings->itsDisableGraphs) {
        ui->graphicsViewCounter->setEnabled(false);
        ui->graphicsViewLevel->setEnabled(false);
        itsAudioLevelGraphics->Clear();
        itsAudioCounterGraphics->Clear();
    }

    // setup phone call handler
    itsUserNotifier = new UserNotifier(itsSettings);
    connect(itsUserNotifier, SIGNAL(notifyFinished()), this, SLOT(notifyFinished()));

    // start activation timeout
    itsActivationDelayTimer.start(itsSettings->itsActivationDelay*1000);

    // the active flag is set immediately
    itsMonitorActive = true;
}


/*!
  deactivateMonitor deletes the UserNotifier instance and switches to monitor
  mode. Also it updates the user interface items properties to the inactive
  status.
*/
void MainWindow::deactivateMonitor()
{
    // switch OFF
    itsMonitorActive = false;

    // terminate phone call handler
    disconnect(itsUserNotifier, SIGNAL(notifyFinished()), this, SLOT(notifyFinished()));
    delete itsUserNotifier;

    // update GUI
    ui->pushButton->setText(tr("Inactive - Click to start"));
    ui->menuSettings->setEnabled(true);
    ui->lineEdit_phone->setEnabled(true);
    if (itsSettings->itsDisableGraphs) {
        ui->graphicsViewCounter->setEnabled(true);
        ui->graphicsViewLevel->setEnabled(true);
    }
}


/*!
  activationTimerExpired gets called as the activation actually starts.
  It just modifies the button's label.
*/
void MainWindow::activationTimerExpired()
{
    ui->pushButton->setText(tr("Active - Click to Stop"));
}


/*!
  notifyFinished represents the slot that gets called as the UserNotifier
  finished its phone call and switches the monitoring state to active again.
  This is done by starting a single shot timer with the configured recall 
  timeout to inhibit immediate re-calling of the parents.
*/
void MainWindow::notifyFinished()
{
    // notifcation finished
    itsNotificationPending = false;

    // update GUI
    ui->pushButton->setEnabled(true);
    ui->pushButton->setText(tr("Active - Waiting for reactivation delay"));

    // bring the application back to focus
    // we need a timeout for the phone application to quit
    QTimer::singleShot(itsSettings->REFOCUS_TIMER, this, SLOT(bringWindowToFront()));

    // reset audio monitor warning
    itsAudioMonitor->itsCounter = 0;

    // start activation timeout
    itsActivationDelayTimer.start(itsSettings->itsRecallTimer*1000);

    // restart audio monitoring
    startAudio();
}


/*!
  bringWindowToFront puts the application window to full screen foreground.
*/
void MainWindow::bringWindowToFront()
{
    // bring the application back to focus
    activateWindow();
    raise();
}


/*!
  startAudio performs the periodical reactivation of the audio capturing.
  There is a repeating activate and deactivate with given timeouts to keep the 
  audio stream synchronous to real time. This needs to be considered as a 
  work around to inhibit audio capturing problems.
*/
void MainWindow::startAudio()
{
    itsAudioMonitor->start();

    // setup safety timer
    QTimer::singleShot(itsSettings->AUDIO_ON_TIMER, this, SLOT(stopAudio()));
}


/*!
  stopAudio performs the periodical deactivation of the audio capturing.
  See startAudio for further explaination.
*/
void MainWindow::stopAudio()
{
    // suspend audio monitoring
    itsAudioMonitor->stop();

    // in case of notification, we get an event on its end
    // otherwise we must restart automatically via a timer
    if (!itsNotificationPending)
        QTimer::singleShot(itsSettings->AUDIO_OFF_TIMER, this, SLOT(startAudio()));
}


/*!
  refreshAudioData periodically receives the audio samples from the AudioMonitor,
  updates the audio graphs and performs the threshold check to initiate a phone
  call if needed. If the phone call initiation failed it displays an error 
  pop-up.
*/
void MainWindow::refreshAudioData(int counter, int value)
{
    // update GUI if inactive or if set to always update
    if (!itsMonitorActive || !itsSettings->itsDisableGraphs) {
        itsAudioLevelGraphics->AddValue(value);
        itsAudioCounterGraphics->AddValue(counter);
    }

    // check for noise
    if (itsMonitorActive && !itsActivationDelayTimer.isActive()
        && (counter > itsSettings->THRESHOLD_VALUE))
    {
        qDebug() << tr("Audio threshold reached. Notifying user.");

        // store event
        itsNotificationPending = true;

        // update GUI
        ui->pushButton->setEnabled(false);

        // stop monitoring
        stopAudio();

        // notify user
        if (itsUserNotifier->Notify() == false)
        {
            // the notify command yielded an error
            QMessageBox *msgBox = new QMessageBox(QMessageBox::Critical, tr("Babyphone"),
                  tr("Could not establish phone call. Please check the phone number settings."));
            // the message box needs a timeout to keep the application active for retry
            QTimer *msgBoxCloseTimer = new QTimer(msgBox);
            msgBoxCloseTimer->setInterval(itsSettings->MSG_BOX_TIMEOUT);
            msgBoxCloseTimer->setSingleShot(true);
            connect(msgBoxCloseTimer, SIGNAL(timeout()), msgBox, SLOT(accept())); // or reject
            msgBoxCloseTimer->start();
            msgBox->exec();

            // restart audio capturing
            notifyFinished();
        }
    }
}
