/*
 * position-watcher.c - Source for PositionWatcher
 * Copyright (C) 2010 Guillaume Desmottes
 * @author Guillaume Desmottes <gdesmott@gnome.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <stdlib.h>

#include <telepathy-glib/channel.h>
#include <telepathy-glib/contact.h>
#include <telepathy-glib/dbus.h>
#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/gtypes.h>
#include <telepathy-glib/util.h>

#include "position-watcher.h"

#include "connection-watcher.h"

static TpContactFeature features[] = { TP_CONTACT_FEATURE_ALIAS,
      TP_CONTACT_FEATURE_AVATAR_TOKEN };

G_DEFINE_TYPE(PositionWatcher, position_watcher, G_TYPE_OBJECT)

/* signal enum */
enum
{
    POSITION_CHANGED,
    LAST_SIGNAL,
};

//static guint signals[LAST_SIGNAL] = {0};

/* private structure */
typedef struct _PositionWatcherPrivate PositionWatcherPrivate;

struct _PositionWatcherPrivate
{
  ConnectionWatcher *conn_watcher;
  GSList *connections;

  gboolean dispose_has_run;
};

#define POSITION_WATCHER_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE ((o), POSITION_WATCHER_TYPE, PositionWatcherPrivate))

static void conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    PositionWatcher *self);

static void
remove_connection (PositionWatcher *self,
    TpConnection *conn)
{
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (self);

  g_signal_handlers_disconnect_by_func (conn, G_CALLBACK (conn_invalidated_cb),
      self);
  priv->connections = g_slist_remove (priv->connections, conn);
  g_object_unref (conn);
}

static void
position_watcher_init (PositionWatcher *obj)
{
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (obj);

  priv->conn_watcher = connection_watcher_new ();
  g_assert (priv->conn_watcher != NULL);
}

static void position_watcher_dispose (GObject *object);
static void position_watcher_finalize (GObject *object);

static void
position_watcher_class_init (PositionWatcherClass *position_watcher_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (position_watcher_class);

  g_type_class_add_private (position_watcher_class, sizeof (PositionWatcherPrivate));

  object_class->dispose = position_watcher_dispose;
  object_class->finalize = position_watcher_finalize;

  /*
  signals[POSITION_CHANGED] = g_signal_new ("position-changed",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    g_cclosure_marshal_VOID__OBJECT,
    G_TYPE_NONE, 1, TP_TYPE_CONNECTION);
    */
}

void
position_watcher_dispose (GObject *object)
{
  PositionWatcher *self = POSITION_WATCHER (object);
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (self);
  GSList *l;

  if (priv->dispose_has_run)
    return;

  for (l = priv->connections; l != NULL; l = g_slist_next (l))
    {
      g_object_unref (l->data);
    }

  priv->dispose_has_run = TRUE;

  g_object_unref (priv->conn_watcher);

  if (G_OBJECT_CLASS (position_watcher_parent_class)->dispose)
    G_OBJECT_CLASS (position_watcher_parent_class)->dispose (object);
}

void
position_watcher_finalize (GObject *object)
{
  PositionWatcher *self = POSITION_WATCHER (object);
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (self);

  g_slist_free (priv->connections);

  G_OBJECT_CLASS (position_watcher_parent_class)->finalize (object);
}

PositionWatcher *
position_watcher_new (void)
{
  return g_object_new (POSITION_WATCHER_TYPE,
      NULL);
}

static void
conn_invalidated_cb (TpProxy *conn,
    guint domain,
    gint code,
    gchar *message,
    PositionWatcher *self)
{
  g_print ("connection %s invalidated; removing\n", tp_proxy_get_object_path (
        conn));

  remove_connection (self, TP_CONNECTION (conn));
}

static void
get_contacts_cb (TpConnection *connection,
    guint n_contacts,
    TpContact * const *contacts,
    guint n_failed,
    const TpHandle *failed,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  GHashTable *locations = user_data;
  guint i;

  if (error != NULL)
    {
      g_print ("failed to get TpContact: %s\n", error->message);
      return;
    }

  for (i = 0; i < n_contacts; i++)
    {
      TpContact *contact = contacts[i];
      TpHandle handle;
      GHashTable *location;

      handle = tp_contact_get_handle (contact);
      location = g_hash_table_lookup (locations, GUINT_TO_POINTER (handle));
      if (location == NULL)
        continue;

      /* TODO: fire sig */
      g_print ("-- position of %s\n", tp_contact_get_alias (contact));
      tp_asv_dump (location);
    }
}

static void
get_locations_cb (TpConnection *conn,
    GHashTable *locations,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  GHashTableIter iter;
  gpointer key, value;
  GArray *handles;

  if (error != NULL)
    {
      g_print ("GetLocations failed: %s\n", error->message);
      return;
    }

  handles = g_array_new (FALSE, FALSE, sizeof (TpHandle));

  g_hash_table_iter_init (&iter, locations);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      TpHandle handle = GPOINTER_TO_UINT (key);
      GHashTable *location = value;

      if (g_hash_table_size (location) == 0)
        continue;

      g_array_append_val (handles, handle);
    }

  if (handles->len > 0)
    {
      tp_connection_get_contacts_by_handle (conn, handles->len,
          (TpHandle *) handles->data, G_N_ELEMENTS (features), features,
          get_contacts_cb,
          g_hash_table_ref (locations), (GDestroyNotify) g_hash_table_unref,
          weak_object);
    }

  g_array_free (handles, TRUE);
}

static void
chan_ready_cb (TpChannel *chan,
    const GError *error,
    gpointer user_data)
{
  const TpIntSet *members;
  GArray *handles;

  if (error != NULL)
    {
      g_print ("channel is not ready\n");
      return;
    }

  members = tp_channel_group_get_members (chan);
  if (tp_intset_size (members) == 0)
    return;

  handles = tp_intset_to_array (members);

  tp_cli_connection_interface_location_call_get_locations (
      tp_channel_borrow_connection (chan), -1, handles, get_locations_cb,
      NULL, NULL, G_OBJECT (user_data));

  g_array_free (handles, TRUE);
}

static void
ensure_stored_cb (TpConnection *conn,
    gboolean yours,
    const gchar *path,
    GHashTable *properties,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  TpChannel *chan;
  GError *err = NULL;

  if (error != NULL)
    {
      g_print ("Failed to request 'stored' list: %s\n", error->message);
      return;
    }

  chan = tp_channel_new_from_properties (conn, path, properties, &err);
  if (chan == NULL)
    {
      g_print ("Failed to create TpChannel object: %s\n", error->message);
      g_error_free (err);
      return;
    }

  tp_channel_call_when_ready (chan, chan_ready_cb, weak_object);
}

static void
get_contact_cb (TpConnection *connection,
    guint n_contacts,
    TpContact * const *contacts,
    guint n_failed,
    const TpHandle *failed,
    const GError *error,
    gpointer user_data,
    GObject *weak_object)
{
  GHashTable *location = user_data;
  TpContact *contact;

  if (error != NULL)
    {
      g_print ("failed to get TpContact: %s\n", error->message);
      return;
    }

  if (n_failed > 0)
    return;

  contact = contacts[0];

  /* TODO: fire sig */
  g_print ("-- updated pos of %s\n", tp_contact_get_alias (contact));
  tp_asv_dump (location);
}

static void
location_updated_cb (TpConnection *conn,
    TpHandle handle,
    GHashTable *location,
    gpointer user_data,
    GObject *weak_object)
{
  if (location == NULL || g_hash_table_size (location) == 0)
    return;

  tp_connection_get_contacts_by_handle (conn, 1, (TpHandle *) &handle,
      G_N_ELEMENTS (features), features, get_contact_cb,
      g_hash_table_ref (location), (GDestroyNotify) g_hash_table_unref,
      weak_object);
}

static void
connection_added_cb (ConnectionWatcher *watcher,
    TpConnection *conn,
    PositionWatcher *self)
{
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (self);
  GHashTable *request;

  if (g_slist_find (priv->connections, conn) != NULL)
    return;

  if (!tp_proxy_has_interface_by_id (conn,
        TP_IFACE_QUARK_CONNECTION_INTERFACE_LOCATION))
    return;

  if (!tp_proxy_has_interface_by_id (conn,
        TP_IFACE_QUARK_CONNECTION_INTERFACE_CONTACTS))
    return;

  g_print ("conn added: %s\n", tp_proxy_get_object_path (conn));
  priv->connections = g_slist_prepend (priv->connections, g_object_ref (conn));

  g_signal_connect (conn, "invalidated",
      G_CALLBACK (conn_invalidated_cb), self);

  request = tp_asv_new (
      TP_IFACE_CHANNEL ".ChannelType", G_TYPE_STRING,
        TP_IFACE_CHANNEL_TYPE_CONTACT_LIST,
      TP_IFACE_CHANNEL ".TargetHandleType", G_TYPE_UINT, TP_HANDLE_TYPE_LIST,
      TP_IFACE_CHANNEL ".TargetID", G_TYPE_STRING, "stored",
      NULL);

  tp_cli_connection_interface_requests_call_ensure_channel (conn, G_MAXINT,
      request, ensure_stored_cb, NULL, NULL, G_OBJECT (self));

  tp_cli_connection_interface_location_connect_to_location_updated (conn,
      location_updated_cb, NULL, NULL, G_OBJECT (self), NULL);

  g_hash_table_unref (request);
}

void
position_watcher_start (PositionWatcher *self)
{
  PositionWatcherPrivate *priv = POSITION_WATCHER_GET_PRIVATE (self);

  connection_watcher_start (priv->conn_watcher);

  g_signal_connect (priv->conn_watcher, "connection-added",
      G_CALLBACK (connection_added_cb), self);
}
