#include "readwidget.h"

#include <QVBoxLayout>
#include <QLabel>
#include <QProgressBar>
#include <QPushButton>
#include <QLineEdit>
#include <QTime>
#include <QMessageBox>
#include <QFileInfo>

#include "audiobook.h"

VolumeDialog::VolumeDialog(Phonon::AudioOutput* aAudioOutput,QWidget* aParent ):QDialog(aParent)
{
    mVolumeSlider = new Phonon::VolumeSlider(this);
    mVolumeSlider->setAudioOutput(aAudioOutput);
    QVBoxLayout* mainLayout = new QVBoxLayout(this);
    mainLayout->addWidget(mVolumeSlider);
}

ReadWidget::ReadWidget( QWidget* aParent ): QWidget(aParent), mCurrentBook(0),mMediaObject(0)
{
    mAudioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    mAudioOutput->setObjectName("audio");
    mMediaObject = new Phonon::MediaObject(this);
    mMediaObject->setObjectName("media");
    mMediaObject->setTickInterval(1000);   

    connect(mMediaObject,SIGNAL(tick(qint64)),this,SLOT(readProgress(qint64)));
    connect(mMediaObject,SIGNAL(totalTimeChanged(qint64)),this,SLOT(totalTimeChanged(qint64)));
    connect(mMediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
             this, SLOT(stateChanged(Phonon::State, Phonon::State)));
    connect(mMediaObject,SIGNAL(finished()),
            this,SLOT(finished()));

    Phonon::createPath(mMediaObject, mAudioOutput);

    setupUi();
}

ReadWidget::~ReadWidget()
{    
}

void ReadWidget::setupUi()
{
    mBookName = new QLabel("",this);
    mCurrentSourceName = new QLabel("",this);

    mReadProgress = new Phonon::SeekSlider(this);
    mReadProgress->setMediaObject(mMediaObject);

    mProgress = new QLabel("",this);
    mTotal = new QLabel("",this);

    mPlayButton = new QPushButton(">",this);
    mPlayButton->setEnabled(false);
    connect(mPlayButton,SIGNAL(clicked()),mMediaObject,SLOT(play()));

    mPauseButton = new QPushButton("||",this);
    mPauseButton->setEnabled(false);
    connect(mPauseButton,SIGNAL(clicked()),mMediaObject,SLOT(pause()));

    mStopButton = new QPushButton("X",this);
    mStopButton->setEnabled(false);
    connect(mStopButton,SIGNAL(clicked()),mMediaObject,SLOT(stop()));

    mSoundButton = new QPushButton("<))",this);
    connect(mSoundButton,SIGNAL(clicked()),this,SLOT(setVolume()));

    QHBoxLayout* progressLabelLayout = new QHBoxLayout;
    progressLabelLayout->addWidget(mProgress);
    progressLabelLayout->addStretch();
    progressLabelLayout->addWidget(mTotal);

    QVBoxLayout* progressSliderLayout = new QVBoxLayout;
    progressSliderLayout->addWidget(mReadProgress);
    progressSliderLayout->addLayout(progressLabelLayout);
    progressSliderLayout->addStretch();

    QHBoxLayout* controlLayout = new QHBoxLayout;
    controlLayout->addWidget(mPlayButton);
    controlLayout->addWidget(mPauseButton);
    controlLayout->addWidget(mStopButton);
    controlLayout->addWidget(mSoundButton);

    QVBoxLayout* mainLayout = new QVBoxLayout;
    mainLayout->addWidget(  mBookName );    
    mainLayout->addWidget( mCurrentSourceName );
    mainLayout->addStretch();
    mainLayout->addLayout( progressSliderLayout);    
    mainLayout->addStretch();
    mainLayout->addLayout( controlLayout );

    this->setLayout(mainLayout);
}

void ReadWidget::readBook( AudioBook* aBook )
{    
    mCurrentBook = aBook;

    if( aBook == 0 )
    {        
        return;
    }
    mCurrentSourceIndex = mCurrentBook->currentSource();

    mAudioOutput->setVolume( mCurrentBook->volume() );
    mMediaObject->setCurrentSource( aBook->source().at(mCurrentSourceIndex) );
    mBookName->setText( aBook->bookName());
    const QString& currentSource = aBook->source().at(aBook->currentSource());
    QFileInfo info( currentSource);
    mCurrentSourceName->setText(info.fileName());
    readProgress(mCurrentBook->progress());
    mPlayButton->setEnabled(true);
}

void ReadWidget::back()
{
    mMediaObject->pause();
    //emit backClicked();
}

void ReadWidget::totalTimeChanged(qint64 aTotalTime)
{    
    mTotal->setText(timeToString(aTotalTime));    
}

void ReadWidget::readProgress(qint64 aCurrentPosition )
{    
    mProgress->setText(timeToString(aCurrentPosition));
    mCurrentBook->setProgress(aCurrentPosition);
}

QString ReadWidget::timeToString(qint64 aTime)
{
    QTime time;
    time = time.addMSecs(aTime);
    return time.toString(Qt::TextDate);
}

void ReadWidget::stateChanged(Phonon::State aNewState, Phonon::State /*aOldState */)
 {
     switch (aNewState) {
         case Phonon::ErrorState:
             if (mMediaObject->errorType() == Phonon::FatalError) {
                 QMessageBox::warning(this, tr("Fatal Error"),
                 mMediaObject->errorString());
             } else {
                 QMessageBox::warning(this, tr("Error"),
                 mMediaObject->errorString());
             }
             break;

         case Phonon::PlayingState:
                mMediaObject->seek(mCurrentBook->progress());
                mPlayButton->setEnabled(false);
                mPauseButton->setEnabled(true);
                mStopButton->setEnabled(true);
                break;
         case Phonon::StoppedState:
                mPlayButton->setEnabled(true);
                mPauseButton->setEnabled(false);
                mStopButton->setEnabled(false);
                break;
         case Phonon::PausedState:
                mPlayButton->setEnabled(true);
                mPauseButton->setEnabled(false);
                mStopButton->setEnabled(true);
                break;
                }
 }

void ReadWidget::finished ()
{
    mCurrentSourceIndex++;
    if( mCurrentSourceIndex >= mCurrentBook->source().count() )
    {
        mCurrentSourceIndex = 0;
    }

    mCurrentBook->setProgress(0);
    mCurrentBook->setCurrentSource( mCurrentSourceIndex );
    mMediaObject->setCurrentSource( mCurrentBook->source().at(mCurrentSourceIndex) );
    const QString& currentSource = mCurrentBook->source().at(mCurrentBook->currentSource());
    QFileInfo info( currentSource);
    mCurrentSourceName->setText(info.fileName());
    mMediaObject->play();
}

void ReadWidget::setVolume()
{
    VolumeDialog dialog(mAudioOutput);
    dialog.exec();
    mCurrentBook->setVolume(mAudioOutput->volume());
}
