/*
 * ObjReader.cpp
 *
 *  Created on: 28.12.2009
 *      Author: pavel
 */

#include "ObjReader.h"

#include <QFile>
#include <QTextStream>
#include <iostream>
#include <QTime>
#include <QFileInfo>
#include <cmath>

using namespace std;

void BBox::include(QVector<float> v) {
	minPoint.setX(min(minPoint.x(), qreal(v[0])));
	minPoint.setY(min(minPoint.y(), qreal(v[1])));
	minPoint.setZ(min(minPoint.z(), qreal(v[2])));

	maxPoint.setX(max(maxPoint.x(), qreal(v[0])));
	maxPoint.setY(max(maxPoint.y(), qreal(v[1])));
	maxPoint.setZ(max(maxPoint.z(), qreal(v[2])));
}

QVector3D BBox::size() {
	return maxPoint - minPoint;
}

ObjReader::ObjReader(const QString path) {
	QFile objFile(path);
	QTextStream in(&objFile);
	QString line;
	QStringList parts;
	TexSlice ts;

	QTextStream cout(stdout);
	QTime t;

	relPath = QFileInfo(path).path();
	objFile.open(QIODevice::ReadOnly | QIODevice::Text);
	t.start();

	while (not in.atEnd()) {
		line = in.readLine();
		parts = line.split(" ", QString::SkipEmptyParts);

		if (parts.length() == 0)
			continue; // skip empty lines

		if (parts[0] == "v") {
			readV(parts);
		} else if (parts[0] == "vn") {
			readVN(parts);
		} else if (parts[0] == "vt") {
			readVT(parts);
		} else if (parts[0] == "f") {
			readF(parts);
		} else if (parts[0] == "mtllib") {
			readMtlLib(parts);
		} else if (parts[0] == "usemtl") {
			curMtl = parts[1];
		}
	}

	objFile.close();

	cout << QString("%1 stats:").arg(path) << endl;
	cout << QString(" File read in: %1ms").arg(t.elapsed()) << endl;
	cout << QString(" Vertices: %1").arg(verts.size()) << endl;
	cout << QString(" Normals: %1").arg(normals.size()) << endl;
	cout << QString(" Tex UV: %1").arg(tex_uv.size()) << endl;
}

QVector<float>& ObjReader::getVBO() {
	QList<QVector<float> > vals = slices.values();

	for (int i = 0; i < vals.size(); i++) {
		vbo << vals[i];
	}

	return vbo;
}

QVector<TexSlice> ObjReader::getSlices() {
	TexSlice ts;
	QList<QString> keys = mtrls.keys();
	unsigned int offset = 0;

	QTextStream cout(stdout);

	for (int i = 0; i < keys.size(); i++) {
		ts.start = offset;
		ts.count = slices[keys[i]].size() / 8;
		ts.path = mtrls[keys[i]];

		texSlices << ts;

		offset += ts.count;
	}

	return texSlices;
}

void ObjReader::readV(QStringList& parts) {
	QVector<float> v(3);

	for (int i = 0; i < 3; i++) {
		v[i] = parts[i + 1].toFloat();
	}

	verts.append(v);
	boundings.include(v);
}

void ObjReader::readVN(QStringList& parts) {
	QVector<float> vn(3);

	for (int i = 0; i < 3; i++) {
		vn[i] = parts[i + 1].toFloat();
	}

	normals.append(vn);
}

void ObjReader::readVT(QStringList& parts) {
	QVector<float> vt(2);

	for (int i = 0; i < 2; i++) {
		vt[i] = parts[i + 1].toFloat();
	}

	tex_uv.append(vt);
}

void ObjReader::readF(QStringList& parts) {
	QStringList fp;
	int vi, ni, ti;

	for (int i = 0; i < 3; i++) {
		fp = parts[i + 1].split("/");

		if (fp.size() == 1) {
			vi = ni = fp[0].toInt() - 1;
		} else {
			vi = fp[0].toInt() - 1;
			ni = fp[2].toInt() - 1;
			ti = fp[1].toInt() - 1;
		}

		slices[curMtl] << verts[vi] << normals[ni] << tex_uv[ti];
	}
}

void ObjReader::readMtlLib(QStringList& parts) {
	MtlReader mr(relPath + "/" + parts[1]);
	mtrls = mr.getMaterials();

	QList<QString> keys = mtrls.keys();

	for (int i = 0; i < keys.size(); i++) {
		slices.insert(keys[i], QVector<float> ());
	}
}

MtlReader::MtlReader(const QString path) {
	QFile mtlFile(path);
	QTextStream in(&mtlFile);
	QString line;
	QStringList parts;
	QTextStream cout(stdout);
	QString curmtl;

	relPath = QFileInfo(path).path();

	mtlFile.open(QIODevice::ReadOnly | QIODevice::Text);

	while (not in.atEnd()) {
		line = in.readLine();
		parts = line.split(" ", QString::SkipEmptyParts);

		if (parts.length() == 0)
			continue; // skip empty lines

		if (parts[0] == "newmtl") {
			curmtl = parts[1];
		} else if (parts[0] == "map_Kd") {
			materials.insert(curmtl, relPath + "/" + parts[1]);
		}
	}

	mtlFile.close();
}

QHash<QString, QString> MtlReader::getMaterials() {
	return materials;
}
