#!/usr/bin/env python

"""GUI module for alarmed"""
import os
import sys
import signal
import time
import datetime
from PySide import QtCore, QtGui, QtMaemo5
#~ from PyQt4 import QtCore, QtGui
import alarmed_backend

class AlarmedGUI(QtGui.QWidget):
    def __init__(self, parent=None):
        """Main window displaying the list of events and 3 buttons"""
        QtGui.QWidget.__init__(self, parent)
        self.setWindowTitle("Alarmed")
        self.newEventButton = QtGui.QPushButton("New event")
        self.connect(self.newEventButton, QtCore.SIGNAL("clicked()"),
                     self.openEventDialog)
        self.deleteEventButton = QtGui.QPushButton("Remove")
        self.connect(self.deleteEventButton, QtCore.SIGNAL("clicked()"),
                     self.removeEvent)
        self.editEventButton = QtGui.QPushButton("Edit")
        self.connect(self.editEventButton, QtCore.SIGNAL("clicked()"),
                     self.editEvent)
        self.eventList = QtGui.QListWidget()
        self.eventList.setSelectionMode(
                     QtGui.QAbstractItemView.SingleSelection)
        self.connect(self.eventList, QtCore.SIGNAL("itemClicked(QListWidgetItem*)"),
                     self.toggleEvent)
        self.addRemoveHBox = QtGui.QHBoxLayout()
        self.addRemoveHBox.addWidget(self.newEventButton)
        self.addRemoveHBox.addWidget(self.deleteEventButton)
        self.addRemoveHBox.addWidget(self.editEventButton)
        self.allVBox = QtGui.QVBoxLayout()
        self.allVBox.addLayout(self.addRemoveHBox)
        self.allVBox.addWidget(self.eventList)
        self.setLayout(self.allVBox)
        #load existing jobs (if any)
        eventList = alarmed_backend.loadEventList()
        for eventInfo in eventList:
            self.addEventToList(eventInfo)
        self.checkSum = self.checkSumEventList()
        #read the available fields from the backend
        self.fields = alarmed_backend.loadFieldsDict()
        self.weekdays = []

    def showError(self, msg=""):
        """Shows a QMaemo5InformationBox containing the error message"""
        banner = QtMaemo5.QMaemo5InformationBox()
        banner.information(self, msg, 3000)

    def addEventToList(self, eventInfo, editing=False):
        """Adds a QListWidgetItem to the list of events"""
        if editing:
            self.eventList.takeItem(
                           self.eventList.currentRow())
        newEventTitle, newEventContent = alarmed_backend.generateEventString(eventInfo)
        newEvent = QtGui.QListWidgetItem(newEventTitle + "\n" + newEventContent)
        newEvent.setFlags(newEvent.flags() | QtCore.Qt.ItemIsUserCheckable | QtCore.Qt.ItemIsSelectable | QtCore.Qt.ItemIsEnabled)
        try:
            if eventInfo["enabled"] == 0:
                newEvent.setCheckState(QtCore.Qt.Unchecked)
            else:
                newEvent.setCheckState(QtCore.Qt.Checked)
        except:
            newEvent.setCheckState(QtCore.Qt.Checked)
        newEvent.eventInfo = eventInfo
        self.eventList.addItem(newEvent)

    def removeEvent(self):
        """Calls backend to remove an event"""
        if (self.eventList.currentItem()):
            alarmed_backend.deleteEvent(int(
             self.eventList.currentItem().eventInfo["id"]))
            self.eventList.takeItem(
             self.eventList.currentRow())
        else:
            self.showError('''<font color="#000000">Select an item first</font>''')

    def editEvent(self):
        """Same as a normal addition, but will delete existing"""
        if (self.eventList.currentItem()):
            self.openEventDialog(self.eventList.currentItem())
        else:
            self.showError('''<font color="#000000">Select an item first</font>''')

    def checkSumEventList(self):
        checkSum = 0
        for i in range(self.eventList.count()):
            checkSum = checkSum + int(self.eventList.item(i).checkState())
        return checkSum

    def toggleEvent(self, item):
        """Enable or disable an existing event"""
        tempCheckSum = self.checkSumEventList()
        if self.checkSum != tempCheckSum:
            self.checkSum = tempCheckSum
            state = item.checkState()
            id = int(item.eventInfo["id"])
            if state == 0:
                alarmed_backend.deleteEventAlarmd(item.eventInfo)
            if state == 2:
                a = alarmed_backend.makeEvent(item.eventInfo, int(item.eventInfo["id"]))
                if a == "invalid":
                    self.showError('''<font color="#000000">Please pick a date & time in the future</font>''')
                    return False
                

    def openEventDialog(self, item=None):
        """Present a dialog to set and edit events"""
        self.newEventDialog = QtGui.QDialog(self)
        self.newEventDialog.setWindowTitle("Add event")

        #There are three rows of widgets
        self.row1 = QtGui.QHBoxLayout(self)
        self.row2 = QtGui.QHBoxLayout(self)
        self.row3 = QtGui.QHBoxLayout(self)
        self.container = QtGui.QVBoxLayout(self)
        self.container.addLayout(self.row1)
        self.container.addLayout(self.row2)
        self.container.addLayout(self.row3)
        self.newEventDialog.setLayout(self.container)

        #picker for event types
        self.typePicker = QtGui.QComboBox()
        for i in self.fields["types"]:
            self.typePicker.addItem(self.fields["types"][i])
        self.connect(self.typePicker, QtCore.SIGNAL("activated(int)"),
                     self.updateInputField)
        self.row1.addWidget(self.typePicker)
        #picker for schedules (i.e. "weekly", "last day of month"...)
        self.schedulePicker = QtGui.QComboBox()
        for i in self.fields["schedules"]:
            self.schedulePicker.addItem(self.fields["schedules"][i])
        self.connect(self.schedulePicker,
                     QtCore.SIGNAL("activated(int)"),
                     self.updateScheduling)
        self.row1.addWidget(self.schedulePicker)
        #picker for how many steps
        self.numberPicker = QtGui.QSpinBox()
        self.row1.addWidget(self.numberPicker)
        self.numberPicker.hide()
        #picker for what type of step (i.e. seconds, minutes...)
        self.intervalPicker = QtGui.QComboBox()
        for i in self.fields["intervals"]:
            self.intervalPicker.addItem(self.fields["intervals"][i])
        self.row1.addWidget(self.intervalPicker)
        self.intervalPicker.hide()
        #picker for relative weekday (i.e. "next Tuesday")
        self.rWeekdayPicker = QtGui.QComboBox()
        for i in self.fields["weekdays"]:
            self.rWeekdayPicker.addItem(self.fields["weekdays"][i])
        self.row1.addWidget(self.rWeekdayPicker)
        self.rWeekdayPicker.hide()
        #picker for "on which weekdays" something should happen:
        self.weekdayPicker = QtGui.QPushButton("Days of week")
        self.connect(self.weekdayPicker, QtCore.SIGNAL("clicked()"),
                     self.pickWeekdays)
        self.row1.addWidget(self.weekdayPicker)
        self.weekdayPicker.hide()
        #picker for date using QtMaemo5
        self.datePicker = QtMaemo5.QMaemo5ValueButton("Date")
        self.datePicker.setValueLayout(
                        QtMaemo5.QMaemo5ValueButton.ValueUnderText)
        self.dateWidget = QtMaemo5.QMaemo5DatePickSelector()
        self.datePicker.setPickSelector(self.dateWidget)
        self.row1.addWidget(self.datePicker)
        #picker for time using QtMaemo5
        self.timePicker = QtMaemo5.QMaemo5ValueButton("Time")
        self.timePicker.setValueLayout(
                        QtMaemo5.QMaemo5ValueButton.ValueUnderText)
        self.timeWidget = QtMaemo5.QMaemo5TimePickSelector()
        self.timePicker.setPickSelector(self.timeWidget)
        self.row1.addWidget(self.timePicker)
        #input field for cronlike syntax
        self.cronPicker = QtGui.QLineEdit("* * * * *")
        self.row1.addWidget(self.cronPicker)
        self.cronPicker.hide()
        #input field for date syntax
        self.datesyntaxPicker = QtGui.QLineEdit("enter date string here")
        self.row1.addWidget(self.datesyntaxPicker)
        self.datesyntaxPicker.hide()
        #event name picker
        self.titlePicker = QtGui.QLineEdit("Enter name")
        self.row2.addWidget(self.titlePicker)
        #add button
        self.addButton = QtGui.QPushButton("Save Event")
        self.connect(self.addButton, QtCore.SIGNAL("clicked()"),
                     self.saveEvent)
        self.row2.addWidget(self.addButton)
        #picker for builtin functions
        self.builtinPicker = QtGui.QComboBox()
        for i in self.fields["builtins"]:
            self.builtinPicker.addItem(self.fields["builtins"][i])
        self.row3.addWidget(self.builtinPicker)
        self.builtinPicker.hide()
        #picker for profiles (will get populated each time it loads!)
        self.profilePicker = QtGui.QComboBox()
        self.row3.addWidget(self.profilePicker)
        self.profilePicker.hide()
        #picker for saved networks (will get populated each time it loads!)
        self.networkPicker = QtGui.QComboBox()
        self.row3.addWidget(self.networkPicker)
        self.networkPicker.hide()
        #picker for command line execution
        self.execPicker = QtGui.QLineEdit("Enter command")
        self.row3.addWidget(self.execPicker)
        self.execPicker.hide()
        #button for saving command to preset
        self.execSavePresetButton = QtGui.QPushButton("Save Preset")
        self.row3.addWidget(self.execSavePresetButton)
        self.execSavePresetButton.hide()
        self.connect(self.execSavePresetButton, QtCore.SIGNAL("clicked()"),
                     self.savePreset)
        #picker for loading presets
        self.execPresetPicker = QtGui.QPushButton("Load Preset")
        self.row3.addWidget(self.execPresetPicker)
        self.execPresetPicker.hide()
        self.connect(self.execPresetPicker, QtCore.SIGNAL("clicked()"),
                     self.pickPreset)

        # If we're editing, adjust the GUI to the right values
        if (item):
            self.newEventDialog.setWindowTitle("Edit event")
            #adjust schedule type
            try:
                i = self.schedulePicker.findText(
                                             item.eventInfo["schedule"])
                self.schedulePicker.setCurrentIndex(i)
                #update gui to fit interval type
                #self.updateScheduling(item.eventInfo["schedule"])
                self.updateScheduling(0)
            except KeyError: pass
            #adjust type
            try:
                i = self.typePicker.findText(item.eventInfo["type"])
                self.typePicker.setCurrentIndex(i)
            except KeyError: pass
            #construct a QDateTime from ttime, then adjust date and time
            dateTime = QtCore.QDateTime.fromTime_t(
                       item.eventInfo["ttime"])
            self.dateWidget.setCurrentDate(QtCore.QDate.fromString(
                            dateTime.toString("ddMMyyyy"),"ddMMyyyy"))
            self.timeWidget.setCurrentTime(QtCore.QTime.fromString(
                            dateTime.toString("hhmm"),"hhmm"))
            #adjust selected weekdays
            try: self.weekdays = (item.eventInfo["weekdays"])
            except KeyError: pass
            #adjust title
            self.titlePicker.setText(item.eventInfo["title"])
            #adjust relative weekday
            #~ self.rWeekdayPicker.setCurrentIndex(
             #~ item.eventInfo["weekday"])
            #adjust number of steps
            try: self.numberPicker.setValue(int(item.eventInfo["number"]))
            except KeyError: pass
            #adjust type of step
            try: i = self.intervalPicker.findText(
                                         item.eventInfo["interval"])
            except KeyError: pass
            try: self.intervalPicker.setCurrentIndex(i)
            except KeyError: pass
            #depending on type, adjust gui to fit event info
            try: self.updateTypeDependent(item.eventInfo["type"],
                                     item.eventInfo["content"])
            except KeyError: pass
            try:
                c = ""
                for part in item.eventInfo["cron"]:
                    c = c + " " + part
                c = c.lstrip()
                self.cronPicker.setText(c)
            except KeyError: pass
            try:
                self.datesyntaxPicker.setText(item.eventInfo["date"])
            except KeyError: pass
            self.editing = True
        else:
            self.editing = False
        self.updateScheduling(0)
        self.newEventDialog.open()

    def updateInputField(self, index):
        """updates the GUI to fit the current event type"""
        type = self.typePicker.currentText()
        t = self.fields["types"]
        if type == t["alarm_type"]:
            self.execPicker.hide()
            self.builtinPicker.hide()
            self.profilePicker.hide()
            self.networkPicker.hide()
            self.execSavePresetButton.hide()
            self.execPresetPicker.hide()
        elif type == t["exec_type"]:
            self.builtinPicker.hide()
            self.execPicker.show()
            self.profilePicker.hide()
            self.networkPicker.hide()
            self.execSavePresetButton.show()
            self.execPresetPicker.show()
        elif type == t["builtin_type"]:
            self.execPicker.hide()
            self.builtinPicker.show()
            self.profilePicker.hide()
            self.networkPicker.hide()
            self.execSavePresetButton.hide()
            self.execPresetPicker.hide()
        elif type == t["profile_type"]:
            self.execPicker.hide()
            self.builtinPicker.hide()
            self.execSavePresetButton.hide()
            self.execPresetPicker.hide()
            self.networkPicker.hide()
            self.profilePicker.show()
            self.profilePicker.clear()
            profiles = alarmed_backend.getAvailableProfiles()
            for p in profiles:
                self.profilePicker.addItem(p)
        elif type == t["netconnect_type"]:
            self.execPicker.hide()
            self.builtinPicker.hide()
            self.execSavePresetButton.hide()
            self.execPresetPicker.hide()
            self.profilePicker.hide()
            self.networkPicker.show()
            self.networkPicker.clear()
            networks = alarmed_backend.getSavedNetworks()
            for n in networks:
                self.networkPicker.addItem(n)
            self.networkPicker.model().sort(QtCore.Qt.AscendingOrder)

    def updateTypeDependent(self, type, content):
        """Updates the GUI contents of special types"""
        t = self.fields["types"]
        if type == t["builtin_type"]:
            i = self.builtinPicker.findText(content)
            self.builtinPicker.setCurrentIndex(i)
            self.builtinPicker.show()
        elif type == t["exec_type"]:
            self.execPicker.setText(content)
            self.execPicker.show()
            self.execSavePresetButton.show()
            self.execPresetPicker.show()
        elif type == t["profile_type"]:
            self.profilePicker.clear()
            profiles = alarmed_backend.getAvailableProfiles()
            for p in profiles:
                self.profilePicker.addItem(p)
            i = self.profilePicker.findText(
             self.eventList.currentItem().eventInfo["content"])
            if i == -1:
                self.showError(
                    '''<font color="#000000">This event has become invalid because you deleted profile %s.</font>'''
                    % self.eventList.currentItem().eventInfo["content"])
            self.profilePicker.setCurrentIndex(i)
            self.profilePicker.show()
        elif type == t["netconnect_type"]:
            self.networkPicker.clear()
            networks = alarmed_backend.getSavedNetworks()
            for n in networks:
                self.networkPicker.addItem(n)
            self.networkPicker.model().sort(QtCore.Qt.AscendingOrder)
            network = self.eventList.currentItem().eventInfo["content"]
            ident = ""
            for key in networks:
                if networks[key] == network:
                    ident = key
                    break
            i = self.networkPicker.findText(ident)
            if i == -1:
                self.showError(
                    '''<font color="#000000">This event has become invalid because you deleted network %s.</font>'''
                    % ident)
            self.networkPicker.setCurrentIndex(i)
            self.networkPicker.show()
            

    def updateScheduling(self, index):
        """Updates the GUI to fit the current schedule type"""
        schedule = self.schedulePicker.currentText()
        s = self.fields["schedules"]
        if schedule == s["every_minute"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.timePicker.hide()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule in [s["hourly"], s["daily"]]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.timePicker.show()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule in [s["only_once"], s["once_a_month"],
                          s["once_a_year"]]:
            self.weekdayPicker.hide()
            self.datePicker.show()
            self.timePicker.show()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule in [s["weekly"], s["biweekly"]]:
            self.weekdayPicker.show()
            self.datePicker.hide()
            self.timePicker.show()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule == s["end_of_month"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.timePicker.show()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule == s["in"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.timePicker.hide()
            self.numberPicker.show()
            self.intervalPicker.show()
            self.timePicker.hide()
            self.rWeekdayPicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule == s["next"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.show()
            self.timePicker.show()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule == s["tomorrow"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.timePicker.show()
            self.cronPicker.hide()
            self.datesyntaxPicker.hide()
        elif schedule == s["cron"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.timePicker.hide()
            self.cronPicker.show()
            self.datesyntaxPicker.hide()
        elif schedule == s["date"]:
            self.weekdayPicker.hide()
            self.datePicker.hide()
            self.numberPicker.hide()
            self.intervalPicker.hide()
            self.rWeekdayPicker.hide()
            self.timePicker.hide()
            self.cronPicker.hide()
            self.datesyntaxPicker.show()

    def pickPreset(self):
        """Dialog for loading and deleting presets"""
        self.presetDialog = QtGui.QDialog()
        self.presetDialog.setWindowTitle("Load or delete presets")
        self.presetDialogVBox = QtGui.QVBoxLayout()
        self.presetDialogHBox = QtGui.QHBoxLayout()
        # the list of presets on the right side
        self.presetList = QtGui.QListWidget()
        self.presetList.setSelectionMode(
                     QtGui.QAbstractItemView.SingleSelection)
        self.loadPresetList()
        self.presetDialogHBox.addWidget(self.presetList)
        # the add and pick buttons on the right
        self.presetPickButton = QtGui.QPushButton("Use preset")
        self.connect(self.presetPickButton, QtCore.SIGNAL("clicked()"),
                     self.usePreset)
        self.presetDeleteButton = QtGui.QPushButton("Delete preset")
        self.connect(self.presetDeleteButton, QtCore.SIGNAL("clicked()"),
                     self.deletePreset)
        self.presetDialogVBox.addWidget(self.presetPickButton)
        self.presetDialogVBox.addWidget(self.presetDeleteButton)
        self.presetDialogHBox.addLayout(self.presetDialogVBox)
        self.presetDialog.setLayout(self.presetDialogHBox)
        self.presetDialog.exec_()

    def loadPresetList(self):
        self.presetList.clear()
        presets = alarmed_backend.loadPresetList()
        for preset in presets:
            addedPreset = QtGui.QListWidgetItem(preset[0])
            addedPreset.preset = preset
            self.presetList.addItem(addedPreset)

    def usePreset(self):
        if (self.presetList.currentItem()):
            self.titlePicker.setText(
                                self.presetList.currentItem().preset[0])
            self.execPicker.setText(
                                self.presetList.currentItem().preset[1])
            self.presetDialog.accept()
        else:
            self.showError('''<font color="#000000">Select an item first</font>''')

    def deletePreset(self):
        """Calls backend to remove a preset """
        if (self.presetList.currentItem()):
            alarmed_backend.deletePreset(
                            self.presetList.currentItem().preset)
            self.presetList.removeItemWidget(
                            self.presetList.currentItem())
            self.loadPresetList()
        else:
            self.showError('''<font color="#000000">Select an item first</font>''')

    def savePreset(self):
        preset = []
        preset.append(str(self.titlePicker.text()))
        preset.append(str(self.execPicker.text()))
        alarmed_backend.savePreset(preset)
        self.showError('''<font color="#000000">Preset saved</font>''')

    def pickWeekdays(self):
        """Presents 7 checkboxes to select days"""
        self.weekdayDialog = QtGui.QDialog()
        self.weekdayDialog.setWindowTitle("Pick weekdays")
        self.weekdayDialogHBox = QtGui.QHBoxLayout()
        self.weekdayDialogVBox = QtGui.QVBoxLayout()
        w = self.fields["weekdays"]
        self.monBox = QtGui.QCheckBox(w["mon"])
        self.tueBox = QtGui.QCheckBox(w["tue"])
        self.wedBox = QtGui.QCheckBox(w["wed"])
        self.thuBox = QtGui.QCheckBox(w["thu"])
        self.friBox = QtGui.QCheckBox(w["fri"])
        self.satBox = QtGui.QCheckBox(w["sat"])
        self.sunBox = QtGui.QCheckBox(w["sun"])
        self.weekdayDialogOK = QtGui.QPushButton("OK")
        self.connect(self.weekdayDialogOK, QtCore.SIGNAL("clicked()"),
                     self.pickWeekdaysDone)
        self.weekdayDialogVBox.addWidget(self.monBox)
        self.weekdayDialogVBox.addWidget(self.tueBox)
        self.weekdayDialogVBox.addWidget(self.wedBox)
        self.weekdayDialogVBox.addWidget(self.thuBox)
        self.weekdayDialogVBox.addWidget(self.friBox)
        self.weekdayDialogVBox.addWidget(self.satBox)
        self.weekdayDialogVBox.addWidget(self.sunBox)
        self.weekdayDialogHBox.addLayout(self.weekdayDialogVBox, 1)
        self.weekdayDialogHBox.addWidget(self.weekdayDialogOK)
        #if we're in editing mode, self.weekdays is set
        if self.weekdays:
            if w["mon"] in self.weekdays: self.monBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["tue"] in self.weekdays: self.tueBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["wed"] in self.weekdays: self.wedBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["thu"] in self.weekdays: self.thuBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["fri"] in self.weekdays: self.friBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["sat"] in self.weekdays: self.satBox.setCheckState(
                                                QtCore.Qt.Checked)
            if w["sun"] in self.weekdays: self.sunBox.setCheckState(
                                                QtCore.Qt.Checked)
        self.weekdayDialog.setLayout(self.weekdayDialogHBox)
        self.weekdayDialog.exec_()

    def pickWeekdaysDone(self):
        """Updates self.weekdays when user is done selecting days"""
        self.weekdays = []
        w = self.fields["weekdays"]
        if self.monBox.isChecked(): self.weekdays.append(w["mon"])
        if self.tueBox.isChecked(): self.weekdays.append(w["tue"])
        if self.wedBox.isChecked(): self.weekdays.append(w["wed"])
        if self.thuBox.isChecked(): self.weekdays.append(w["thu"])
        if self.friBox.isChecked(): self.weekdays.append(w["fri"])
        if self.satBox.isChecked(): self.weekdays.append(w["sat"])
        if self.sunBox.isChecked(): self.weekdays.append(w["sun"])
        self.weekdayDialog.accept()

    def saveEvent(self):
        """takes the GUI input and prepares it for the backend"""
        if self.editing:
            old_id = self.eventList.currentItem().eventInfo["id"]
        else:
            old_id = False
        eventInfo = {}
        s = self.fields["schedules"]
        i = self.fields["intervals"]
        t = self.fields["types"]
        eventInfo["schedule"] = self.schedulePicker.currentText()
        eventInfo["type"] = self.typePicker.currentText()
        eventInfo["title"] = self.titlePicker.text()
        if self.schedulePicker.currentText() in [s["weekly"],
                                                 s["biweekly"]]:
            if not self.weekdays:
                self.showError('''<font color="#000000">Please select at least one weekday</font>''')
                return False
        sdate = self.dateWidget.currentDate()
        stime = self.timeWidget.currentTime()
        if self.schedulePicker.currentText() == s["in"]:
            i_type = self.intervalPicker.currentText()
            if i_type == i["seconds"]:
                secs = self.numberPicker.value()
            elif i_type == i["minutes"]:
                secs = self.numberPicker.value()*60
            elif i_type == i["hours"]:
                secs = self.numberPicker.value()*60*60
            elif i_type == i["days"]:
                secs = self.numberPicker.value()*60*60*24
            dateTime = QtCore.QDateTime.fromTime_t(int(time.time()))
            dateTime = dateTime.addSecs(secs)
            eventInfo["ttime"] = int(dateTime.toTime_t())
            eventInfo["number"] = int(self.numberPicker.value())
            eventInfo["interval"] = self.intervalPicker.currentText()
        elif self.schedulePicker.currentText() == s["next"]:
            day = self.rWeekdayPicker.currentText()
            w = self.fields["weekdays"]
            if day == w["mon"]: day = 1
            if day == w["tue"]: day = 2
            if day == w["wed"]: day = 3
            if day == w["thu"]: day = 4
            if day == w["fri"]: day = 5
            if day == w["sat"]: day = 6
            if day == w["sun"]: day = 7
            dateTime = QtCore.QDateTime.fromTime_t(int(time.time()))
            dateTime = dateTime.addDays(1)
            while dateTime.date().dayOfWeek() != day:
                dateTime = dateTime.addDays(1)
            dateTime.setTime(stime)
            eventInfo["ttime"] = int(dateTime.toTime_t())
            eventInfo["weekday"] = int(
                                     self.rWeekdayPicker.currentIndex())
        elif self.schedulePicker.currentText() == s["tomorrow"]:
            dateTime = QtCore.QDateTime.fromTime_t(int(time.time()))
            dateTime = dateTime.addDays(1)
            dateTime.setTime(stime)
            eventInfo["ttime"] = int(dateTime.toTime_t())
        elif self.schedulePicker.currentText() == s["cron"]:
            s = str(self.cronPicker.text())
            status, cronstring = alarmed_backend.interpretCronString(s)
            if not status:
                self.showError('''<font color="#000000">''' +
                               cronstring + '''</font>''')
                return False
            td = datetime.datetime.today()
            td = td + datetime.timedelta(seconds=+10)
            eventInfo["ttime"] = int(time.mktime(td.timetuple()))
            eventInfo["cron"] = cronstring
        elif self.schedulePicker.currentText() == s["date"]:
            eventInfo["date"] = self.datesyntaxPicker.text()
            ttime = alarmed_backend.interpretDateString(
                                    eventInfo["date"])
            if not ttime[0]:
                self.showError('''<font color="#000000">''' + ttime[1] +
                               '''</font>''')
                return False
            else:
                eventInfo["ttime"] = ttime[1]
                eventInfo["schedule"] = s["date"]
        else:
            dateTime = QtCore.QDateTime.fromTime_t(int(time.time()))
            dateTime.setDate(sdate)
            dateTime.setTime(stime)
            eventInfo["ttime"] = int(dateTime.toTime_t())

        type = self.typePicker.currentText()
        content = ""
        if type == t["exec_type"]:
            content = self.execPicker.text()
            eventInfo["content"] = content
        elif type == t["builtin_type"]:
            content = self.builtinPicker.currentText()
            eventInfo["content"] = content
        elif type == t["profile_type"]:
            content = self.profilePicker.currentText()
            eventInfo["content"] = content
        elif type == t["netconnect_type"]:
            content = alarmed_backend.getSavedNetworks()[self.networkPicker.currentText()]
            content = content.replace("@32@", " ")
            eventInfo["content"] = content
        if self.weekdays:
            eventInfo["weekdays"] = self.weekdays
        #call backend to make event
        a = alarmed_backend.makeEvent(eventInfo, old_id)
        if a == "invalid":
            self.showError('''<font color="#000000">Please pick a date & time in the future</font>''')
            return False
        #add event to gui list
        self.addEventToList(eventInfo, self.editing)
        self.checkSum = self.checkSumEventList()
        self.newEventDialog.accept()

app = QtGui.QApplication(sys.argv)
widget = AlarmedGUI()
widget.show()
lock = open("/tmp/alarmed.lock", "w")
lock.write("1")
lock.close()
r = app.exec_()
os.remove("/tmp/alarmed.lock")
sys.exit(r)
