/* -*- mode:c++; tab-width:4; c-basic-offset:4;
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

// ------------------------------------------------------------------------
/// \file c_xmldoc.cpp
/// \brief The implementation of the c_xmldoc class

#include <aegis_common.h>
using namespace std;
using namespace aegis;

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <memory.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

/// \brief XML handlers. The instance pointer to xmldoc is passed in 
/// the user_data argument 

static void XMLCALL 
_exp_element_start (void* user_data, 
					const XML_Char* el, 
					const XML_Char** attr)
{
	c_xmldoc* to_doc = (c_xmldoc*) user_data;

	if (to_doc) {
		// Quick&dirty: cast XML_Char directly to char
		to_doc->xml_element_start((const char*)el, (const char**)attr);
		return;
	}
	// AEGIS_DEBUG
	AEGIS_DEBUG(2, "Element start %s\n", el);
	for (int i = 0; attr[i]; i += 2) 
		AEGIS_DEBUG(2, "  Attribute %s=%s\n", attr[i], attr[i + 1]);
}


static void XMLCALL 
_exp_element_end (void* user_data, 
				  const XML_Char* el)
{
	c_xmldoc* to_doc = (c_xmldoc*) user_data;

	if (to_doc) {
		// Quick&dirty: cast XML_Char directly to char
		to_doc->xml_element_end ((char*) el);
		return;
	}
	// AEGIS_DEBUG
	AEGIS_DEBUG(2, "Element end %s\n", el);
}

static void XMLCALL 
_exp_character_data (void* user_data, 
					 const XML_Char* data, 
					 int len)
{
	c_xmldoc* to_doc = (c_xmldoc*) user_data;

	if (to_doc) {
		// Quick&dirty: cast XML_Char directly to char
		to_doc->xml_character_data ((char*) data, len);
		return;
	}

	if (data && len) {
		string tmp(data, len);
		AEGIS_DEBUG (2, "  Content '%s'", tmp.c_str());
	}
}


static void XMLCALL 
_exp_start_cdata (void* user_data)
{
	c_xmldoc* to_doc = (c_xmldoc*) user_data;
	to_doc->xml_cdata_start();
}


static void XMLCALL 
_exp_end_cdata (void* user_data)
{
	c_xmldoc* to_doc = (c_xmldoc*) user_data;
	to_doc->xml_cdata_end();
}


// Class members
// -------------

c_xmldoc::c_xmldoc()
    : trim_whitespace(false), 
      expat_parser(NULL), 
      root_node(NULL), 
      cur_node(NULL), 
      xml_str_buf(NULL)
{
    ;
}


void 
c_xmldoc::init_parser()
{
	if (expat_parser == NULL) {
		expat_parser = XML_ParserCreate(NULL);
        if (expat_parser) {
            XML_SetStartElementHandler(expat_parser, _exp_element_start);
            XML_SetEndElementHandler(expat_parser, _exp_element_end);
            XML_SetCharacterDataHandler(expat_parser, _exp_character_data);
            XML_SetCdataSectionHandler(expat_parser, _exp_start_cdata, _exp_end_cdata);
            XML_SetUserData (expat_parser, (void*) this);
        }
	}
}


void 
c_xmldoc::release_parser()
{
	if (expat_parser) {
		XML_ParserFree(expat_parser);
		expat_parser = NULL;
	}
}


void 
c_xmldoc::xml_parsing_error()
{
#if 0
	string errordesc;
	XML_Error rc = XML_GetErrorCode(expat_parser);
	int i, offset, size, prefix = 0;
	const char* input_context;
	const char* estring = XML_ErrorString(rc);

	errordesc.append(255, 
					 "XML error %d at line %d\n", 
					 rc, 
					 XML_GetCurrentLineNumber(expat_parser));

	input_context = XML_GetInputContext(expat_parser, &offset, &size);

	// Find the last newline before the error
	for (i = 0; i < offset; i++) {
		if (input_context[i] == '\n')
			prefix = i + 1;
	}

	// Truncate to the next newline after error
	for (i = offset; i < size; i++) {
		if (input_context[i] == '\n')
			size = i;
	}

	// Add context
	errordesc.append(1 + size, "%s\n", input_context);
	for (i = prefix; i < offset; i++) {
		if (input_context[i] == '\t')
			errordesc.append('\t', 1);
		else
			errordesc.append(' ', 1);
	}

	// Add a pointer to the error point
	errordesc.append(strlen(estring) + 3, "^ %s\n", estring);
#endif
    if (expat_parser) {
        AEGIS_ERROR("%s(%lu): %s", m_filename.c_str(), 
                    (unsigned long)XML_GetCurrentLineNumber(expat_parser),				
                    XML_ErrorString(XML_GetErrorCode(expat_parser)));
    }
}


static const int xml_parser_buffer_size = 0x1000;

void 
c_xmldoc::parse_file(const char* file_name)
{
	int fd = -1;

	release_content();
	release_parser ();
	init_parser();

    if (!expat_parser || !file_name)
        return;

	AEGIS_DEBUG(1, "%s: '%s'", __func__, file_name);

	m_filename.assign(file_name);
	fd = open(file_name, O_RDONLY);
	if (fd != -1) {
		for (;;) {
			int bytes_read;
			enum XML_Status status;

			void *buff = XML_GetBuffer(expat_parser, xml_parser_buffer_size);
			if (buff == NULL) {
				AEGIS_ERROR("cannot allocate XML parser buffer");
				goto end;
			}
			bytes_read = read(fd, buff, xml_parser_buffer_size);
			if (bytes_read < 0) {
				AEGIS_ERROR("cannot read '%s' (%s)", file_name, strerror(errno));
				goto end;

			} else if (bytes_read > 0) {
				status = XML_ParseBuffer(expat_parser, bytes_read, bytes_read == 0);
      
				switch (status) {
				case XML_STATUS_ERROR:
					xml_parsing_error();
					goto end;
				case XML_STATUS_SUSPENDED:
					close(fd);
					goto end;
				default:
					;
				}
			} else {
				break;
			}
		}
	}
 end:
	if (fd != -1)
		close(fd);
	cur_node = NULL;
	AEGIS_EXIT;
}


void 
c_xmldoc::parse_string(const char* xml_as_string, int length)
{
	release_content();
	init_parser();
    if (!expat_parser)
        return;

	if (length == 0)
		length = strlen(xml_as_string);
	if (length == 0) {
		root_node = NULL;
		cur_node = NULL;
		return;
	};
	if (XML_Parse(expat_parser, xml_as_string, length, true) != XML_STATUS_OK)
		xml_parsing_error();
	cur_node = NULL;
}


c_xmldoc::~c_xmldoc ()
{
	release_content();
	release_parser();
}


void 
c_xmldoc::release_content()
{
	if (root_node) {
		delete(root_node);
		root_node = NULL;
	}
	cur_node = NULL;
}


c_xmlnode* 
c_xmldoc::create(const char* root_node_name)
{
	root_node = new c_xmlnode(NULL, root_node_name);
	return(root_node);
}


c_xmlnode* 
c_xmldoc::root()
{
	return(root_node);
}


bool
c_xmldoc::save(const char* to_file)
{
	int fd = -1;
	ssize_t alen = 0;
	size_t slen = 0;
	string tmpname;

	if (NULL == to_file) {
		AEGIS_ERROR("invalid filename (null)");
		return false;
	}
	tmpname = to_file;
	string::size_type pos = tmpname.rfind("/");
	if (pos < tmpname.size())
		tmpname.insert(pos + 1, 1, '.');
	tmpname.append("~");

	/* Prevent fooling with a link.
	 */
	unlink(tmpname.c_str());

	fd = open(tmpname.c_str(), 
			  O_WRONLY | O_TRUNC | O_CREAT | O_SYNC,
			  0644);

	if (0 > fd) {
		AEGIS_ERROR("cannot open file '%s' for writing (%s)", 
                    tmpname.c_str(),
                    strerror(errno));
		return false;
	}

	/* TODO: for large files it would be more efficient to 
	 * write in pieces or use a memory mapping.
	 */
	string contents = as_string(true);
	const char* dta = contents.c_str();
	slen = strlen(dta);

	AEGIS_DEBUG(1, "%s: writing %lu bytes from %p", __func__,
				(unsigned long)slen, dta);
	alen = write(fd, dta, slen);

	if (0 > alen) {
		AEGIS_ERROR("failed to write to '%s' (%s)", 
					tmpname.c_str(), strerror(errno));
		close(fd);
		unlink(tmpname.c_str());
		return false;
	} else if ((size_t)alen < slen) {
		AEGIS_ERROR("write to '%s' truncated, %lu bytes omitted", 
					tmpname.c_str(), (unsigned long)slen - (unsigned long)alen);
		close(fd);
		unlink(tmpname.c_str());
		return false;
	}
	close(fd);

	AEGIS_DEBUG(1, "%s: written %lu bytes of XML to file '%s'", 
				__func__, (unsigned long)alen, tmpname.c_str());

	if (0 > rename(tmpname.c_str(), to_file)) {
		AEGIS_ERROR("%s: failed to rename '%s' to '%s' (%s)", 
					__func__, tmpname.c_str(), to_file, strerror(errno));
		unlink(tmpname.c_str());
		return false;
	}

	AEGIS_DEBUG(1, "%s: renamed to '%s'", __func__, to_file); 
	return true;
}

/*
 * expat catchers
 */
void 
c_xmldoc::xml_element_start(const char* element_name, 
							const char** attributes)
{
	c_xmlnode* tgt_node;

	if (cur_node == NULL) {
		tgt_node = new c_xmlnode(NULL, element_name);
		root_node = tgt_node;
	} else {
		// A node cannot have both textual and compound content
		tgt_node = cur_node->append_child(element_name);
	}
	for (int i = 0; attributes[i]; i += 2) 
		tgt_node->append_attribute (attributes[i], attributes[i + 1]);
	cur_node = tgt_node;
}


void 
c_xmldoc::xml_element_end(const char* element_name)
{
	// Sanity check
	if (!cur_node) {
		AEGIS_ERROR("document cannot start with element end");
		return;
	}
	if (strcmp(element_name, cur_node->name())) {
		AEGIS_ERROR("name mismatch '<%s>..</%s>", cur_node->name(), element_name);
		return;
	}
	
	if (trim_whitespace) 
		cur_node->trim_whitespace();

	cur_node = cur_node->parent();
}


void 
c_xmldoc::xml_character_data(const char* data, const int len)
{
	if (!cur_node || cur_node-> nbrof_children() > 0)
		return;
	cur_node->append_content(data, len);
}


void 
c_xmldoc::xml_cdata_start()
{
	if (cur_node)
		cur_node->set_cdata(true);	
}


void 
c_xmldoc::xml_cdata_end()
{
	if (cur_node)
		cur_node->set_cdata(false);	
}

static const char xml_hdr[] = "<?xml version=\"1.0\"?>\n";

string 
c_xmldoc::as_string (bool pretty_printing)
{
	string result(xml_hdr);

	if (root_node) {
		if (pretty_printing) {
			result.append(root_node->as_string(true, 4, 0));
		} else {
			result.append(root_node->as_string(false, 0, 0));
		}
	}
	return(result);
}
