/* -*- mode:c++; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
//#include "/usr/include/expat_external"
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#ifdef USE_CREDS
#include <sys/creds.h>
#endif

#define AEGIS_SHOW_ERRORS
#include <aegis_common.h>
#include <aegis_crypto.h>
#include <aegis_storage.h>

using namespace std;
using namespace aegis;

static void 
usage(const char* bin_name)
{
	printf(
		"Usage: %s [-I]  -r <token> -s <store-name:{G|S|P}{s|e}>\n"
        "  -m <filename>]\n"
		" -s to give the store name and attributes, separated by a colon\n"
		"    Attribute flags are visibility G=global, S=shared, P=private\n"
        "      and protection s=signed, e=encrypted\n"
		"    The default is a private, signed store if no attributes are given\n"
		" -m (default) the index file to merge\n"
        "    (Merge the index of a separate store in the current one)\n"
        " -I to show credentials of the process\n",
        bin_name
     );
}

#define UNDEFINED_VISIBILITY (storage::visibility_t)-1
#define UNDEFINED_PROTECTION (storage::protection_t)-1

#define BUFFER_SIZE 1024

static long fcopy(const char *dest, const char *source)
{
	FILE *d, *s;
	char *buffer;
	size_t incount;
	long totcount = 0L;

	s = fopen(source, "rb");
	if(s == NULL)
		return -1L;

	d = fopen(dest, "wb");
	if(d == NULL)
	{
		fclose(s);
		return -1L;
	}

	buffer = (char*)malloc(BUFFER_SIZE);
	if(buffer == NULL)
	{
		fclose(s);
		fclose(d);
		return -1L;
	}

	incount = fread(buffer, sizeof(char), BUFFER_SIZE, s);

	while(!feof(s))
	{
		totcount += (long)incount;
		fwrite(buffer, sizeof(char), incount, d);
		incount = fread(buffer, sizeof(char), BUFFER_SIZE, s);
	}

	totcount += (long)incount;
	fwrite(buffer, sizeof(char), incount, d);

	free(buffer);
	fclose(s);
	fclose(d);

	return totcount;
}

static bool verify_storage(storage *ss)
{
	storage::stringlist files;
	bool ret = true;
	
	ss->get_files(files);
	for (size_t i = 0; i < files.size(); i++) {
		if (!ss->verify_file(files[i])) {
			AEGIS_DEBUG(1, "File %s failed verification", files[i]);
			ret = false;
			break;
		}
	}
	ss->release(files);
	return ret;
}

static bool add_file(storage *ss, const char* filename)
{
    struct stat fs;

    AEGIS_DEBUG(1, "%s: %s", __func__, filename);

	/* Ignore links */
	if (0 > stat(filename, &fs)) {
		return true;
	}

	if (!ss->contains_file(filename)) {
		AEGIS_DEBUG(1, "%s: add missing file %s",
					__func__, filename);
		ss->add_file(filename);
		if(!ss->contains_file(filename)) {
			AEGIS_ERROR("Failed to add file %s",
						filename);
			return false;
		}
	}
	return true;
}

static bool extract_links(storage *ss,
						  std::map<std::string, std::string> &links)
{
	struct stat fs;
	storage::stringlist files;
	int size = ss->get_files(files);

	AEGIS_ENTER;

	links.clear();

	for(int i = 0; i < size; i++) {
		if (0 > ss->stat_file(files[i], &fs)) {
			AEGIS_ERROR("%s (CORRUPTED)", files[i]);
			return false;
		}

		if (S_ISLNK(fs.st_mode)) {
			string points_to;
			ss->read_link(files[i], points_to);
			links[files[i]] = points_to.c_str();
			/*if(!target->contains_link(filename)){
					target->add_link(filename, points_to.c_str());
					if (!target->contains_link(filename)) {
						AEGIS_ERROR("Failed to add link %s",
									points_to.c_str());
						return false;
					}
			}*/
		}
	}
	ss->release(files);
	return true;
}


int main(int argc, char* argv[])
{
	int a, rc = 0;
	string storagename, indexname, backupindex;
	char *flags = NULL, *filename = NULL;
	storage* ss = NULL;
	bool was_changed = false;
	storage::visibility_t storvis  = UNDEFINED_VISIBILITY;
	storage::protection_t storprot = UNDEFINED_PROTECTION;
    c_xmldoc xdoc;
	std::map<std::string, std::string> links;
	storage::stringlist files;
	int size, i;
 
	if (1 == argc) {
		usage(argv[0]);
		return(-1);
	}
	
    while (1) {
		a = getopt(argc, argv, ":s:m:I");
		if (a < 0) {
			break;
		}
		switch(a) 
		{
		case 's':
			flags = strchr(optarg, ':');
			if (flags) {
				storagename.assign(optarg, flags - optarg);
				flags++;
				while (*flags) {
					switch (*flags) 
						{
						case 'G':
							storvis = storage::vis_global;
							break;

						case 'S':
							storvis = storage::vis_shared;
							break;

						case 'P':
							storvis = storage::vis_private;
							break;

						case 'e':
							storprot = storage::prot_encrypted;
							break;
							
						case 's':
							storprot = storage::prot_signed;
							break;

						default:
							AEGIS_ERROR("Invalid attribute '%c'", *flags);
						}
					flags++;
				}
            }
            if (   UNDEFINED_VISIBILITY == storvis
                || UNDEFINED_PROTECTION == storprot)
            {
                fprintf(stderr, "ERROR: missing storage attributes in '%s'\n",
                        optarg);
                return -1;
            }
			break;

		case 'm':
            filename = optarg;
			break;

        case 'I':
#ifdef USE_CREDS
            {
                int i;
                creds_t creds;
                
                creds = creds_gettask(0);
                printf("Credentials:\n");
                for (i = 0; i < 1000; i++) {
                    creds_value_t cval;
                    creds_type_t cred = creds_list(creds, i, &cval);
                    if (CREDS_BAD != cred) {
                        char cred_name[256];
                        rc = creds_creds2str(cred, cval, cred_name, sizeof(cred_name));
                        if (0 <= rc)
                            printf("\t%s\n", cred_name);
                    } else
                        break;
                }
                creds_free(creds);
            }
            return 0;
#endif
            break;
   
		default:
			usage(argv[0]);
			return -1;
		}
	}
	
    if (0 == aegis_crypto_init()) {
		printf("Crypto init failed, exit\n%s\n", aegis_crypto_last_error_str());
		return -1;
    }


	if (NULL == filename) {
		AEGIS_ERROR("No index file given");
		return -1;
	}
	
	if (0 == access(filename, R_OK)) {
    } else if (0 == access(filename, F_OK)) {
        AEGIS_ERROR("'%s' is not readable by %d, fail", 
                    filename, (int)geteuid());
		return -1;
    }

	/* Open the existing storage to populate the class */
	if ("" != storagename) {
		ss = new storage(storagename.c_str(), NULL,
						storvis, storprot);
		if (!ss) {
			AEGIS_ERROR("cannot open/create storage (%s)", aegis_crypto_last_error_str());
			return -1;
		}
		if (ss->nbrof_files() == 0) {
            AEGIS_DEBUG(1, "Created new storage '%s' in %s.", 
                        ss->name(), ss->filename());
		} else {
			AEGIS_DEBUG(1, "Storage '%s' contains %d files.", 
						   ss->name(), ss->nbrof_files());
		}

		/* Test if we do have commit access */
		if(!ss->commit()){
			AEGIS_DEBUG(1, "Write access denied, merge cannot continue!");
			delete ss;
			return -1;
		}
	}

	if (NULL == ss) {
		AEGIS_ERROR("No storage given");
		return -1;
	}

	/* Get the full path of the index file */
	size = ss->get_ufiles(files);
	for (i = 0; i < size; i++) {
		if(!ss->contains_file(files[i])){
			indexname = files[i];
			break;
		}
	}

	ss->release(files);
	/* Save the list of files for adding after copy */
	size = ss->get_files(files);
	/* Save the links for adding after copy */
	if (false == extract_links(ss, links)){
		AEGIS_ERROR("Failed to extract links");
		goto cleanup;
	}

	/* The store needs to be closed because the index will be overwritten */
	delete ss;
	ss = NULL;

	backupindex = indexname + "-aegis-backup";
	/* Save the old index file in case it's needed for restore */
	rename(indexname.c_str(), backupindex.c_str());

	/* Try and copy the merge file to the index file location
	   so we can verify the signature */
	if (fcopy(indexname.c_str(), filename) < 0) {
		AEGIS_ERROR("Failed to copy the merge file to %s", indexname.c_str());
		rc = -1;
		goto cleanup;
	}

	ss = new storage(storagename.c_str(), NULL,
						storvis, storprot);
	if (!ss) {
		AEGIS_ERROR("cannot open/create storage (%s)", aegis_crypto_last_error_str());
		was_changed = true;
		goto safe_error;
	}
	
	if(0 == ss->nbrof_files()){
		AEGIS_DEBUG(1, "%s is empty, nothing to do", filename);
		was_changed = true;
		goto safe_error;
	}
	else{
		if( !verify_storage(ss)){
			AEGIS_ERROR("The merge file %s failed verification", filename);
			rc = -1;
			goto safe_error;
		}
		/* Add back the files from the old index(links are excluded) */
		for (i = 0; i < size; i++) {
			if( !add_file(ss, files[i])){
				AEGIS_ERROR("Merging of file %s failed", files[i]);
				/* Don't commit the changes if the merging failed */
				goto safe_error;
			}
		}

		/* Add back the links from the old index */
		for (
			map<string, string>::const_iterator ii = links.begin();
			ii != links.end();
			ii++
		) {
			ss->add_link(ii->first.c_str(), ii->second.c_str());
		}
	}
	/* In case the new contents cannot be commited revert to the old index */
	if (false == ss->commit()) {
		goto safe_error;
	}
	goto cleanup;
safe_error:
	/* We have to commit because the index file was overwritten
	   when verifying the integrity of the index to merge */
	AEGIS_DEBUG(1, "Reverting to the old index.");
	delete ss;
	ss = NULL;
	/* Copy back the old index file because the new one is not valid */
	rename(backupindex.c_str(), indexname.c_str());
cleanup:
	/* No matter what happened we can safely delete the old index */
	unlink(backupindex.c_str());
    if (ss) {
        delete ss;
        ss = NULL;
    }
    aegis_crypto_finish();

	return(rc);
}
