# -*- coding: utf-8 -*-

'''
Advanced Power Common UI for configuration
2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import gtk, hildon

class AbstractConfigUI():

  def __init__(self):
    self.classname = self.__class__.__name__

  def get_dialog_buttons(self):
    raise NotImplementedError('%s.get_dialog_buttons() must be implemented' % self.classname)

  def get_area(self):
    raise NotImplementedError('%s.get_area() must be implemented' % self.classname)

  def create_control(self, title, vtype, value=('False', False), options=set()):
    raise NotImplementedError('%s.create_control(title, vtype, value=(), options=()) must be implemented' % self.classname)

  def get_value(self, control, value, vtype):
    raise NotImplementedError('%s.get_value(control, value, vtype) must be implemented' % self.classname)

'''
Maemo 4 UI
'''
class ConfigUIMaemo4(AbstractConfigUI):

  def get_dialog_buttons(self):
    return (gtk.STOCK_SAVE, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_DELETE_EVENT)

  def get_area(self):
    sa = gtk.ScrolledWindow()
    sa.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    return sa

  def create_control(self, title, vtype, value=('False', False), options=set()):
    control = None
    if vtype == type(None):
      pass
    elif vtype == bool:
      control_ts = gtk.CheckButton()
      control_ts.set_active(value[1])
      control = hildon.Caption(None, title, control_ts, None, hildon.CAPTION_OPTIONAL)
    elif vtype in (list, tuple, dict):
      control_list_store = gtk.ListStore(str, vtype)
      control_ts = gtk.ComboBox(control_list_store)
      cell = gtk.CellRendererText()
      control_ts.pack_start(cell, True)
      control_ts.add_attribute(cell, 'text',0)
      control = hildon.Caption(None, title, control_ts, None, hildon.CAPTION_OPTIONAL)
      control_list_store.append([value[0], value[1]])
      control_ts.set_active(0)
    else:
      control_list_store = gtk.ListStore(str, vtype)
      control_ts = gtk.ComboBox(control_list_store)
      cell = gtk.CellRendererText()
      control_ts.pack_start(cell, True)
      control_ts.add_attribute(cell, 'text',0)
      control = hildon.Caption(None, title, control_ts, None, hildon.CAPTION_OPTIONAL)
      if options:
        for option in options:
          control_list_store.append([config_option[0], config_option[1]])
        if not value in options:
          control_list_store.append([value[0], value[1]])
        control_ts.set_active(options.index(value))
      else:
        control_list_store.append([value[0], value[1]])
        control_ts.set_active(0)
    return control

  def get_value(self, control, value, vtype):
    if vtype == bool:
      value = vtype(control.get_child().get_active())
    elif vtype in (list, tuple, dict):
      pass
    else:
      ts = control.get_child()
      model = ts.get_model()
      citer = ts.get_active_first()
      value = vtype(model.get_value(citer, 1))
    return value

'''
Maemo 5 UI
'''
class ConfigUIMaemo5(AbstractConfigUI):

  def get_dialog_buttons(self):
    return (gtk.STOCK_SAVE, gtk.RESPONSE_OK)

  def get_area(self):
    return hildon.PannableArea()

  def create_control(self, title, vtype, value=('False', False), options=set()):
    control = None
    if vtype == type(None):
      pass
    elif vtype == bool:
      control = hildon.CheckButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
      control.set_label(title)
      control.set_active(value[1])
    elif vtype in (list, tuple, dict):
      control = hildon.PickerButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      control.set_title(title)
      control.set_alignment(0, 0.5, 0, 0)
      control.set_value(str(value[0]))
    else:
      control = hildon.PickerButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      control.set_title(title)
      control.set_alignment(0, 0.5, 0, 0)
      control_ts = hildon.TouchSelector()
      control.set_selector(control_ts)
      control_list_store = gtk.ListStore(str, vtype)
      control_ts.append_text_column(control_list_store, True)
      control_ts.set_column_selection_mode(hildon.TOUCH_SELECTOR_SELECTION_MODE_SINGLE)
      if options:
        for option in options:
          control_list_store.append([option[0], option[1]])
        if not value in options:
          control_list_store.append([value[0], value[1]])
        control.set_active(options.index(value))
      else:
        control_list_store.append([value[0], value[1]])
        control.set_active(0)
    return control

  def get_value(self, control, value, vtype):
    if vtype == bool:
      value = vtype(control.get_active())
    elif vtype in (list, tuple, dict):
      pass
    else:
      ts = control.get_selector()
      model = ts.get_model(0)
      citer = model.get_iter_first()
      ts.get_selected(0, citer)
      value = vtype(model.get_value(citer, 1))
    return value
