/*
 * 3g2g-mode-selection-applet: Lets you switch the network access mode
 *
 * Copyright (C) 2009 Faheem Pervez <trippin1@gmail.com>. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *      
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *      
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

/* mode-selection-applet.c */

#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>

/* libcsnet D-Bus definitions */
#define NETWORK_BUS_NAME		"com.nokia.phone.net"
#define NETWORK_INTERFACE		"Phone.Net"
#define NETWORK_PATH			"/com/nokia/phone/net"
#define NETWORK_PATH_GET_RADIO_ACCESS	"get_selected_radio_access_technology"
#define NETWORK_PATH_SET_RADIO_ACCESS	"set_selected_radio_access_technology"

#define SSC_DBUS_NAME  "com.nokia.phone.SSC"
#define SSC_DBUS_IFACE SSC_DBUS_NAME
#define SSC_DBUS_PATH  "/com/nokia/phone/SSC"
#define SSC_MODEM_STATE_SIG  "modem_state_changed_ind"

enum
{
	MODE_DUAL,
	MODE_GSM,
	MODE_3G,
	NUM
};

/* Start of GObject boilerplate shit */
#define MODE_TYPE_SELECTION_APPLET (mode_selection_applet_get_type ())

#define MODE_SELECTION_APPLET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionApplet))

#define MODE_SELECTION_APPLET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletClass))

#define MODE_IS_SELECTION_APPLET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
						MODE_TYPE_SELECTION_APPLET))

#define MODE_IS_SELECTION_APPLET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), \
						MODE_TYPE_SELECTION_APPLET))

#define MODE_SELECTION_APPLET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletClass))

#define MODE_SELECTION_APPLET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletPrivate))

typedef struct _ModeSelectionApplet ModeSelectionApplet;
typedef struct _ModeSelectionAppletClass ModeSelectionAppletClass;
typedef struct _ModeSelectionAppletPrivate ModeSelectionAppletPrivate;

struct _ModeSelectionApplet
{
	HDStatusMenuItem parent;

	ModeSelectionAppletPrivate *priv;
};

struct _ModeSelectionAppletClass
{
	HDStatusMenuItemClass parent_class;
};

struct _ModeSelectionAppletPrivate
{
	GtkWidget *hbox_applet_contents;
	GtkWidget *toggle_button[NUM];

	gulong signal_id[NUM];

	DBusGConnection *dbus_conn;
	DBusGProxy *phonet_proxy;
	DBusGProxy *ssc_proxy;
};
/* End of GObject boilerplate shit */

const struct
{
	const gchar *msgid;
	const gchar *fallback_string;
} items[] =
{
	{"conn_va_phone_network_mode_b", "Dual"},
	{"conn_va_phone_network_mode_2g", "GSM"},
	{"conn_va_phone_network_mode_3g", "3G"},
};

HD_DEFINE_PLUGIN_MODULE (ModeSelectionApplet, mode_selection_applet, HD_TYPE_STATUS_MENU_ITEM)

static inline G_CONST_RETURN gchar* g_dgettext_localised_or_english (const gchar* domainname, const gchar* msgid, const gchar *fallback_string)
{
	const gchar *retval = g_dgettext (domainname, msgid);

	return G_UNLIKELY (!strcmp (retval, msgid)) ? fallback_string : retval;
}

static guint mode_selection_applet_get_selected_radio_access_technology (const ModeSelectionApplet *plugin)
{
	guchar network_access;
	gint some_next_value;

	dbus_g_proxy_call (plugin->priv->phonet_proxy, NETWORK_PATH_GET_RADIO_ACCESS, NULL, G_TYPE_INVALID, G_TYPE_UCHAR, &network_access, G_TYPE_INT, &some_next_value, G_TYPE_INVALID);

	return (guint) network_access;
}

static void mode_selection_applet_set_selected_radio_access_technology (const ModeSelectionApplet *plugin, const guchar mode)
{
	dbus_g_proxy_call_no_reply (plugin->priv->phonet_proxy, NETWORK_PATH_SET_RADIO_ACCESS, G_TYPE_UCHAR, mode, G_TYPE_INVALID);
}

static void mode_selection_applet_set_button_value (const ModeSelectionApplet *plugin)
{
	register guint i = 0;

	const guint radio_access = mode_selection_applet_get_selected_radio_access_technology (plugin);

	for (i = 0; i < NUM; i++)
	{
		g_signal_handler_block (plugin->priv->toggle_button[i], plugin->priv->signal_id[i]);
		if (i == radio_access)
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (plugin->priv->toggle_button[radio_access]), TRUE);
		g_signal_handler_unblock (plugin->priv->toggle_button[i], plugin->priv->signal_id[i]);
	}
}

static void mode_selection_applet_on_button_toggled (GtkToggleButton *button, const ModeSelectionApplet *plugin)
{
	if (gtk_toggle_button_get_active (button))
	{
		const guint mode = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (button), "toggled-button-no."));
		gtk_widget_hide (gtk_widget_get_toplevel (GTK_WIDGET (button)));

		mode_selection_applet_set_selected_radio_access_technology (plugin, mode);
	}
}

static void mode_selection_applet_update_on_phone_state_changed (const DBusGProxy *object G_GNUC_UNUSED, const gchar *status G_GNUC_UNUSED, const ModeSelectionApplet *plugin)
{
	mode_selection_applet_set_button_value (plugin);
}

static void mode_selection_applet_finalize (GObject *object)
{
	ModeSelectionApplet *plugin = MODE_SELECTION_APPLET (object);

	if (plugin->priv->ssc_proxy)
	{
		dbus_g_proxy_disconnect_signal (plugin->priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_CALLBACK (mode_selection_applet_update_on_phone_state_changed), plugin);
		g_object_unref (plugin->priv->ssc_proxy);
	}

	if (plugin->priv->phonet_proxy)
		g_object_unref (plugin->priv->phonet_proxy);

	/* if (plugin->priv->dbus_conn)
		dbus_g_connection_unref (plugin->priv->dbus_conn); */ /* Allegedly, hd_status_plugin_item_get_dbus_g_connection () returns a shared connection. */

	G_OBJECT_CLASS (mode_selection_applet_parent_class)->finalize (object);
}

static void mode_selection_applet_class_init (ModeSelectionAppletClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = (GObjectFinalizeFunc) mode_selection_applet_finalize;

	g_type_class_add_private (klass, sizeof (ModeSelectionAppletPrivate));
}

static void mode_selection_applet_class_finalize (ModeSelectionAppletClass *klass G_GNUC_UNUSED)
{
}

static void mode_selection_applet_setup_buttons (const guint i, ModeSelectionApplet *plugin)
{
	plugin->priv->toggle_button[i] = !i ? hildon_gtk_radio_button_new (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH, NULL) : hildon_gtk_radio_button_new_from_widget (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH, GTK_RADIO_BUTTON (plugin->priv->toggle_button[i-1]));
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (plugin->priv->toggle_button[i]), FALSE);
	gtk_button_set_label (GTK_BUTTON (plugin->priv->toggle_button[i]), g_dgettext_localised_or_english ("osso-connectivity-ui", items[i].msgid, items[i].fallback_string));
	gtk_label_set_ellipsize (GTK_LABEL (GTK_BIN (plugin->priv->toggle_button[i])->child), PANGO_ELLIPSIZE_END);
	g_object_set_data (G_OBJECT (plugin->priv->toggle_button[i]), "toggled-button-no.", GUINT_TO_POINTER (i));
	plugin->priv->signal_id[i] = g_signal_connect (plugin->priv->toggle_button[i], "toggled", G_CALLBACK (mode_selection_applet_on_button_toggled), plugin);
	gtk_container_add (GTK_CONTAINER (plugin->priv->hbox_applet_contents), plugin->priv->toggle_button[i]);
}

static void mode_selection_applet_init (ModeSelectionApplet *plugin)
{
	guint i;
	plugin->priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);
	if (G_UNLIKELY (!plugin->priv))
		return;

	plugin->priv->dbus_conn = hd_status_plugin_item_get_dbus_g_connection (HD_STATUS_PLUGIN_ITEM (&plugin->parent), DBUS_BUS_SYSTEM, NULL);
	if (G_UNLIKELY (!plugin->priv->dbus_conn))
		return;

	plugin->priv->phonet_proxy = dbus_g_proxy_new_for_name (plugin->priv->dbus_conn, NETWORK_BUS_NAME, NETWORK_PATH, NETWORK_INTERFACE);
	if (G_UNLIKELY (!plugin->priv->phonet_proxy))
		return;

	plugin->priv->ssc_proxy = dbus_g_proxy_new_for_name (plugin->priv->dbus_conn, SSC_DBUS_NAME, SSC_DBUS_PATH, SSC_DBUS_IFACE);

	/* Button setup */
	plugin->priv->hbox_applet_contents = gtk_hbutton_box_new ();
	hildon_gtk_widget_set_theme_size (plugin->priv->hbox_applet_contents, HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);

	for (i = 0; i < NUM; i++)
		mode_selection_applet_setup_buttons (i, plugin);

	gtk_container_add (GTK_CONTAINER (plugin), plugin->priv->hbox_applet_contents);

	mode_selection_applet_set_button_value (plugin);

	if (G_LIKELY (plugin->priv->ssc_proxy))
	{
		dbus_g_proxy_add_signal (plugin->priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_TYPE_STRING, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (plugin->priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_CALLBACK (mode_selection_applet_update_on_phone_state_changed), plugin, NULL);
	}

	gtk_widget_show_all (GTK_WIDGET (plugin));
}

