/*
 * Copyright 2010  Open Source & Linux Lab (OSLL)  maemo@osll.spb.ru
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * The advertising clause requiring mention in adverts must never be included.
 */

/*! ---------------------------------------------------------------
 * $Id$
 *
 * \file Game.cpp
 * \brief Game implementation
 *
 * File description
 *
 * PROJ: 15shki
 * ---------------------------------------------------------------- */

#include "Game.h"
#include <QObject>
#include <QDebug>
#include <stdlib.h>
#include <time.h>
#include <QMessageBox>
#include <QPalette>
#include <QTimer>
#include <math.h>
#include <QMessageBox>
#include <QSound>
#include <Phonon>

#ifdef SYMBIAN_OS
//necessary header files for backlight in Symbian
#include <e32std.h>
#endif

Game::Game(QWidget *parent) :
        QWidget(parent),
        accelerometer(),
        gamePlace(this),
        sound(QString(SOUNDS_DIRECTORY) + QString("click.wav"))
{
    this->setFixedWidth(gamePlace.width());
    this->setFixedHeight(gamePlace.height());
    timer.setInterval(1000);

    mediaObject = new Phonon::MediaObject(this);
    mediaObject->setCurrentSource(Phonon::MediaSource(
                                  QString(SOUNDS_DIRECTORY) + 
                                  QString("click.wav")));
    audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    Phonon::createPath(mediaObject, audioOutput);

    connect(&model, SIGNAL(shifted(int, int, int)),
                    this, SLOT(onModelShifted(int, int, int)));
    connect(&model, SIGNAL(finished()), this, SLOT(onModelFinished()));
    connect(&timer, SIGNAL(timeout()), SLOT(onTimerTimeout()));

    QTimer *acc_timer = new QTimer(this);
    acc_timer->setInterval(ACC_UPDATE_INTERVAL);
    connect(acc_timer, SIGNAL(timeout()), this, SLOT(onAccelerometerChanged()));
    accelerometer.start();
    acc_timer->start();

    gameStarted = 0;
    movements = 0;
    
    /* keep backlight on with QtMobility*/
#ifndef SYMBIAN_OS
    systemScreenSaver.setScreenSaverInhibit();
#endif    

    emit movementsChanged(movements);
}

Game::~Game()
{}

int Game::solvableArray(int *array)
{
    int summ = 0;
    for (int i = 0; i < 14; i++)
    {
        for (int j = i + 1; j < 15; j++)
        {
            if (array[i] > array[j])
                summ++;
        }
    }
    qDebug("%d", summ);
    return (summ % 2);
}

void Game::newGame()
{
    int array[16];
    int numbers[16];
    int number;
    do
    {
        for (int i = 1; i < 16; i++)
        {
            array[i] = -1;
            numbers[i] = 0;
        }
        array[0] = 0;
        // fill array
        srand(time(NULL));
        qsrand(qrand());
        for (int i = 1; i < 16; i++)
        {
            do
            {
                number = qrand() % 15 + 1;
                if (numbers[number] == 0)
                {
                    array[i] = number;
                    numbers[number] = 1;
                }

            } while(array[i] == -1);
        }
        array[0] = array[15];
        array[15] = 0;
    } while (solvableArray(array));
    model.setArray(array);
    gamePlace.drawPlace(array);
    gameStarted = 1;
    movements = 0;
    duration = 0;
    timer.start();
    emit movementsChanged(movements);

    mediaObject->play();
}

void Game::setScreen(int width, int height)
{
        int sceneWidth = GamePlace::getNumberWidth() * 4 +
                         GamePlace::getDistanceBetweenNumbers() * 5 +
                         SCENE_EXTRA_WIDTH;
        int sceneHeight = GamePlace::getNumberHeight() * 4 +
                          GamePlace::getDistanceBetweenNumbers() * 5 +
                          SCENE_EXTRA_HEIGHT;
        qreal scaleWidth = (qreal) width / (qreal) sceneWidth;
        qreal scaleHeight = (qreal) height / (qreal) sceneHeight;
        qreal scale = (scaleWidth < scaleHeight) ? scaleWidth : scaleHeight;
        gamePlace.scale(scaleWidth, scaleHeight);
        gamePlace.setFixedWidth(scaleWidth * sceneWidth);
        gamePlace.setFixedHeight(scaleHeight * sceneHeight);
        this->setFixedWidth(scaleWidth * sceneWidth);
        this->setFixedHeight(scaleHeight * sceneHeight);
        gamePlace.rotate(0);
}

void Game::onModelShifted(int number, int oldField, int newField)
{
	//QSound::play("click.wav");
	sound.stop();
	sound.play();
    gamePlace.shiftNumber(number, oldField, newField);
    //mediaObject->play();
    

    /* keep backlight on */
#ifdef SYMBIAN_OS
    /* symbian realization */
    User::ResetInactivityTime();
#endif
}

void Game::onModelFinished()
{
    if (gameStarted)
    {
        qDebug("finished");
        timer.stop();

        QString min, sec;
        if ((duration / 60) < 10)
            min = "0" + QString::number(duration / 60);
        else
            min = QString::number(duration / 60);

        if ((duration % 60) < 10)
            sec = "0" + QString::number(duration % 60);
        else
            sec = QString::number(duration % 60);

        QMessageBox::information(this, "15shki",
                                 tr("Game's finished. You've done ") + \
                                 QString::number(movements + 1) + \
                                 tr(" movements. Spent time: ") + \
                                 min + ":" + sec);
        gameStarted = 0;

        emit finished();
        //movements = 0;
        //emit movementsChanged(movements);
    }
}

void Game::onAccelerometerChanged()
{
    if (gameStarted)
    {
        qreal x = accelerometer.reading()->x();
        qreal y = accelerometer.reading()->y();
        qreal z = accelerometer.reading()->z();

        qreal beta = 180 - (atan2(z, -y) * 180/M_PI);
        qreal gamma = (atan2(-z, -x) * 180/M_PI) + 90;
        //QMessageBox::information(this, "15shki", QString::number(beta) + QString(" ") + QString::number(gamma));
        if (model.setNewAngles(beta, gamma))
        {
            movements++;
            emit movementsChanged(movements);
        }
    }
}
//#else
//void Game::onAccelerometerChanged(QAccelerometerValue value)
//{
//    if (gameStarted)
//    {
//        //qDebug() << value.getBeta() << " " << value.getGamma();
//        if (model.setNewAngles(value.getBeta(), value.getGamma()))
//        {
//            movements++;
//            emit movementsChanged(movements);
//        }
//    }
//}
//#endif

void Game::onTimerTimeout()
{
    duration++;
    emit durationChanged(duration);
}



/* Next stuff may be usefull for testing */

void Game::accLeftImitator()
{
    qDebug("Imitate left shaking");
    if (model.shakeLeft())
    {
        movements++;
        emit movementsChanged(movements);
    }
}

void Game::accRightImitator()
{
    qDebug("Imitate right shaking");
    if (model.shakeRight())
    {
        movements++;
        emit movementsChanged(movements);
    }
}

void Game::accUpImitator()
{
    qDebug("Imitate up shaking");
    if (model.shakeUp())
    {
        movements++;
        emit movementsChanged(movements);
    }
}

void Game::accDownImitator()
{
    qDebug("Imitate down shaking");
    if (model.shakeDown())
    {
        movements++;
        emit movementsChanged(movements);
    };
}




/* ===[ End of file $HeadURL$ ]=== */
