/*
 * parse.c - parsing functions for X-Sys
 * by mike9010
 * Copyright (C) 2003, 2004, 2005 Michael Shoup
 * Copyright (C) 2005, 2006, 2007 Tony Vroon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <time.h>
#include <dirent.h>
#include <sys/types.h>

#include "match.h"
#include "hwmon.h"
#include "xsys.h"

int xs_parse_cpu(char *model, char *vendor, double *freq, char *cache, unsigned int *count)
{
	char buffer[bsize], *pos;
	FILE *fp = fopen("/proc/cpuinfo", "r");
	pos = NULL;
	if(fp == NULL)
		return 1;
	if(count != NULL) *count = 0;
	strcpy(cache,"unknown\0");
	
	while(fgets(buffer, bsize, fp) != NULL)
	{
		find_match_char(buffer, "Processor", model);
		find_match_char(buffer, "Hardware", vendor);
		find_match_double(buffer, "BogoMIPS", freq);
		find_match_char(buffer, "CPU architecture", cache);
		find_match_int(buffer, "Features", count);
	}
	*count = *count + 1;
	fclose(fp);
	
	return 0;
}

int xs_parse_uptime(int *weeks, int *days, int *hours, int *minutes, int *seconds)
{
	char buffer[bsize];
	long long uptime = 0;
	FILE *fp = fopen("/proc/uptime", "r");
	if(fp == NULL)
		return 1;

	if(fgets(buffer, bsize, fp) != NULL)
		uptime = strtol(buffer, NULL, 0);
	
	*seconds = uptime%60;
	*minutes = (uptime/60)%60;
	*hours   = (uptime/3600)%24;
	*days    = (uptime/86400)%7;
	*weeks   = uptime/604800;
	
	fclose(fp);
	
	return 0;
}

int xs_parse_os(char *user, char *host, char *kernel)
{
	struct utsname osinfo;
	char hostn[bsize], *usern = getenv("USER");
	
	if(uname(&osinfo)<0)
		return 1;	
	if(gethostname(hostn, bsize)<0)
		return 1;
	
	strncpy(user, usern, bsize);
	strcpy(host, hostn);
	snprintf(kernel, bsize, "%s %s %s %s", osinfo.sysname, osinfo.nodename, osinfo.release, osinfo.machine);
	
	return 0;
}

int xs_parse_netdev(const char *device, unsigned long long *bytes_recv, unsigned long long *bytes_sent)
{
	FILE *fp;
	char buffer[bsize], *pos;
	int i;
	
	fp=fopen("/proc/net/dev", "r");
	if(fp==NULL)
		return 1;
	
	while(fgets(buffer, bsize, fp) != NULL)
	{
		for(i=0; isspace(buffer[i]); i++);
		if(strncmp(device, &buffer[i], strlen(device)) == 0) break;
	}
	fclose(fp);
	pos = strstr(buffer, ":");
	pos++;
	*bytes_recv = strtoull(pos, &pos, 0);

	for(i=0;i<7;i++) strtoull(pos, &pos, 0);
	
	*bytes_sent = strtoull(pos, NULL, 0);

	return 0;
}

int xs_parse_df(const char *mount_point, char *result)
{
	FILE *pipe;
	char buffer[bsize], *pos;
	unsigned long long total_k=0, free_k=0;
	int i=0;
	
	pipe = popen("df -k", "r");
	if(pipe==NULL)
		return 1;
	
	while(fgets(buffer, bsize, pipe) != NULL)
	{
		/* Skip over pseudo-filesystems and description line */
		if(isalpha(buffer[0]))
			continue;
		
		for(pos=buffer; !isspace(*pos); pos++);
		for(;isspace(*pos);pos++);
		if(mount_point == NULL)
		{
			total_k += strtoull(pos, &pos, 0);
			strtoull(pos, &pos, 0);
			free_k += strtoull(pos, &pos, 0);
			continue;
		}
		total_k = strtoull(pos, &pos, 0);
		strtoull(pos, &pos, 0);
		free_k = strtoull(pos, &pos, 0);
		strtoull(pos, &pos, 0);
		for(;isspace(*pos);pos++);
		for(;*pos!='/';pos++);
		for(i=0;*(buffer+i)!='\n';i++);
		*(buffer+i)='\0';
		
		if(strncasecmp(mount_point, "ALL", 3)==0)
		{
			char *tmp_buf = pretty_freespace(pos, &free_k, &total_k);
			strcat(tmp_buf, " | ");
			strcat(result, tmp_buf);
			free(tmp_buf);
		}
		else if(strncmp(mount_point, pos, strlen(mount_point)) == 0)
		{
			char *tmp_buf = pretty_freespace(mount_point, &free_k, &total_k);
			strncpy(result, tmp_buf, bsize);
			free(tmp_buf);
			break;
		}
		else snprintf(result, bsize, "Mount point %s not found!", mount_point);
	}
	
	if(mount_point != NULL && strncasecmp(mount_point, "ALL", 3)==0)
		*(result+strlen(result)-3) = '\0';
	
	if(mount_point == NULL)
	{
		char *tmp_buf = pretty_freespace("Total", &free_k, &total_k);
		strncpy(result, tmp_buf, bsize);
		free(tmp_buf);
	}
	pclose(pipe);
	return 0;
}

int xs_parse_meminfo(unsigned long long *mem_tot, unsigned long long *mem_free, int swap)
{
	FILE *fp;
        char buffer[bsize];
	unsigned long long freemem = 0, buffers = 0, cache = 0;
        *mem_tot = 0;
        *mem_free = 0;

        if((fp = fopen("/proc/meminfo", "r")) == NULL)
                return 1;

        while(fgets(buffer, bsize, fp) != NULL)
        {
                if(!swap)
                {
			find_match_ll(buffer, "MemTotal:", mem_tot);
			find_match_ll(buffer, "MemFree:", &freemem);
			find_match_ll(buffer, "Buffers:", &buffers);
			find_match_ll(buffer, "Cached:", &cache);
                }
                else
                {
			find_match_ll(buffer, "SwapTotal:", mem_tot);
			find_match_ll(buffer, "SwapFree:", mem_free);
                }
        }
	if (!swap) {
		*mem_free = freemem + buffers + cache;
	}
        fclose(fp);
        return 0;
}

int xs_parse_distro(char *name)
{
	FILE *fp = NULL;
	char buffer[bsize], *pos = NULL;
	
	if((fp = fopen("/etc/lsb_release", "r")) != NULL)
	{
		char id[bsize], codename[bsize], release[bsize];
		strcpy(id, "?");
		strcpy(codename, "?");
		strcpy(release, "?");
		while(fgets(buffer, bsize, fp) != NULL)
		{
			find_match_char(buffer, "DISTRIB_ID", id);
			find_match_char(buffer, "DISTRIB_CODENAME", codename);
			find_match_char(buffer, "DISTRIB_RELEASE", release);
		}
		snprintf(buffer, bsize, "%s \"%s\" %s", id, codename, release);
	}
	else if((fp = fopen("/etc/make.conf", "r")) != NULL)
	{
		char keywords[bsize];
		while(fgets(buffer, bsize, fp) != NULL)
			find_match_char(buffer, "ACCEPT_KEYWORDS", keywords);
		if (strstr(keywords, "\"") == NULL)
			snprintf(buffer, bsize, "Gentoo Linux (stable)");
		else
			snprintf(buffer, bsize, "Gentoo Linux %s", keywords);
	}		
	else if((fp = fopen("/etc/redhat-release", "r")) != NULL)
		fgets(buffer, bsize, fp);
	else if((fp = fopen("/etc/slackware-version", "r")) != NULL)
		fgets(buffer, bsize, fp);
	else if((fp = fopen("/etc/mandrake-release", "r")) != NULL)
		fgets(buffer, bsize, fp);
	else if((fp = fopen("/etc/osso_software_version", "r")) != NULL)
	{
		char release[bsize];
		fgets(release, bsize, fp);
		snprintf(buffer, bsize, "Maemo %s", release);
	}
	else if((fp = fopen("/etc/debian_version", "r")) != NULL)
	{
		char release[bsize];
		fgets(release, bsize, fp);
		snprintf(buffer, bsize, "Debian %s", release);
	}
	else if((fp = fopen("/etc/SuSE-release", "r")) != NULL)
		fgets(buffer, bsize, fp);
	else if((fp = fopen("/etc/turbolinux-release", "r")) != NULL)
		fgets(buffer, bsize, fp);
	else
		snprintf(buffer, bsize, "Unknown Distro");
	if(fp != NULL) fclose(fp);
	
	pos=strchr(buffer, '\n');
	if(pos != NULL) *pos = '\0';
	strcpy(name, buffer);
	return 0;
}

int xs_parse_hwmon_chip(char *chip)
{
	if (!hwmon_chip_present())
		return 1;
	else
		get_hwmon_chip_name(chip);
	return 0;
}

int xs_parse_hwmon_temp(char *temp, unsigned int *sensor)
{
	unsigned int *value;
	float celsius;
        value = malloc(sizeof(int));

	if (!hwmon_chip_present())
		return 1;
	else
		get_hwmon_temp(value, sensor);
		celsius = (float)*value;
		snprintf(temp, bsize, "%.1fC", celsius/1000.0);
		free(value);
	return 0;
}
