/*
  WifiInfo - Show current Wifi sttaus
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "StatusLinux.h"

#include <errno.h>
#include <sys/ioctl.h>
#include <netinet/ether.h>
#include <arpa/inet.h>
#include <fcntl.h>

extern "C" {
  #include <linux/if.h>
  #include <linux/wireless.h>
}

#include <cstring>
#include <cstdlib>
#include <fstream>
#include <iostream>

#include <Lum/Base/String.h>

#include <Lum/OS/Display.h>

bool StatusLinux::UpdateStatus()
{
         std::ifstream file;
         std::string   line,interface;
         int           handle;
  struct ifreq         ifr;
  struct iwreq         data;
  struct iw_statistics stats;
  struct iw_range      range;
         char          buffer[1024];

  Lum::OS::Guard<Lum::OS::Mutex> guard(mutex);

  ClearStatus();

  file.open("/proc/net/wireless",std::ios::in);

  if (!file) {
    std::cerr << "Cannot read /proc/net/wireless" << std::endl;
    return false;
  }

  if (std::getline(file,line) && std::getline(file,line)) {
    while (std::getline(file,line)) {
      std::string::size_type start,end;
      std::string            entry;

      start=line.find_first_not_of(" ");
      end=line.find(":");

      if (start!=std::string::npos && end!=std::string::npos) {
        entry=line.substr(start,end-start);

        if (GetDefaultInterface().empty() || entry==GetDefaultInterface()) {
          interface=entry;
        }
      }
    }
  }

  file.close();

  if (interface.empty()) {
    std::cerr << "Cannot find valid interface in /proc/net/wireless" << std::endl;
    return false;
  }

  handle=socket(AF_INET,SOCK_DGRAM,0);

  if (handle<0) {
    std::cerr << "Cannot open socket:" << strerror(errno) << std::endl;
    return false;
  }

  std::cout << "Reading interface '" << interface << "'..." << std::endl;

  strcpy(ifr.ifr_name,interface.c_str());

  if (ioctl(handle,SIOCGIFADDR,&ifr)>=0) {

    ip=Lum::Base::StringToWString(inet_ntoa(((struct sockaddr_in*)&ifr.ifr_addr)->sin_addr));
  }

  if (ioctl(handle,SIOCGIFHWADDR,&ifr)>=0) {

    mac=Lum::Base::StringToWString(ether_ntoa((ether_addr*)ifr.ifr_hwaddr.sa_data));
  }

  // Type

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWNAME,&data)>=0) {
    typeName=Lum::Base::StringToWString(data.u.name);
  }

  // Mode

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWMODE,&data)>=0) {
    switch (data.u.mode) {
    case IW_MODE_AUTO:
      type=typeAuto;
      break;
    case IW_MODE_ADHOC:
      type=typeAdHoc;
      break;
    case IW_MODE_INFRA:
      type=typeInfrastructure;
      break;
    case IW_MODE_MASTER:
      type=typeMaster;
      break;
    case IW_MODE_REPEAT:
      type=typeRepeater;
      break;
    case IW_MODE_SECOND:
      type=typeSecond;
      break;
    case IW_MODE_MONITOR:
      type=typeMonitor;
      break;
#if WIRELESS_EXT>=22 && defined(IW_MODE_MESH)
    case IW_MODE_MESH:
      type=typeMesh;
      break;
#endif
    default:
      type=typeNone;
      break;
    }
  }

  // ESSID

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  memset(buffer,0,1024);
  data.u.essid.pointer=buffer,
  data.u.essid.length=1024;

  if (ioctl(handle,SIOCGIWESSID,&data)>=0) {
    // TODO: Encode special characters!
    essid=Lum::Base::StringToWString(buffer);
  }

  // Bitrate

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWRATE,&data)>=0) {
    bitrate=data.u.bitrate.value;
  }

  // Quality

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  data.u.data.pointer=(caddr_t)&range;
  data.u.data.length=sizeof(iw_range);
  data.u.data.flags=0;

  if (ioctl(handle,SIOCGIWRANGE,&data)>=0) {

    // Frequency
    memset(&data,0,sizeof(struct iwreq));
    sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
    if (ioctl(handle,SIOCGIWFREQ,&data)>=0) {
      if (data.u.freq.m<1000 && data.u.freq.e==0) {
        channel=data.u.freq.m;
      }
      else {
        for (size_t i =0; i<range.num_frequency; i++) {
          if (data.u.freq.m==range.freq[i].m && data.u.freq.e==range.freq[i].e) {
            channel=(int)range.freq[i].i;
            break;
          }
        }
      }
    }

    // Signal/Noise ratio
    memset(&data,0,sizeof(struct iwreq));
    sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
    data.u.data.pointer=(caddr_t)&stats;
    data.u.data.length=sizeof(iw_statistics);
    data.u.data.flags=1;

    if (ioctl(handle,SIOCGIWSTATS,&data)>=0) {
      if (!(IW_QUAL_QUAL_INVALID & data.u.data.flags)) {
        signal=stats.qual.level;
        noise=stats.qual.noise;
        maxQuality=range.max_qual.qual;

        if (maxQuality<=0 || maxQuality>100) {
          maxQuality=0;
          quality=GetQualityFromSignalNoiseDbm(signal-0x100,noise-0x100);
        }
        else {
          quality=100*stats.qual.qual/maxQuality;
        }
      }
      else {
        std::cerr << "Quality not set in stats" << std::endl;
      }
    }
    else {
      std::cerr << "Cannot read interface stats: " << strerror(errno) << std::endl;
    }
  }
  else {
    std::cerr << "Cannot read interface value ranges: " << strerror(errno) << std::endl;
  }

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWAP,&data)>=0) {
    ether_addr *in=(ether_addr*)&data.u.ap_addr.sa_data;

    accesspoint=Lum::Base::StringToWString(ether_ntoa(in));
  }

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  data.u.power.flags=0;
  if (ioctl(handle,SIOCGIWPOWER,&data)>=0) {
    powerSaving=data.u.power.disabled ? powerSavingOff : powerSavingOn;
  }
  else {
    powerSaving=powerSavingUnknown;
  }

  close(handle);

  Lum::OS::display->QueueActionForEventLoop(statusChangedAction);

  return true;
}

bool StatusLinux::SupportsNetworkRetrieval() const
{
  return true;
}

bool StatusLinux::UpdateNetworks()
{
         std::ifstream file;
         std::string   line,interface;
         int           handle;
  struct ifreq         ifr;
  struct iwreq         data;

  Lum::OS::Guard<Lum::OS::Mutex> guard(mutex);

  networks.clear();

  file.open("/proc/net/wireless",std::ios::in);

  if (!file) {
    return false;
  }

  if (std::getline(file,line) && std::getline(file,line) && std::getline(file,line)) {
    std::string::size_type start,end;

    start=line.find_first_not_of(" ");
    end=line.find(":");

    if (start!=std::string::npos && end!=std::string::npos) {
      interface=line.substr(start,end-start);
    }
  }

  file.close();

  if (interface.empty()) {
    return false;
  }

  handle=socket(AF_INET,SOCK_DGRAM,0);

  if (handle<0) {
    return false;
  }

  strcpy(ifr.ifr_name,interface.c_str());

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  data.u.essid.pointer=(caddr_t)NULL;
  data.u.essid.flags=IW_SCAN_THIS_ESSID;
  data.u.essid.length=0;

  if (ioctl(handle,SIOCSIWSCAN,&data)>=0) {
    int    counter=3;
    size_t buflen=IW_SCAN_MAX_DATA;
    char   *buffer=(char*)malloc(IW_SCAN_MAX_DATA);
    bool   success=false;

    memset(buffer,0,IW_SCAN_MAX_DATA);

    while (counter>0) {
      std::cerr << "Try reading scan results..." << counter << std::endl;
      memset(&data,0,sizeof(struct iwreq));
      sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
      data.u.data.pointer=buffer;
      data.u.data.flags=0;
      data.u.data.length=IW_SCAN_MAX_DATA;

      if (ioctl(handle,SIOCGIWSCAN,&data)<0) {
        if (errno==E2BIG) {
          std::cerr << "To much data!" << std::endl;
          if (data.u.data.length>buflen) {
            buflen=data.u.data.length;
          }
          else {
            buflen*=2;
          }

          char* newBuffer=(char*)realloc(buffer,buflen);
          if (newBuffer==NULL) {
            std::cerr << "Cannot allocate buffer of size " << buflen << " for scan result!" << std::endl;
            free(buffer);
            buffer=NULL;
            break;
          }
          else {
            buffer=newBuffer;
          }
        }
        else if (errno==EAGAIN) {
          std::cout << "We have to wait for the result..." << std::endl;
          counter--;
          sleep(1);
        }
        else {
          std::cout << "Error: " << errno << " text: " << strerror(errno) << std::endl;
          break;
        }
      }
      else {
        std::cerr << "Sucessfully scanned..." << std::endl;
        success=true;
        break;
      }
    }

    if (success) {
      std::cout << "successfully read data block of size " << data.u.data.length << "..." << std::endl;
      if (data.u.data.length>0) {
        size_t          readLength=0;
        Status::Network network;
        bool            filled=false;

        while (readLength+IW_EV_LCP_LEN<data.u.data.length) {
          struct iw_event iwe;

          memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN);

          //std::cout << "command: " << iwe.cmd << " data length: " << iwe.len << std::endl;

          if (iwe.len<=IW_EV_LCP_LEN) {
            std::cerr << " Invalid event length, quitting..." << std::endl;
            break;
          }

          if (iwe.cmd==SIOCGIWAP) {
            std::cout << "------------" << std::endl;
            if (filled) {
              if (!network.essid.empty()) {
                networks.push_back(network);
              }
              network.essid.clear();
              filled=false;
            }
          }
          else if (iwe.cmd==SIOCGIWESSID) {
            memcpy((char*)&iwe+IW_EV_LCP_LEN+IW_EV_POINT_OFF,
                   ((char*)data.u.data.pointer)+readLength+IW_EV_LCP_LEN,IW_EV_POINT_LEN);

            iwe.u.essid.pointer=(char*)data.u.data.pointer+readLength+IW_EV_LCP_LEN+IW_EV_POINT_LEN-IW_EV_POINT_OFF;

            if (iwe.u.essid.pointer!=NULL && iwe.u.essid.length>0) {
              std::string tmp;

              if (((const char*)iwe.u.essid.pointer)[iwe.u.essid.length-1]=='\0') {
                tmp.assign((const char*)iwe.u.essid.pointer,iwe.u.essid.length-1);
              }
              else {
                tmp.assign((const char*)iwe.u.essid.pointer,iwe.u.essid.length);
              }

              network.essid=Lum::Base::StringToWString(tmp);
              std::cout << "ESSID: '" << tmp << "'" << std::endl;
            }
            else {
              std::cerr << "No essid!" << std::endl;
            }
            filled=true;
          }
          else if (iwe.cmd==SIOCGIWMODE) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,iwe.len);

            //network.type=iwe.u.mode;
            //std::cout << "Mode: " << network.mode << std::endl;
          }
          else if (iwe.cmd==SIOCGIWRATE) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN+IW_EV_PARAM_LEN);

            network.bitrate=iwe.u.bitrate.value;
            std::cout << "Bitrate: " << network.bitrate << std::endl;
          }
          else if (iwe.cmd==SIOCGIWFREQ) {
            //std::cout << "Frequence: " << std::endl;
          }
          else if (iwe.cmd==SIOCGIWENCODE) {
            //std::cout << "Encoding: " << std::endl;
          }
          else if (iwe.cmd==IWEVQUAL) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN+IW_EV_QUAL_LEN);

            network.quality=(100*iwe.u.qual.qual)/maxQuality;
            std::cout << "Quality: " << network.quality << " " << maxQuality << std::endl;
          }
          else if (iwe.cmd==IWEVCUSTOM) {
            std::cout << "Custom: " << std::endl;
          }
          else if (iwe.cmd==IWEVGENIE) {
            std::cout << "Genie: " << std::endl;
          }
          else {
            std::cout << "Read command " << iwe.cmd << " length " << iwe.len << std::endl;
          }

          readLength+=iwe.len;
        }

        std::cout << "Scanning done!" << std::endl;

        if (filled) {
          if (!network.essid.empty()) {
            networks.push_back(network);
          }
          network.essid.clear();
          filled=false;
        }
      }
    }
    else {
      std::cout << "Timeout!" << std::endl;
    }

    free(buffer);
  }
  else if (errno!=EPERM) {
    std::cerr << "Cannot initiate scan: " << strerror(errno) << std::endl;
  }
  else {
    std::cerr << "Scanning network not allowed!" << std::endl;
  }

  close(handle);

  Lum::OS::display->QueueActionForEventLoop(networksChangedAction);

  return true;
}
