/*
  WifiInfo - Show current Wifi sttaus
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/OS/Probe.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/AppInfo.h>
#include <Lum/Dialog.h>
#include <Lum/FuelGauge.h>
#include <Lum/Label.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#if defined(CAN_USE_STATUS_HILDON)
  #include "StatusHildon.h"
#elif defined(CAN_USE_STATUS_LINUX)
  #include "StatusLinux.h"
#else
  #include "StatusNone.h"
#endif

#include <iostream>
static Lum::AppInfo info;

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::StringRef    type;
  Lum::Model::StringRef    essid;
  Lum::Model::StringRef    accesspoint;
  Lum::Model::StringRef    client;
  Lum::Model::DoubleRef    quality;
  Lum::Model::StringRef    signalNoise;
  Lum::Model::IntRef       refreshTime;
  Lum::Model::ListTableRef tableModel;

  Lum::Model::ActionRef    statusTimer;
  Lum::Model::ActionRef    networksTimer;

  Lum::Model::ActionRef    aboutAction;

#if defined(CAN_USE_STATUS_HILDON)
    StatusHildon           status;
#elif defined(CAN_USE_STATUS_LINUX)
    StatusLinux            status;
#else
    StatusNone             status;
#endif

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textView;

    textView=new Lum::TextValue();
    textView->SetFlex(true,false);
    textView->SetWidth(Lum::Base::Size::stdCharWidth,size);
    textView->SetModel(model);

    return textView;
  }

  void RequestStatus()
  {
    Lum::OS::display->RemoveTimer(statusTimer);

    if (!status.UpdateStatus()) {
      status.type=Status::typeNone;
      UpdateStatus();
    }

    Lum::OS::display->AddTimer(refreshTime->Get(),0,statusTimer);
  }

  void RequestNetworks()
  {
    Lum::OS::display->RemoveTimer(networksTimer);

    if (status.SupportsNetworkRetrieval() && !status.UpdateNetworks()) {
      tableModel->Clear();
      UpdateNetworks();
    }

    Lum::OS::display->AddTimer(refreshTime->Get(),0,networksTimer);
  }

  void UpdateStatus()
  {
    if (status.type==Status::typeNone || status.bitrate==0) {
      type->Set(L"-");
      essid->Set(L"-");
      accesspoint->Set(L"-");
      client->Set(L"-");
      quality->SetNull();
      signalNoise->Set(L"-");
      return;
    }

    std::wstring tmr;

    essid->Set(status.essid);

    tmr=status.typeName;

    tmr.append(L" ");

    switch (status.type) {
    case Status::typeAuto:
      tmr.append(L"Auto");
      break;
    case Status::typeAdHoc:
      tmr.append(L"Ad-Hoc");
      break;
    case Status::typeInfrastructure:
      tmr.append(L"Infrastructure");
      break;
    case Status::typeMaster:
      tmr.append(L"Master");
      break;
    case Status::typeRepeater:
    tmr.append(L"Repeater");
      break;
    case Status::typeSecond:
      tmr.append(L"Secondary master");
      break;
    case Status::typeMonitor:
      tmr.append(L"Passive monitor");
      break;
    case Status::typeMesh:
      tmr.append(L"Mesh");
      break;
    default:
      tmr.append(L"-");
    }

    tmr.append(L" ");

    if (status.bitrate>1000000 && status.bitrate==(status.bitrate/1000000)*1000000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000000));
      tmr.append(L"Mb/s");
    }
    else if (status.bitrate>1000 && status.bitrate==(status.bitrate/1000)*1000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000));
      tmr.append(L"Kb/s");
    }
    else {
    tmr.append(Lum::Base::NumberToWString(status.bitrate));
      tmr.append(L"b/s");
    }

    type->Set(tmr);

    accesspoint->Set(status.accesspoint);

    client->Set(std::wstring(status.ip)+L" ("+status.mac+L")");

    if (status.quality>=0 && status.quality<=100) {
      quality->Set(status.quality);
    }
    else {
      quality->SetNull();
    }

    signalNoise->Set(Lum::Base::NumberToWString(status.signal-0x100)+L" dBm / "+Lum::Base::NumberToWString(status.noise-0x100)+L" dBm");
  }

  void UpdateNetworks()
  {
    std::list<Status::Network>::const_iterator iter;

    // First try to update existing entries in the current table
    iter=status.networks.begin();
    while (iter!=status.networks.end()) {
      size_t i;

      // Find the right entry
      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->essid==tableModel->GetEntry(i)->GetString(1)) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) { // Found an entry => Update
        Lum::Model::ListTable::Entry    *entry=tableModel->GetEntry(i);
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;

        gauge=dynamic_cast<Lum::FuelGauge*>(entry->GetObject(2));
        data=dynamic_cast<Lum::Model::Double*>(gauge->GetModel());

        data->Set(iter->quality);
      }
      else { // No entry => add
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;
        Lum::Model::ListTable::StdEntry *entry;

        gauge=new Lum::FuelGauge();
        gauge->SetFlex(true,true);
        //gauge->SetMinWidth(Lum::Base::Size::stdCharWidth,30);

        data=new Lum::Model::Double();
        data->SetRange(0,100);
        data->Set(iter->quality);
        gauge->SetModel(data);

        entry=new Lum::Model::ListTable::StdEntry(tableModel);
        entry->SetString(1,iter->essid);
        entry->SetObject(2,gauge);

        tableModel->Append(entry);
      }

      ++iter;
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;

    while(i<=tableModel->GetRows()) {
      std::wstring                               essid=tableModel->GetEntry(i)->GetString(1);
      std::list<Status::Network>::const_iterator iter;

      // Find essid from table entry in network list
      iter=status.networks.begin();
      while (iter!=status.networks.end()) {
        if (iter->essid==essid) {
          break;
        }

        ++iter;
      }

      if (iter==status.networks.end()) { // table entry is not in list
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
  }

public:
  MyWindow()
  : type(new Lum::Model::String()),
    essid(new Lum::Model::String()),
    accesspoint(new Lum::Model::String()),
    client(new Lum::Model::String()),
    quality(new Lum::Model::Double()),
    signalNoise(new Lum::Model::String()),
    refreshTime(new Lum::Model::Int(5)),
    tableModel(new Lum::Model::ListTable(2)),
    statusTimer(new Lum::Model::Action()),
    networksTimer(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action())
  {
    quality->SetRange(0,100);
    refreshTime->SetRange(0,10);

    AttachModel(refreshTime);
    AttachModel(statusTimer);
    AttachModel(networksTimer);
    AttachModel(GetOpenedAction());
    AttachModel(aboutAction);

    AttachModel(status.statusChangedAction);
    AttachModel(status.networksChangedAction);
  }

  virtual void PreInit()
  {
    Lum::FuelGauge   *fuelgauge;
    Lum::Label       *label;
    Lum::Panel       *hPanel,*vPanel;
    Lum::Slider      *slider;

    Lum::Text        *text;
    std::wstring     tmp;

    hPanel=new Lum::HPanel();
    hPanel->SetFlex(true,true);

    label=new Lum::Label();
    label->SetFlex(true,true);

    label->AddLabel(L"ESSID:",CreateTextValue(essid,32));
    label->AddLabel(L"Type:",CreateTextValue(type,42));
    label->AddLabel(L"AP:",CreateTextValue(accesspoint,17));
    label->AddLabel(L"Client:",CreateTextValue(client,28));

    fuelgauge=new Lum::FuelGauge();
    fuelgauge->SetFlex(true,false);
    fuelgauge->SetModel(quality);
    label->AddLabel(L"Quality:",fuelgauge);
    label->AddLabel(L"Signal/Noice:",CreateTextValue(signalNoise,28));

    if (status.SupportsNetworkRetrieval()) {
      Lum::Table                         *table;
      Lum::Model::HeaderRef              headerModel;
      Lum::Model::SingleLineSelectionRef selection;

      selection=new Lum::Model::SingleLineSelection;

      headerModel=new Lum::Model::HeaderImpl();
      headerModel->AddColumn(L"ESSID",Lum::Base::Size::stdCharWidth,15,true);
      headerModel->AddColumn(L"Quality",Lum::Base::Size::stdCharWidth,10);

      table=new Lum::Table();
      table->SetFlex(true,true);
      table->SetHeaderModel(headerModel);
      table->SetModel(tableModel);
      //table->SetSelection(selection);
      //table->SetSelectionAction(showDetail);
      //table->SetDoubleClickAction(showDetail);
      table->SetShowHeader(true);
      table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,4);
      table->GetTableView()->SetAutoFitColumns(true);
      table->GetTableView()->SetAutoHSize(true);
      label->AddLabel(L"Networks:",table);
    }

    hPanel->Add(label);

    hPanel->Add(new Lum::HSpace());

    vPanel=new Lum::VPanel();
    vPanel->SetFlex(false,true);

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetAlignment(Lum::Text::centered);
    text->SetText(L"Poll (s)");
    vPanel->Add(text);

    vPanel->Add(new Lum::VSpace());

    slider=new Lum::VSlider();
    slider->SetFlex(false,true);
    slider->SetScale(true);
    slider->SetModel(refreshTime);

    vPanel->Add(slider);

    hPanel->Add(vPanel);

    SetMain(hPanel);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->AddMenuItemAction(_ld(menuHelpAbout),aboutAction);

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      UpdateStatus();
      RequestStatus();

      UpdateNetworks();
      RequestNetworks();

    }
    else if (model==statusTimer && statusTimer->IsFinished()) {
      RequestStatus();
    }
    else if (model==networksTimer && networksTimer->IsFinished()) {
      RequestNetworks();
    }
    else if (model==refreshTime && IsOpen()) {
      RequestStatus();
      RequestNetworks();
    }
    else if (model==status.statusChangedAction && status.statusChangedAction->IsFinished()) {
      UpdateStatus();
    }
    else if (model==status.networksChangedAction && status.networksChangedAction->IsFinished()) {
      UpdateNetworks();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MyWindow>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"How is your Wifi?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

