/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DataInput.h"

#include <Lum/Base/L10N.h>

#include <Lum/Array.h>
#include <Lum/Button.h>
#include <Lum/DateSelect.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>
#include <Lum/TimeSelect.h>

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Values.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include "Configuration.h"
#include "Trend.h"
#include "Util.h"

#include <iostream>

class WeightEdit : public Lum::Dlg::ValuesInput
{
public:
  WeightEdit(Lum::Model::Time* time,
             Lum::Model::String* weight)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"WEIGHT_EDIT_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,6);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"WEIGHT_EDIT_LABEL_WEIGHT",L"Weight ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",weightUnit->Get());
    v.label=tmp;
    v.model=weight;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool EditWeight(Lum::OS::Window* parent,
                         const std::wstring& caption,
                         const std::wstring& text,
                         const std::wstring& note,
                         Lum::Model::Time* time,
                         Lum::Model::String* weight)
  {
    assert(time!=NULL && weight!=NULL);

    WeightEdit *dlg;
    bool       res;

    dlg=new WeightEdit(time,weight);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);
    dlg->SetNote(note);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodEdit : public Lum::Dlg::ValuesInput
{
public:
  FoodEdit(Lum::Model::Time* time,
           Lum::Model::String* amount,
           const std::wstring& unit)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"FOOD_REGISTER_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"FOOD_REGISTER_LABEL_AMOUNT",L"Amount ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",unit);
    v.label=tmp;
    v.model=amount;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool EditFood(Lum::OS::Window* parent,
                           const std::wstring& caption,
                           const std::wstring& text,
                           Lum::Model::Time* time,
                           Lum::Model::String* amount,
                           const std::wstring& unit)
  {
    assert(time!=NULL && amount!=NULL);

    FoodEdit *dlg;
    bool     res;

    dlg=new FoodEdit(time,amount,unit);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class EntriesEntry : public Lum::Model::ListTable::Entry
{
public:
  ::Entry *entry;

public:
  EntriesEntry(Lum::Model::ListTable *table, ::Entry* entry)
  : Lum::Model::ListTable::Entry(table),
    entry(entry)
  {
    // no code
  }

  std::wstring GetString(size_t column) const
  {
    if (column==1) {
      std::wstring res;

      res=Lum::Base::NumberToWString(entry->time.GetHour());
      res.append(L":");
      if (entry->time.GetMinute()<10) {
        res.append(L"0");
      }
      else {
        res.append(Lum::Base::NumberToWString(entry->time.GetMinute()/10));
      }
      res.append(Lum::Base::NumberToWString(entry->time.GetMinute()%10));

      return res;
    }
    else if (column==2) {
      if (entry->item.empty()) {
        return _(L"LABEL_WEIGHT",L"Weight");
      }
      else {
        return entry->item+L" ("+AmountToString(entry->amount)+L" "+entry->unit+L")";
      }
    }
    else if (column==3) {
      if (entry->item.empty()) {
        return AmountToString(entry->amount)+L" "+weightUnit->Get();
      }
      else {
        return ValueToString((((Value)entry->amount)*entry->value)/10)+L" "+valueUnit->Get();
      }
    }

    assert(false);
  }

  bool IsGreater(const Entry* other, size_t column)
  {
    return entry->time>dynamic_cast<const EntriesEntry*>(other)->entry->time;
  }
};

DataInput::DataInput()
 : prevDayAction(new Lum::Model::Action()),
   todayAction(new Lum::Model::Action()),
   nextDayAction(new Lum::Model::Action()),

   //addFoodAction(new Lum::Model::Action()),
   addWeightAction(new Lum::Model::Action()),
   removeItemAction(new Lum::Model::Action()),
   editItemAction(new Lum::Model::Action()),

   averageDiff(new Lum::Model::String(L"")),
   summary(new Lum::Model::String(L"")),
   entries(new Lum::Model::ListTable()),
   entrySelection(new Lum::Model::SingleLineSelection()),

   trend(new Lum::Model::Int(Trend::none))
{
  date->SetToCurrent();

  Observe(date);

  Observe(prevDayAction);
  Observe(todayAction);
  Observe(nextDayAction);

  //Observe(addFoodAction);
  Observe(addWeightAction);
  Observe(removeItemAction);
  Observe(editItemAction);

  Observe(trendType);
  Observe(trendDirection);
  Observe(refValue);
}

void DataInput::CalcSize()
{
  Lum::DateSelect       *dateSelect;
  Lum::Menu             *menu;
  Lum::Panel            *et;
  Lum::Panel            *ds;
  Lum::Panel            *horiz;
  Lum::Panel            *vert;
  Lum::Table            *table;
  Lum::TextValue        *text;
  Lum::Model::HeaderRef headerModel;
  Trend                 *trend;

  // Date select

  ds=Lum::HPanel::Create(true,false);
  ds->Add(Lum::Button::Create(L"_<",prevDayAction));
  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  dateSelect=new Lum::DateSelect();
  dateSelect->SetModel(date);
  ds->Add(dateSelect);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));
  ds->Add(Lum::Button::Create(L"_>",nextDayAction));
  ds->AddSpace();

  trend=new Trend();
  trend->SetModel(this->trend);
  ds->Add(trend);

  ds->AddSpace();

  text=Lum::TextValue::Create(averageDiff,Lum::TextValue::left,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,6);
  ds->Add(text);

  ds->AddSpace();

  text=Lum::TextValue::Create(summary,Lum::TextValue::right,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,9);
  ds->Add(text);

  ds->AddSpace(true);
  /*
  button=new Lum::Button(L"+F");
  button->SetModel(addFoodAction);
  ds->Add(button);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));*/

  ds->Add(Lum::Button::Create(L"+",addWeightAction));
  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));
  ds->Add(Lum::Button::Create(L"-",removeItemAction));

  // Entries table

  et=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_TIME",L"Time"),Lum::Base::Size::stdCharWidth,5);
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_ITEM",L"Item"),Lum::Base::Size::stdCharWidth,15,true);
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_VALUE",L"Value"),Lum::Base::Size::stdCharWidth,8);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeaderModel(headerModel);
  table->SetModel(entries);
  table->SetSelection(entrySelection);
  table->SetDoubleClickAction(editItemAction);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->GetTableView()->SetAutoHSize(true);
  et->Add(table);

  menu=new Lum::Menu();
  menu->SetParent(GetWindow());
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_ADD",L"_Add weight..."),addWeightAction);
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_EDIT",L"_Edit entry..."),editItemAction);
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_REMOVE",L"_Remove entry"),removeItemAction);
  table->SetMenu(menu->GetWindow());

  // Layout comonents

  horiz=Lum::HPanel::Create();

  vert=Lum::VPanel::Create(true,true);

  vert->Add(ds);
  vert->AddSpace();
  vert->Add(et);

  horiz->Add(vert);

  container=horiz;

  Component::CalcSize();
}

void DataInput::UpdateEntries()
{
  if (!entries.Valid() || !date.Valid()) {
    return;
  }

  entries->Clear();

  // Entries

  Day *day=data.GetDay(date->Get());


  if (day==NULL) {
    return;
  }

  entries->Off();

  for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
    entries->Append(new EntriesEntry(entries,&*iter));
  }

  entries->Sort(1);

  entries->On();
}

void DataInput::UpdateSummary()
{
  Value current=0;

  Day *day=data.GetDay(date->Get());

  if (day!=NULL) {
    for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
      if (!iter->item.empty()) {
        current+=((Value)iter->amount)*iter->value;
      }
    }
  }

  if (current>0) {
    current=(current+5)/100; // Round
  }
  else {
    current=(current-5)/100; // Round
  }

  summary->Set(Lum::Base::NumberToWString(current)+
               L"/"+
               Lum::Base::NumberToWString(((int)maxValue->Get())/10-current));

  if (trendType->Get()==trendRefRel) {
    if (day!=NULL) {
      for (std::list<Entry>::reverse_iterator iter=day->entries.rbegin();
           iter!=day->entries.rend();
           ++iter) {
        if (iter->item.empty()) {
          if (iter->amount>=refValue->Get()) {
            averageDiff->Set(AmountToString(iter->amount-refValue->Get())+weightUnit->Get());
            if (trendDirection->Get()!=1) {
              trend->Set(Trend::up);
            }
            else {
              trend->Set(Trend::down);
            }
          }
          else {
            averageDiff->Set(std::wstring(L"-")+AmountToString(refValue->Get()-iter->amount)+weightUnit->Get());
            if (trendDirection->Get()!=1) {
              trend->Set(Trend::down);
            }
            else {
              trend->Set(Trend::up);
            }
          }
          return;
        }
      }
    }
    trend->Set(Trend::none);
    averageDiff->Set(L"?");
  }
  else if (trendType->Get()==trendTwoWeeks) {
    Lum::Base::Calendar d=date->Get();
    size_t weightLastWeek=0;
    size_t weightLastWeekCount=0;
    size_t weightPriorWeek=0;
    size_t weightPriorWeekCount=0;

    for (size_t i=1; i<=7; i++) {
      Day *day=data.GetDay(d);

      if (day!=NULL) {
        for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
          if (iter->item.empty()) {
            weightLastWeek+=iter->amount;
            weightLastWeekCount++;
          }
        }
      }

      d.AddDays(-1);
    }

    for (size_t i=1; i<=7; i++) {
      Day *day=data.GetDay(d);

      if (day!=NULL) {
        for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
          if (iter->item.empty()) {
            weightPriorWeek+=iter->amount;
            weightPriorWeekCount++;
          }
        }
      }

      d.AddDays(-1);
    }

    if (weightLastWeekCount==0 || weightPriorWeekCount==0) {
      averageDiff->Set(L"?");
      trend->Set(Trend::none);
    }
    else {
      weightLastWeek=weightLastWeek/weightLastWeekCount;
      weightPriorWeek=weightPriorWeek/weightPriorWeekCount;

      if (weightPriorWeek<=weightLastWeek) {
        averageDiff->Set(AmountToString(weightLastWeek-weightPriorWeek)+weightUnit->Get());
        if (trendDirection->Get()!=1) {
          trend->Set(Trend::up);
        }
        else {
          trend->Set(Trend::down);
        }
      }
      else {
        averageDiff->Set(std::wstring(L"-")+AmountToString(weightPriorWeek-weightLastWeek)+weightUnit->Get());
        if (trendDirection->Get()!=1) {
          trend->Set(Trend::down);
        }
        else {
          trend->Set(Trend::up);
        }
      }
    }
  }
  else {
    averageDiff->Set(L"???");
    trend->Set(Trend::none);
  }
}

void DataInput::AddWeight()
{
  Lum::Model::StringRef value;
  size_t                amount;

  value=new Lum::Model::String(L"");

  if (!WeightEdit::EditWeight(this->GetWindow(),
                              _(L"WEIGHT_EDIT_TITLE",L"My weight is..."),
                              _(L"WEIGHT_EDIT_TEXT",
                                L"Please enter your weight at the given time!"),
                              _(L"WEIGHT_EDIT_NOTE",
                                L"(The weight should be entered as a decimal number\n"
                                L"with a maximum of one digit after the colon.)"),
                              timestamp,
                              value)) {
    return;
  }

  if (value->IsNull()) {
    return;
  }

  if (!StringToAmount(value->Get(),amount)) {
    return;
  }

  Entry e;

  e.time=timestamp->Get();
  e.item=L"";
  e.unit=L"";
  e.value=0;
  e.amount=amount;

  entries->Append(new EntriesEntry(entries,data.AddEntry(date->Get(),e)));

  entries->Sort(1);

  UpdateSummary();
}

void DataInput::RemoveItem()
{
  data.DeleteEntry(date->Get(),
                   dynamic_cast<EntriesEntry*>(entries->GetEntry(entrySelection->GetLine()))->entry);
  entries->Delete(entrySelection->GetLine());

  UpdateSummary();
}

void DataInput::EditItem()
{
  EntriesEntry *entry=(EntriesEntry*)entries->GetEntry(entrySelection->GetLine());

  if (entry->entry->item.empty()) {
    Lum::Model::StringRef value;
    size_t                amount;

    timestamp->Push();
    timestamp->Set(entry->entry->time);
    value=new Lum::Model::String(AmountToString(entry->entry->amount));

    if (!WeightEdit::EditWeight(this->GetWindow(),
                                _(L"WEIGHT_EDIT_TITLE",L"My weight is..."),
                                _(L"WEIGHT_EDIT_TEXT",
                                  L"Please enter your weight at the given time!"),
                                _(L"WEIGHT_EDIT_NOTE",
                                  L"(The weight should be entered as a decimal number\n"
                                  L"with a maximum of one digit after the colon.)"),
                                timestamp,
                                value)) {
      timestamp->Pop();
      return;
    }

    if (value->IsNull()) {
      timestamp->Pop();
      return;
    }

    if (!StringToAmount(value->Get(),amount)) {
      timestamp->Pop();
      return;
    }

    entry->entry->amount=amount;
    entry->entry->time=timestamp->Get();
    entries->RedrawRow(entrySelection->GetLine());

    entries->Sort(1);

    timestamp->Pop();
  }
  else {
    Lum::Model::StringRef value;
    size_t                amount;
    std::wstring          t1,t2;

    timestamp->Push();
    timestamp->Set(entry->entry->time);
    value=new Lum::Model::String(AmountToString(entry->entry->amount));

    t1=_(L"FOOD_REGISTER_TITLE",L"I have consumed $food...");
    Lum::Base::L10N::Substitute(t1,L"$food",entry->entry->item);
    t2=_(L"FOOD_REGISTER_TEXT",L"Please enter the amount of $food\nyou have consumed!");
    Lum::Base::L10N::Substitute(t2,L"$food",entry->entry->item);
    if (!FoodEdit::EditFood(this->GetWindow(),
                            t1,t2,timestamp,value,entry->entry->unit)) {

      timestamp->Pop();
      return;
    }

    if (value->IsNull() || value->Empty()) {
      timestamp->Pop();
      return;
    }

    if (!StringToAmount(value->Get(),amount)) {
      timestamp->Pop();
      return;
    }

    entry->entry->time=timestamp->Get();
    entry->entry->amount=amount;
    entries->RedrawRow(entrySelection->GetLine());

    entries->Sort(1);

    timestamp->Pop();
  }

  UpdateSummary();
}

void DataInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==date) {
    UpdateEntries();
    UpdateSummary();
  }
  if (model==prevDayAction && prevDayAction->IsFinished()) {
    Lum::Base::Calendar d=date->Get();

    d.AddDays(-1);
    date->Set(d);
  }
  else if (model==todayAction && todayAction->IsFinished()) {
    date->SetToCurrent();
  }
  else if (model==nextDayAction && nextDayAction->IsFinished()) {
    Lum::Base::Calendar d=date->Get();

    d.AddDays(1);
    date->Set(d);
  }
  else if (model==addWeightAction && addWeightAction->IsFinished()) {
    AddWeight();
  }
  else if (model==removeItemAction && removeItemAction->IsFinished()) {
    if (entrySelection->HasSelection()) {
      RemoveItem();
    }
  }
  else if (model==editItemAction && editItemAction->IsFinished()) {
    if (entrySelection->HasSelection()) {
      EditItem();
    }
  }
  else if (model==trendType || model==trendDirection || model==refValue) {
    UpdateSummary();
  }

  Lum::Component::Resync(model,msg);
}

void DataInput::InitializeGUI()
{
  UpdateEntries();
  UpdateSummary();
}
