/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FoodInput.h"

#include <Lum/Base/L10N.h>

#include <Lum/Array.h>
#include <Lum/Button.h>
#include <Lum/DateSelect.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>
#include <Lum/TimeSelect.h>

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Values.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include "Configuration.h"
#include "Trend.h"
#include "Util.h"

#include <iostream>

class FoodEdit : public Lum::Dlg::ValuesInput
{
public:
  FoodEdit(Lum::Model::String* name,
           Lum::Model::String* unit,
           Lum::Model::String* value)
  {
    Lum::String  *string;
    Value        v;
    std::wstring tmp;

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,20);

    v.label=_(L"FOOD_EDIT_LABEL_NAME",L"Name:");
    v.model=name;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,8);

    v.label=_(L"FOOD_EDIT_LABEL_UNIT",L"Unit:");
    v.model=unit;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"FOOD_EDIT_LABEL_VALUE",L"Value ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
    v.label=tmp;
    v.model=value;
    v.control=string;
    v.validator=new ValueValidator();
    AddValue(v);
  }

  static bool EditFood(Lum::OS::Window* parent,
                       const std::wstring& caption,
                       const std::wstring& text,
                       Lum::Model::String* name,
                       Lum::Model::String* unit,
                       Lum::Model::String* value)
  {
    assert(name!=NULL && unit!=NULL && value!=NULL);

    FoodEdit *dlg;
    bool     res;

    dlg=new FoodEdit(name,unit,value);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodRegister : public Lum::Dlg::ValuesInput
{
public:
  FoodRegister(Lum::Model::Time* time,
          Lum::Model::String* amount,
          const std::wstring& unit)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"FOOD_REGISTER_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"FOOD_REGISTER_LABEL_AMOUNT",L"Amount ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",unit);
    v.label=tmp;
    v.model=amount;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool RegisterFood(Lum::OS::Window* parent,
                           const std::wstring& caption,
                           const std::wstring& text,
                           Lum::Model::Time* time,
                           Lum::Model::String* amount,
                           const std::wstring& unit)
  {
    assert(time!=NULL && amount!=NULL);

    FoodRegister *dlg;
    bool    res;

    dlg=new FoodRegister(time,amount,unit);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodsEntry : public Lum::Model::ListTable::Entry
{
public:
  const ::Food *entry;

public:
  FoodsEntry(Lum::Model::ListTable *table,
             const ::Food* entry)
  : Lum::Model::ListTable::Entry(table),
    entry(entry)
  {
    // no code
  }

  std::wstring GetString(size_t column) const
  {
    if (column==1) {
      return entry->name+L" ("+entry->unit+L")";
    }
    else if (column==2) {
      return ValueToString(entry->value)+L" "+valueUnit->Get();
    }

    assert(false);
  }

  bool IsGreater(const Entry* other, size_t column)
  {
    return entry->name>dynamic_cast<const FoodsEntry*>(other)->entry->name;
  }
};

class FoodFormatProvider : public Lum::TableView::FormatProvider
{
private:
  Lum::Model::ListTable *table;
  Lum::OS::ColorRef     goodFoodColor;
  Lum::OS::ColorRef     badFoodColor;
  Lum::OS::FillRef      goodFoodBackground;
  Lum::OS::FillRef      badFoodBackground;

public:
  FoodFormatProvider(Lum::Model::ListTable* table)
   : table(table),
     goodFoodColor(0.82,1.0,0.82,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     badFoodColor(1.0,0.82,0.82,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     goodFoodBackground(new Lum::OS::PlainFill(goodFoodColor)),
     badFoodBackground(new Lum::OS::PlainFill(badFoodColor))
  {
    // no code
  }

  Lum::OS::Fill* GetBackground(size_t column, size_t row) const
  {
    FoodsEntry* entry=(FoodsEntry*)table->GetEntry(row);

    if (entry->entry->value<=(Value)goodValueLimit->Get()) {
      return goodFoodBackground.Get();
    }
    else if (entry->entry->value>=(Value)badValueLimit->Get()) {
      return badFoodBackground.Get();
    }
    else {
      return Lum::TableView::FormatProvider::GetBackground(column,row);
    }
  }
};

FoodInput::FoodInput()
 : addFoodEntryAction(new Lum::Model::Action()),
   removeFoodEntryAction(new Lum::Model::Action()),
   editFoodEntryAction(new Lum::Model::Action()),
   registerFoodAction(new Lum::Model::Action()),

   foodSearch(new Lum::Model::String(L"")),
   foods(new Lum::Model::ListTable(3)),
   foodSelection(new Lum::Model::SingleLineSelection()),

   searchTimerAction(new Lum::Model::Action())
{
  AttachModel(foodSearch);

  AttachModel(addFoodEntryAction);
  AttachModel(removeFoodEntryAction);
  AttachModel(editFoodEntryAction);
  AttachModel(registerFoodAction);

  AttachModel(searchTimerAction);

  AttachModel(goodValueLimit);
  AttachModel(badValueLimit);
}

void FoodInput::CalcSize()
{
  Lum::Menu             *menu;
  Lum::Panel            *ft;
  Lum::Panel            *sb;
  Lum::Panel            *horiz;
  Lum::Panel            *vert;
  Lum::String           *string;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  // Search box

  sb=new Lum::VPanel();
  sb->SetFlex(true,false);

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetModel(foodSearch);
  sb->Add(string);

  // Food table

  ft=new Lum::HPanel();
  ft->SetFlex(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_FOOD",L"Food"),Lum::Base::Size::stdCharWidth,20,true);
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_VALUE",L"Value"),Lum::Base::Size::stdCharWidth,8);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeaderModel(headerModel);
  table->SetModel(foods);
  table->SetSelection(foodSelection);
  table->SetDoubleClickAction(registerFoodAction);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetFormatProvider(new FoodFormatProvider(foods));
  ft->Add(table);

  menu=new Lum::Menu();
  menu->SetParent(GetWindow());
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_ADD",L"_Add entry..."),addFoodEntryAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_EDIT",L"_Edit entry..."),editFoodEntryAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_REMOVE",L"_Remove entry"),removeFoodEntryAction);
  table->SetMenu(menu->GetWindow());

  // Layout comonents

  horiz=new Lum::HPanel();

  vert=new Lum::VPanel();
  vert->SetFlex(true,true);

  vert->Add(sb);
  vert->Add(new Lum::VSpace());
  vert->Add(ft);

  horiz->Add(vert);

  container=horiz;

  Component::CalcSize();
}

void FoodInput::UpdateFoods()
{
  std::wstring filter=foodSearch->Get();
  // Food
  foods->Off();

  foods->Clear();
  for (std::set<Food>::iterator iter=data.foods.begin(); iter!=data.foods.end(); ++iter) {
    if (filter.empty() || iter->name.find(filter)!=std::wstring::npos) {
      foods->Append(new FoodsEntry(foods,&*iter));
    }
  }

  foods->Sort(1);

  foods->On();
}

void FoodInput::AddFoodEntry()
{
  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;
  std::wstring          tmp;

  name=new Lum::Model::String(L"");
  unit=new Lum::Model::String(L"");
  value=new Lum::Model::String(L"");

  tmp=_(L"FOOD_ADD_TEXT",L"Please enter the name of the new food component together\n"
        L"with its standard portion unit and its value in $unit!");
  Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
  if (!FoodEdit::EditFood(this->GetWindow(),
                          _(L"FOOD_ADD_TITLE",L"Please enter a new food component..."),
                          tmp,
                          name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));
    return;
  }

  foods->Append(new FoodsEntry(foods,result));

  foods->Sort(1);
}

void FoodInput::EditFoodEntry()
{
  Food       oldFood;
  FoodsEntry *entry=(FoodsEntry*)foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;

  oldFood=*entry->entry;

  name=new Lum::Model::String(entry->entry->name);
  unit=new Lum::Model::String(entry->entry->unit);
  value=new Lum::Model::String(ValueToString(entry->entry->value));

  if (!FoodEdit::EditFood(this->GetWindow(),
                              L"Editing food component...",
                              L"Please enter the name of the food component together\n"
                              L"with its standard portion unit and its value!",
                              name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  foods->Off();

  data.DeleteFood(*dynamic_cast<FoodsEntry*>(foods->GetEntry(foodSelection->GetLine()))->entry);
  foods->Delete(foodSelection->GetLine());

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));

    result=data.AddFood(oldFood);
  }

  foods->Append(new FoodsEntry(foods,result));
  foods->Sort(1);

  foods->On();
}

void FoodInput::RemoveFoodEntry()
{
  data.DeleteFood(*dynamic_cast<FoodsEntry*>(foods->GetEntry(foodSelection->GetLine()))->entry);
  foods->Delete(foodSelection->GetLine());
}

void FoodInput::RegisterFood()
{
  FoodsEntry *entry=(FoodsEntry*)foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef value;
  size_t                amount;
  std::wstring          t1,t2;

  value=new Lum::Model::String(L"1");

  t1=_(L"FOOD_REGISTER_TITLE",L"I have consumed $food...");
  Lum::Base::L10N::Substitute(t1,L"$food",entry->entry->name);
  t2=_(L"FOOD_REGISTER_TEXT",L"Please enter the amount of $food\nyou have consumed!");
  Lum::Base::L10N::Substitute(t2,L"$food",entry->entry->name);
  if (!FoodRegister::RegisterFood(this->GetWindow(),
                                  t1,t2,timestamp,value,entry->entry->unit)) {
    return;
  }

  if (value->IsNull() || value->Empty()) {
    return;
  }

  if (!StringToAmount(value->Get(),amount)) {
    return;
  }

  Entry e;

  e.time=timestamp->Get();
  e.item=entry->entry->name;
  e.unit=entry->entry->unit;
  e.value=entry->entry->value;
  e.amount=amount;

  data.AddEntry(date->Get(),e);
  // We trigger the date model to initiate refresh of data table
  date->Notify();
}

void FoodInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==addFoodEntryAction && addFoodEntryAction->IsFinished()) {
    AddFoodEntry();
  }
  else if (model==editFoodEntryAction && editFoodEntryAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      EditFoodEntry();
    }
  }
  else if (model==removeFoodEntryAction && removeFoodEntryAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RemoveFoodEntry();
    }
  }
  else if (model==registerFoodAction && registerFoodAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RegisterFood();
    }
  }
  else if (model==goodValueLimit || model==badValueLimit) {
    UpdateFoods();
  }
  else if (model==foodSearch) {
    Lum::OS::display->RemoveTimer(searchTimerAction);
    Lum::OS::display->AddTimer(1,500000,searchTimerAction);
  }
  else if (model==searchTimerAction && searchTimerAction->IsFinished()) {

    if (foodSearch->Get()!=lastFoodSearchFilter) {
      UpdateFoods();
      lastFoodSearchFilter=foodSearch->Get();
    }
  }

  Lum::Component::Resync(model,msg);
}

void FoodInput::InitializeGUI()
{
  UpdateFoods();
}
