/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Diagram.h"

#include <cmath>
#include <cstdlib>

#include <Lum/Base/String.h>

#include "Configuration.h"
#include "Util.h"
#include <iostream>
static Diagram::Prefs *prefs=new Diagram::Prefs();

void Diagram::Prefs::Initialize()
{
  Lum::Scrollable::Prefs::Initialize();

  background=Lum::OS::display->GetFill(Lum::OS::Display::graphBackgroundFillIndex);
  frame=Lum::OS::display->GetFrame(Lum::OS::Display::listboxFrameIndex);
}

double exponent(double x)
{
  if (x>=0) {
    return floor(log10(x));
  }
  else {
    return floor(log10(-x));
  }
}

Diagram::Diagram()
 : warnColor(1.0,0.0,0.0,Lum::OS::display->blackColor),
   limitColor(1.0,0.75,0.75,Lum::OS::display->blackColor),
   mode(modeFullDay)
{
  SetPrefs(::prefs);

  AttachModel(topWeight);
  AttachModel(bottomWeight);
  AttachModel(maxWeight);
  AttachModel(minWeight);

  AttachModel(topValue);
  AttachModel(bottomValue);
  AttachModel(maxValue);
  AttachModel(minValue);
}

Diagram::~Diagram()
{
  UnattachModel(minValue);
  UnattachModel(maxValue);
  UnattachModel(bottomValue);
  UnattachModel(topValue);

  UnattachModel(minWeight);
  UnattachModel(maxWeight);
  UnattachModel(bottomWeight);
  UnattachModel(topWeight);
}

void Diagram::CalcSize()
{
  Lum::OS::FontRef font;
  size_t           digitWidth=0;

  fullDayWidth=0;
  smallDayWidth=0;
  font=Lum::OS::display->GetFont();

  for (size_t i=0; i<10; i++) {
    digitWidth=std::max(digitWidth,font->StringWidth(Lum::Base::NumberToWString(i)));
  }

  for (size_t i=0; i<7; i++) {
    fullDayWidth=std::max(fullDayWidth,font->StringWidth(Lum::Base::Calendar::GetWeekDayNameShort(i)));
  }

  for (size_t i=0; i<7; i++) {
    fullDayWidth=std::max(fullDayWidth,digitWidth+
                      font->StringWidth(L". ")+
                      font->StringWidth(Lum::Base::Calendar::GetMonthNameShort(i)));
  }

  fullDayWidth=std::max(fullDayWidth,4*digitWidth)+2;
  smallDayWidth=2*digitWidth+font->StringWidth(L".")+2;
  weightUnitWidth=4*digitWidth+font->StringWidth(L".");
  valueUnitWidth=3*digitWidth+font->StringWidth(L".");

  width=100;
  height=100;
  minWidth=width;
  minHeight=height;

  Scrollable::CalcSize();
}

int Diagram::TransformWeight(double y)
{
  return (int)floor(height-(y-bottomWeight->Get())*(height-1)/(topWeight->Get()-bottomWeight->Get()));
}

int Diagram::TransformValue(double y)
{
  return (int)floor(height-(y-bottomValue->Get())*(height-1)/(topValue->Get()-bottomValue->Get()));
}

size_t Diagram::GetDayWidth() const
{
  switch (mode) {
  case modeFullDay:
    return fullDayWidth;
  case modeSmallDay:
    return smallDayWidth;
  case modeOverview:
    return 6;
  default:
    assert(false);
  }
}

void Diagram::Draw(int x, int y, size_t w, size_t h)
{
  Lum::Scrollable::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  DrawBackground(x,y,w,h);

  Lum::OS::DrawInfo                                  *draw=GetDrawInfo();
  Lum::OS::FontRef                                   font;
  size_t                                             daysCovered=data.GetDaysCovered();
  Lum::Base::Calendar                                start=data.GetStartDate();
  Lum::Base::Calendar                                end;
  Lum::Base::Calendar                                day;
  std::map<Lum::Base::Calendar,Day*>::const_iterator iter;
  int                                                xPos;
  std::wstring                                       tmp;
  size_t                                             dayWidth;

  dayWidth=GetDayWidth();

  start.AddDays((hAdjustment->GetTop()-1)/dayWidth);
  end=start;
  end.AddDays((width+dayWidth/2)/dayWidth);

  hAdjustment->SetDimension(width,std::max(width,daysCovered*dayWidth));
  hAdjustment->SetStepSize(dayWidth);

  //
  // Printing horizontal time scale
  //

  font=Lum::OS::display->GetFont();

  draw->PushClip(x,y,w,h);

  draw->PushForeground(Lum::OS::Display::graphScaleColor);
  draw->PushFont(font);

  draw->PushClip(this->x+weightUnitWidth,this->y,this->width-weightUnitWidth-valueUnitWidth,this->height);

  day=start;
  xPos=this->x+((hAdjustment->GetTop()-1)/dayWidth)*dayWidth-(hAdjustment->GetTop()-1);
  while (xPos<this->x+(int)width) {

    switch (mode) {
    case modeFullDay:
      draw->DrawLine(xPos+dayWidth-1,this->y,xPos+dayWidth-1,this->y+height-1);
      tmp=Lum::Base::Calendar::GetWeekDayNameShort(day.GetDayOfWeek());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-3*font->height+font->ascent,tmp);

      tmp=Lum::Base::NumberToWString(day.GetDayOfMonth())+L". "+Lum::Base::Calendar::GetMonthNameShort(day.GetMonth());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-2*font->height+font->ascent,tmp);

      tmp=Lum::Base::NumberToWString(day.GetYear());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-font->height+font->ascent,tmp);
      break;
    case modeSmallDay:
      draw->DrawLine(xPos+dayWidth-1,this->y,xPos+dayWidth-1,this->y+height-1);
      tmp=Lum::Base::NumberToWString(day.GetDayOfMonth())+L".";
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,
                       this->y+font->ascent+height-font->height,
                       tmp);
      break;
    case modeOverview:
      if (day.GetDayOfYear()==1) {
        tmp=Lum::Base::NumberToWString(day.GetYear());
        draw->DrawString(xPos,this->y+height-2*font->height+font->ascent,tmp);
      }
      if (day.GetDayOfMonth()==1) {
        tmp=Lum::Base::Calendar::GetMonthName(day.GetMonth());
        draw->DrawString(xPos,this->y+height-font->height+font->ascent,tmp);
      }
      break;
    }

    day.AddDays(1);
    xPos+=dayWidth;
  }

  draw->PopClip();

  draw->PopFont();
  draw->PopForeground();

  // Weight limits
  draw->PushForeground(limitColor);
  draw->DrawLine(this->x,this->y+TransformWeight(maxWeight->Get()),
                 this->x+weightUnitWidth-1,this->y+TransformWeight(maxWeight->Get()));
  draw->DrawLine(this->x,this->y+TransformWeight(minWeight->Get()),
                 this->x+weightUnitWidth-1,this->y+TransformWeight(minWeight->Get()));
  draw->PopForeground();

  // Value limits
  draw->PushForeground(limitColor);
  draw->DrawLine(this->x+width-1-valueUnitWidth,this->y+TransformValue(maxValue->Get()),
                 this->x+width-1,this->y+TransformValue(maxValue->Get()));
  draw->DrawLine(this->x+width-1-valueUnitWidth,this->y+TransformValue(minValue->Get()),
                 this->x+width-1,this->y+TransformValue(minValue->Get()));
  draw->PopForeground();

  //
  // Printing data
  //

  xPos=this->x+((hAdjustment->GetTop()-1)/dayWidth)*dayWidth-(hAdjustment->GetTop()-1);

  bool pointsDrawn=false; // We have drawn at least one point
  bool pastEnd=false;     // We have beenmoved past the visible end
  bool finished=false;    // We have drawn one point pastthe visible end and can now finish
  int px=0,py=0;

  iter=data.data.lower_bound(start);
  while (iter!=data.data.begin()) {
    bool hasPoint=false;

    --iter;

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        hasPoint=true;
        break;
      }
    }

    if (hasPoint) {
      break;
    }
  }

  while (iter!=data.data.end() && !finished) {
    int days;
    size_t value=0;

    if (iter->first>end) {
      pastEnd=true;
    }

    days=0;
    day=start;
    if (day<=iter->first) {
      while (day!=iter->first) {
        day.AddDays(1);
        days++;
      }
    }
    else {
      while (day!=iter->first) {
        day.AddDays(-1);
        days--;
      }
    }

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        if (entry->amount>maxWeight->Get() || entry->amount<minWeight->Get()) {
          draw->PushForeground(warnColor);
        }
        else {
          draw->PushForeground(Lum::OS::Display::blackColor);
        }

        int cx,cy;

        cx=xPos+days*dayWidth+(entry->time.GetHour()*dayWidth)/24;
        cy=this->y+TransformWeight(entry->amount);

        draw->FillArc(cx-3,
                      cy-3,
                      6,6,
                      0,360*64);
        draw->PopForeground();

        if (pointsDrawn) {
          int lx1, ly1, lx2, ly2;

          // Project points outside the diagram on the axis of the diagram
          // to avoid overflow errors and similar...
          lx1=px;
          ly1=py;
          lx2=cx;
          ly2=cy;

          if (lx1<this->x) {
            lx1=this->x;
            ly1=(lx1-px)*(cy-py)/(cx-px)+py;
          }

          if (lx2>this->x+this->width-1) {
            lx2=this->x+this->width-1;
            ly2=(lx2-px)*(cy-py)/(cx-px)+py;
          }

          draw->PushPen(2,Lum::OS::DrawInfo::penRound);
          draw->DrawLine(lx1,ly1,lx2,ly2);
          draw->PopPen();
        }

        pointsDrawn=true;
        px=cx;
        py=cy;

        if (pastEnd) {
          finished=true;
        }
      }
      else {
        value+=entry->amount*entry->value;
      }
    }

    if (value!=0) {
      value/=10;
      if (value>maxValue->Get() || value<minValue->Get()) {
        draw->PushForeground(warnColor);
      }
      else {
        draw->PushForeground(Lum::OS::Display::blackColor);
      }
      draw->DrawLine(xPos+days*dayWidth,this->y+TransformValue(value),
                     xPos+(days+1)*dayWidth-1,this->y+TransformValue(value));
      draw->PopForeground();
    }


    ++iter;
  }

  //
  // Printing vertial weight scale
  //

  draw->PushClip(this->x,this->y,this->width,this->height);
  draw->PushForeground(Lum::OS::Display::graphScaleColor);
  draw->PushFont(font);

  size_t step;

  if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+1))>font->height) {
    step=1;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+5))>font->height) {
    step=5;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+10))>font->height) {
    step=10;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+50))>font->height) {
    step=50;
  }
  else {
    step=100;
  }

  for (size_t i=(bottomWeight->Get()/step)*step; i<=topWeight->Get(); i+=step) {
    tmp=AmountToString(i);
    draw->DrawString(this->x+weightUnitWidth-font->StringWidth(tmp),
                     TransformWeight(i)-font->height/2+this->y+font->ascent,
                     tmp);
  }

  //
  // Printing vertial value scale
  //

  if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+1))>font->height) {
    step=1;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+5))>font->height) {
    step=5;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+10))>font->height) {
    step=10;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+50))>font->height) {
    step=50;
  }
  else {
    step=100;
  }

  for (size_t i=(bottomValue->Get()/step)*step; i<=topValue->Get(); i+=step) {
    tmp=AmountToString(i);
    draw->DrawString(this->x+this->width-1-font->StringWidth(tmp),
                     TransformValue(i)-font->height/2+this->y+font->ascent,
                     tmp);
  }

  draw->PopFont();
  draw->PopForeground();
  draw->PopClip();

  draw->PopClip();
}

bool Diagram::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down &&
     PointIsIn(event) &&
     event.button==Lum::OS::MouseEvent::button1) {
    gestureStartX=event.x;
    gestureStartY=event.y;

    if (GetWindow()->IsDoubleClicked()) {
      Lum::Base::Calendar day=data.GetStartDate();

      day.AddDays((event.x-x+hAdjustment->GetTop()-1)/GetDayWidth());

      date->Set(day);
      currentTab->Set(0);
    }
    return true;
  }
  else if (event.IsGrabEnd() && event.button==Lum::OS::MouseEvent::button1) {

    if (std::abs(gestureStartX-event.x)>=std::abs(gestureStartY-event.y) ||
        std::abs(gestureStartY-event.y)<(height*10)/100) {
      return false;
    }

    if (event.y>gestureStartY) {
      // down
      mode=(Mode)((mode+1)%(modeMax+1));
    }
    else {
      // up
      if (mode==0) {
        mode=modeMax;
      }
      else {
        mode=(Mode)((mode-1)%(modeMax+1));
      }
    }

    Draw(oX,oY,oWidth,oHeight);
  }
  else if (event.type==Lum::OS::MouseEvent::down &&
     PointIsIn(event) &&
     event.button==Lum::OS::MouseEvent::button4) {
   if (mode==0) {
     mode=modeMax;
   }
   else {
     mode=(Mode)((mode-1)%(modeMax+1));
   }

   Draw(oX,oY,oWidth,oHeight);

   return true;
  }
  else if (event.type==Lum::OS::MouseEvent::down &&
          PointIsIn(event) &&
          event.button==Lum::OS::MouseEvent::button5) {
   mode=(Mode)((mode+1)%(modeMax+1));

   Draw(oX,oY,oWidth,oHeight);

   return true;
  }

  return false;
}

void Diagram::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hAdjustment->GetTopModel()) {
    Redraw();
  }
  else if (model==topWeight ||
           model==bottomWeight ||
           model==minWeight ||
           model==maxWeight ||
           model==topValue ||
           model==bottomValue ||
           model==minValue ||
           model==maxValue) {
    Redraw();
  }

  Scrollable::Resync(model,msg);
}

