/* valagirparser.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <glib/gstdio.h>


#define VALA_TYPE_GIR_PARSER (vala_gir_parser_get_type ())
#define VALA_GIR_PARSER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GIR_PARSER, ValaGirParser))
#define VALA_GIR_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GIR_PARSER, ValaGirParserClass))
#define VALA_IS_GIR_PARSER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GIR_PARSER))
#define VALA_IS_GIR_PARSER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GIR_PARSER))
#define VALA_GIR_PARSER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GIR_PARSER, ValaGirParserClass))

typedef struct _ValaGirParser ValaGirParser;
typedef struct _ValaGirParserClass ValaGirParserClass;
typedef struct _ValaGirParserPrivate ValaGirParserPrivate;

#define VALA_TYPE_MARKUP_READER (vala_markup_reader_get_type ())
#define VALA_MARKUP_READER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MARKUP_READER, ValaMarkupReader))
#define VALA_MARKUP_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MARKUP_READER, ValaMarkupReaderClass))
#define VALA_IS_MARKUP_READER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MARKUP_READER))
#define VALA_IS_MARKUP_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MARKUP_READER))
#define VALA_MARKUP_READER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MARKUP_READER, ValaMarkupReaderClass))

typedef struct _ValaMarkupReader ValaMarkupReader;
typedef struct _ValaMarkupReaderClass ValaMarkupReaderClass;

#define VALA_TYPE_MARKUP_TOKEN_TYPE (vala_markup_token_type_get_type ())

/**
 * Code visitor parsing all Vala source files.
 */
struct _ValaGirParser {
	ValaCodeVisitor parent_instance;
	ValaGirParserPrivate * priv;
};

struct _ValaGirParserClass {
	ValaCodeVisitorClass parent_class;
};

typedef enum  {
	VALA_MARKUP_TOKEN_TYPE_NONE,
	VALA_MARKUP_TOKEN_TYPE_START_ELEMENT,
	VALA_MARKUP_TOKEN_TYPE_END_ELEMENT,
	VALA_MARKUP_TOKEN_TYPE_TEXT,
	VALA_MARKUP_TOKEN_TYPE_EOF
} ValaMarkupTokenType;

struct _ValaGirParserPrivate {
	char* _package_name;
	ValaMarkupReader* reader;
	ValaCodeContext* context;
	ValaNamespace* glib_ns;
	ValaSourceFile* current_source_file;
	ValaSourceLocation begin;
	ValaSourceLocation end;
	ValaMarkupTokenType current_token;
	char** cheader_filenames;
	gint cheader_filenames_length1;
	gint cheader_filenames_size;
	GeeHashMap* attributes_map;
	GeeHashMap* gtype_callbacks;
};


static gpointer vala_gir_parser_parent_class = NULL;

GType vala_gir_parser_get_type (void);
GType vala_markup_reader_get_type (void);
GType vala_markup_token_type_get_type (void);
#define VALA_GIR_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIR_PARSER, ValaGirParserPrivate))
enum  {
	VALA_GIR_PARSER_DUMMY_PROPERTY
};
void vala_gir_parser_parse (ValaGirParser* self, ValaCodeContext* context);
void vala_gir_parser_parse_file (ValaGirParser* self, ValaSourceFile* source_file);
static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
ValaMarkupReader* vala_markup_reader_new (const char* filename);
ValaMarkupReader* vala_markup_reader_construct (GType object_type, const char* filename);
static void vala_gir_parser_next (ValaGirParser* self);
static void vala_gir_parser_parse_repository (ValaGirParser* self);
ValaMarkupTokenType vala_markup_reader_read_token (ValaMarkupReader* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end);
const char* vala_markup_reader_get_name (ValaMarkupReader* self);
static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self);
static void vala_gir_parser_start_element (ValaGirParser* self, const char* name);
static void vala_gir_parser_end_element (ValaGirParser* self, const char* name);
static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self);
static void vala_gir_parser_parse_include (ValaGirParser* self);
static void vala_gir_parser_parse_package (ValaGirParser* self);
static void vala_gir_parser_parse_c_include (ValaGirParser* self);
char* vala_markup_reader_get_attribute (ValaMarkupReader* self, const char* attr);
static void vala_gir_parser_set_package_name (ValaGirParser* self, const char* value);
static void _vala_array_add1 (char*** array, int* length, int* size, char* value);
static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name);
static void _g_list_free_vala_code_node_unref (GList* self);
static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self, const char* element_name);
static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self);
static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self);
static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self);
static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self);
static void vala_gir_parser_postprocess_gtype_callbacks (ValaGirParser* self, ValaNamespace* ns);
static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name);
static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self);
static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, char** ctype, gint* array_length_index);
static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self, char** ctype);
static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx, gint* closure_idx, gint* destroy_idx);
static ValaField* vala_gir_parser_parse_field (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self, const char* parent_ctype);
static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self);
static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self);
void vala_gir_parser_parse_metadata (ValaGirParser* self, const char* metadata_filename);
ValaGirParser* vala_gir_parser_new (void);
ValaGirParser* vala_gir_parser_construct (GType object_type);
ValaGirParser* vala_gir_parser_new (void);
const char* vala_gir_parser_get_package_name (ValaGirParser* self);
static void vala_gir_parser_finalize (ValaCodeVisitor* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Parses all .gir source files in the specified code
 * context and builds a code tree.
 *
 * @param context a code context
 */
void vala_gir_parser_parse (ValaGirParser* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1_;
	ValaCodeContext* _tmp0_;
	ValaNamespace* _tmp3_;
	ValaSymbol* _tmp2_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->context = (_tmp1_ = (_tmp0_ = context, (_tmp0_ == NULL) ? NULL : vala_code_context_ref (_tmp0_)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1_);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	self->priv->glib_ns = (_tmp3_ = (_tmp2_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (context)), "GLib"), VALA_IS_NAMESPACE (_tmp2_) ? ((ValaNamespace*) _tmp2_) : NULL), (self->priv->glib_ns == NULL) ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL)), _tmp3_);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
}


static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGirParser * self;
	self = (ValaGirParser*) base;
	g_return_if_fail (source_file != NULL);
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gir")) {
		vala_gir_parser_parse_file (self, source_file);
	}
}


void vala_gir_parser_parse_file (ValaGirParser* self, ValaSourceFile* source_file) {
	ValaSourceFile* _tmp1_;
	ValaSourceFile* _tmp0_;
	ValaMarkupReader* _tmp2_;
	GeeArrayList* remove_queue;
	ValaMarkupReader* _tmp17_;
	ValaSourceFile* _tmp18_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->current_source_file = (_tmp1_ = (_tmp0_ = source_file, (_tmp0_ == NULL) ? NULL : vala_source_file_ref (_tmp0_)), (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp1_);
	_tmp2_ = NULL;
	self->priv->reader = (_tmp2_ = vala_markup_reader_new (vala_source_file_get_filename (source_file)), (self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL)), _tmp2_);
	/* xml prolog*/
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_parse_repository (self);
	remove_queue = gee_array_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeList* _tmp3_;
		GeeIterator* _tmp4_;
		GeeIterator* _node_it;
		_tmp3_ = NULL;
		_tmp4_ = NULL;
		_node_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = vala_source_file_get_nodes (source_file))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (gee_collection_object_unref (_tmp3_), NULL)), _tmp4_);
		while (TRUE) {
			ValaCodeNode* node;
			if (!gee_iterator_next (_node_it)) {
				break;
			}
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			if (VALA_IS_CLASS (node)) {
				ValaClass* _tmp5_;
				ValaClass* cl;
				ValaNamespace* _tmp7_;
				ValaSymbol* _tmp6_;
				ValaNamespace* ns;
				char* _tmp8_;
				ValaSymbol* _tmp9_;
				ValaStruct* _tmp10_;
				ValaStruct* class_struct;
				_tmp5_ = NULL;
				cl = (_tmp5_ = VALA_CLASS (node), (_tmp5_ == NULL) ? NULL : vala_code_node_ref (_tmp5_));
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				ns = (_tmp7_ = (_tmp6_ = vala_symbol_get_parent_symbol ((ValaSymbol*) cl), VALA_IS_NAMESPACE (_tmp6_) ? ((ValaNamespace*) _tmp6_) : NULL), (_tmp7_ == NULL) ? NULL : vala_code_node_ref (_tmp7_));
				/* remove Class records*/
				_tmp8_ = NULL;
				_tmp9_ = NULL;
				_tmp10_ = NULL;
				class_struct = (_tmp10_ = (_tmp9_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), _tmp8_ = g_strconcat (vala_symbol_get_name ((ValaSymbol*) cl), "Class", NULL)), VALA_IS_STRUCT (_tmp9_) ? ((ValaStruct*) _tmp9_) : NULL), _tmp8_ = (g_free (_tmp8_), NULL), _tmp10_);
				if (class_struct != NULL) {
					vala_namespace_remove_struct (ns, VALA_STRUCT (class_struct));
					gee_collection_add ((GeeCollection*) remove_queue, (ValaCodeNode*) class_struct);
				}
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
				(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
				(class_struct == NULL) ? NULL : (class_struct = (vala_code_node_unref (class_struct), NULL));
			} else {
				if (VALA_IS_INTERFACE (node)) {
					ValaInterface* _tmp11_;
					ValaInterface* iface;
					ValaNamespace* _tmp13_;
					ValaSymbol* _tmp12_;
					ValaNamespace* ns;
					char* _tmp14_;
					ValaSymbol* _tmp15_;
					ValaStruct* _tmp16_;
					ValaStruct* iface_struct;
					_tmp11_ = NULL;
					iface = (_tmp11_ = VALA_INTERFACE (node), (_tmp11_ == NULL) ? NULL : vala_code_node_ref (_tmp11_));
					_tmp13_ = NULL;
					_tmp12_ = NULL;
					ns = (_tmp13_ = (_tmp12_ = vala_symbol_get_parent_symbol ((ValaSymbol*) iface), VALA_IS_NAMESPACE (_tmp12_) ? ((ValaNamespace*) _tmp12_) : NULL), (_tmp13_ == NULL) ? NULL : vala_code_node_ref (_tmp13_));
					/* remove Iface records*/
					_tmp14_ = NULL;
					_tmp15_ = NULL;
					_tmp16_ = NULL;
					iface_struct = (_tmp16_ = (_tmp15_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), _tmp14_ = g_strconcat (vala_symbol_get_name ((ValaSymbol*) iface), "Iface", NULL)), VALA_IS_STRUCT (_tmp15_) ? ((ValaStruct*) _tmp15_) : NULL), _tmp14_ = (g_free (_tmp14_), NULL), _tmp16_);
					if (iface_struct != NULL) {
						vala_namespace_remove_struct (ns, VALA_STRUCT (iface_struct));
						gee_collection_add ((GeeCollection*) remove_queue, (ValaCodeNode*) iface_struct);
					}
					(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
					(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
					(iface_struct == NULL) ? NULL : (iface_struct = (vala_code_node_unref (iface_struct), NULL));
				}
			}
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
	{
		GeeIterator* _node_it;
		_node_it = gee_iterable_iterator ((GeeIterable*) remove_queue);
		while (TRUE) {
			ValaCodeNode* node;
			if (!gee_iterator_next (_node_it)) {
				break;
			}
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			vala_source_file_remove_node (source_file, node);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
	_tmp17_ = NULL;
	self->priv->reader = (_tmp17_ = NULL, (self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL)), _tmp17_);
	_tmp18_ = NULL;
	self->priv->current_source_file = (_tmp18_ = NULL, (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp18_);
	(remove_queue == NULL) ? NULL : (remove_queue = (gee_collection_object_unref (remove_queue), NULL));
}


static void vala_gir_parser_next (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	self->priv->current_token = vala_markup_reader_read_token (self->priv->reader, &self->priv->begin, &self->priv->end);
}


static void vala_gir_parser_start_element (ValaGirParser* self, const char* name) {
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp0_ = FALSE;
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0;
	}
	if (_tmp0_) {
		char* _tmp2_;
		ValaSourceReference* _tmp1_;
		/* error*/
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		vala_report_error (_tmp1_ = vala_gir_parser_get_current_src (self), _tmp2_ = g_strdup_printf ("expected start element of `%s'", name));
		_tmp2_ = (g_free (_tmp2_), NULL);
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL));
	}
}


static void vala_gir_parser_end_element (ValaGirParser* self, const char* name) {
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp0_ = FALSE;
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_END_ELEMENT) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0;
	}
	if (_tmp0_) {
		char* _tmp2_;
		ValaSourceReference* _tmp1_;
		/* error*/
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		vala_report_error (_tmp1_ = vala_gir_parser_get_current_src (self), _tmp2_ = g_strdup_printf ("expected end element of `%s'", name));
		_tmp2_ = (g_free (_tmp2_), NULL);
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL));
	}
	vala_gir_parser_next (self);
}


static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self) {
	ValaSourceReference* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = vala_source_reference_new (self->priv->current_source_file, self->priv->begin.line, self->priv->begin.column, self->priv->end.line, self->priv->end.column);
	return result;
}


static void vala_gir_parser_parse_repository (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "repository");
	vala_gir_parser_next (self);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "namespace") == 0) {
			ValaNamespace* ns;
			ns = vala_gir_parser_parse_namespace (self);
			if (ns != NULL) {
				vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
			}
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "include") == 0) {
				vala_gir_parser_parse_include (self);
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "package") == 0) {
					vala_gir_parser_parse_package (self);
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "c:include") == 0) {
						vala_gir_parser_parse_c_include (self);
					} else {
						char* _tmp1_;
						ValaSourceReference* _tmp0_;
						/* error*/
						_tmp1_ = NULL;
						_tmp0_ = NULL;
						vala_report_error (_tmp0_ = vala_gir_parser_get_current_src (self), _tmp1_ = g_strdup_printf ("unknown child element `%s' in `repository'", vala_markup_reader_get_name (self->priv->reader)));
						_tmp1_ = (g_free (_tmp1_), NULL);
						(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_source_reference_unref (_tmp0_), NULL));
						break;
					}
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "repository");
}


static void vala_gir_parser_parse_include (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "include");
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "include");
}


static void vala_gir_parser_parse_package (ValaGirParser* self) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "package");
	_tmp0_ = NULL;
	vala_gir_parser_set_package_name (self, _tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"));
	_tmp0_ = (g_free (_tmp0_), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "package");
}


static void _vala_array_add1 (char*** array, int* length, int* size, char* value) {
	if ((*length) == (*size)) {
		*size = (*size) ? (2 * (*size)) : 4;
		*array = g_renew (char*, *array, (*size) + 1);
	}
	(*array)[(*length)++] = value;
	(*array)[*length] = NULL;
}


static void vala_gir_parser_parse_c_include (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "c:include");
	_vala_array_add1 (&self->priv->cheader_filenames, &self->priv->cheader_filenames_length1, &self->priv->cheader_filenames_size, vala_markup_reader_get_attribute (self->priv->reader, "name"));
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "c:include");
}


static void _g_list_free_vala_code_node_unref (GList* self) {
	g_list_foreach (self, (GFunc) vala_code_node_unref, NULL);
	g_list_free (self);
}


static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self) {
	ValaNamespace* result;
	gboolean new_namespace;
	char* _tmp0_;
	char* _tmp1_;
	char* namespace_name;
	ValaSymbol* _tmp2_;
	ValaNamespace* ns;
	char* cprefix;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "namespace");
	new_namespace = FALSE;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	namespace_name = (_tmp1_ = vala_gir_parser_transform_namespace_name (self, _tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name")), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	_tmp2_ = NULL;
	ns = (_tmp2_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (self->priv->context)), namespace_name), VALA_IS_NAMESPACE (_tmp2_) ? ((ValaNamespace*) _tmp2_) : NULL);
	if (ns == NULL) {
		ValaNamespace* _tmp3_;
		_tmp3_ = NULL;
		ns = (_tmp3_ = vala_namespace_new (namespace_name, NULL), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp3_);
		new_namespace = TRUE;
	} else {
		if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
			GList* _tmp4_;
			ValaSourceReference* _tmp5_;
			_tmp4_ = NULL;
			((ValaCodeNode*) ns)->attributes = (_tmp4_ = NULL, (((ValaCodeNode*) ns)->attributes == NULL) ? NULL : (((ValaCodeNode*) ns)->attributes = (_g_list_free_vala_code_node_unref (((ValaCodeNode*) ns)->attributes), NULL)), _tmp4_);
			_tmp5_ = NULL;
			vala_code_node_set_source_reference ((ValaCodeNode*) ns, _tmp5_ = vala_gir_parser_get_current_src (self));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_source_reference_unref (_tmp5_), NULL));
		}
	}
	cprefix = vala_markup_reader_get_attribute (self->priv->reader, "c:prefix");
	if (cprefix != NULL) {
		char* _tmp6_;
		char* _tmp7_;
		_tmp6_ = NULL;
		vala_namespace_add_cprefix (ns, _tmp6_ = g_utf8_strup (cprefix, -1));
		_tmp6_ = (g_free (_tmp6_), NULL);
		_tmp7_ = NULL;
		vala_namespace_set_lower_case_cprefix (ns, _tmp7_ = g_strconcat (cprefix, "_", NULL));
		_tmp7_ = (g_free (_tmp7_), NULL);
	}
	{
		char** c_header_collection;
		int c_header_collection_length1;
		int c_header_it;
		c_header_collection = self->priv->cheader_filenames;
		c_header_collection_length1 = self->priv->cheader_filenames_length1;
		for (c_header_it = 0; c_header_it < self->priv->cheader_filenames_length1; c_header_it = c_header_it + 1) {
			const char* _tmp8_;
			char* c_header;
			_tmp8_ = NULL;
			c_header = (_tmp8_ = c_header_collection[c_header_it], (_tmp8_ == NULL) ? NULL : g_strdup (_tmp8_));
			{
				vala_namespace_add_cheader_filename (ns, c_header);
				c_header = (g_free (c_header), NULL);
			}
		}
	}
	vala_gir_parser_next (self);
	while (TRUE) {
		ValaSymbol* sym;
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		sym = NULL;
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "alias") == 0) {
			ValaSymbol* _tmp9_;
			_tmp9_ = NULL;
			sym = (_tmp9_ = (ValaSymbol*) vala_gir_parser_parse_alias (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp9_);
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "enumeration") == 0) {
				ValaSymbol* _tmp10_;
				_tmp10_ = NULL;
				sym = (_tmp10_ = (ValaSymbol*) vala_gir_parser_parse_enumeration (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp10_);
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "bitfield") == 0) {
					ValaSymbol* _tmp11_;
					_tmp11_ = NULL;
					sym = (_tmp11_ = (ValaSymbol*) vala_gir_parser_parse_bitfield (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp11_);
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
						ValaSymbol* _tmp12_;
						_tmp12_ = NULL;
						sym = (_tmp12_ = (ValaSymbol*) vala_gir_parser_parse_method (self, "function"), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp12_);
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
							ValaSymbol* _tmp13_;
							_tmp13_ = NULL;
							sym = (_tmp13_ = (ValaSymbol*) vala_gir_parser_parse_callback (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp13_);
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "record") == 0) {
								ValaSymbol* _tmp14_;
								_tmp14_ = NULL;
								sym = (_tmp14_ = (ValaSymbol*) vala_gir_parser_parse_record (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp14_);
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "class") == 0) {
									ValaSymbol* _tmp15_;
									_tmp15_ = NULL;
									sym = (_tmp15_ = (ValaSymbol*) vala_gir_parser_parse_class (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp15_);
								} else {
									if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "interface") == 0) {
										ValaSymbol* _tmp16_;
										_tmp16_ = NULL;
										sym = (_tmp16_ = (ValaSymbol*) vala_gir_parser_parse_interface (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp16_);
									} else {
										if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:boxed") == 0) {
											ValaSymbol* _tmp17_;
											_tmp17_ = NULL;
											sym = (_tmp17_ = (ValaSymbol*) vala_gir_parser_parse_boxed (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp17_);
										} else {
											if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "union") == 0) {
												ValaSymbol* _tmp18_;
												_tmp18_ = NULL;
												sym = (_tmp18_ = (ValaSymbol*) vala_gir_parser_parse_union (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp18_);
											} else {
												if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constant") == 0) {
													ValaSymbol* _tmp19_;
													_tmp19_ = NULL;
													sym = (_tmp19_ = (ValaSymbol*) vala_gir_parser_parse_constant (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp19_);
												} else {
													char* _tmp21_;
													ValaSourceReference* _tmp20_;
													/* error*/
													_tmp21_ = NULL;
													_tmp20_ = NULL;
													vala_report_error (_tmp20_ = vala_gir_parser_get_current_src (self), _tmp21_ = g_strdup_printf ("unknown child element `%s' in `namespace'", vala_markup_reader_get_name (self->priv->reader)));
													_tmp21_ = (g_free (_tmp21_), NULL);
													(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_source_reference_unref (_tmp20_), NULL));
													(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
													break;
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		if (VALA_IS_CLASS (sym)) {
			vala_namespace_add_class (ns, VALA_CLASS (sym));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				vala_namespace_add_interface (ns, VALA_INTERFACE (sym));
			} else {
				if (VALA_IS_STRUCT (sym)) {
					vala_namespace_add_struct (ns, VALA_STRUCT (sym));
				} else {
					if (VALA_IS_ENUM (sym)) {
						vala_namespace_add_enum (ns, VALA_ENUM (sym));
					} else {
						if (VALA_IS_DELEGATE (sym)) {
							vala_namespace_add_delegate (ns, VALA_DELEGATE (sym));
						} else {
							if (VALA_IS_METHOD (sym)) {
								vala_namespace_add_method (ns, VALA_METHOD (sym));
							} else {
								if (VALA_IS_CONSTANT (sym)) {
									vala_namespace_add_constant (ns, VALA_CONSTANT (sym));
								} else {
									if (sym == NULL) {
										(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
										continue;
									}
								}
							}
						}
					}
				}
			}
		}
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) sym);
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	}
	vala_gir_parser_end_element (self, "namespace");
	vala_gir_parser_postprocess_gtype_callbacks (self, ns);
	if (!new_namespace) {
		ValaNamespace* _tmp22_;
		_tmp22_ = NULL;
		ns = (_tmp22_ = NULL, (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp22_);
	}
	result = ns;
	namespace_name = (g_free (namespace_name), NULL);
	cprefix = (g_free (cprefix), NULL);
	return result;
}


static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self) {
	ValaStruct* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaStruct* _tmp2_;
	ValaStruct* st;
	ValaDataType* _tmp4_;
	char* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "alias");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	st = (_tmp2_ = vala_struct_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	vala_struct_set_base_type (st, _tmp4_ = vala_gir_parser_parse_type_from_name (self, _tmp3_ = vala_markup_reader_get_attribute (self->priv->reader, "target")));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "alias");
	result = st;
	return result;
}


static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self) {
	ValaEnum* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaEnum* _tmp2_;
	ValaEnum* en;
	char* enum_cname;
	char* common_prefix;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "enumeration");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	en = (_tmp2_ = vala_enum_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	enum_cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (enum_cname != NULL) {
		vala_enum_set_cname (en, enum_cname);
	}
	vala_gir_parser_next (self);
	common_prefix = NULL;
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* ev;
			char* cname;
			ev = vala_gir_parser_parse_enumeration_member (self);
			vala_enum_add_value (en, ev);
			cname = vala_enum_value_get_cname (ev);
			if (common_prefix == NULL) {
				char* _tmp4_;
				const char* _tmp3_;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				common_prefix = (_tmp4_ = (_tmp3_ = cname, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), common_prefix = (g_free (common_prefix), NULL), _tmp4_);
				while (TRUE) {
					gboolean _tmp5_;
					char* _tmp6_;
					_tmp5_ = FALSE;
					if (g_utf8_strlen (common_prefix, -1) > 0) {
						_tmp5_ = !g_str_has_suffix (common_prefix, "_");
					} else {
						_tmp5_ = FALSE;
					}
					if (!_tmp5_) {
						break;
					}
					/* FIXME: could easily be made faster*/
					_tmp6_ = NULL;
					common_prefix = (_tmp6_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp6_);
				}
			} else {
				while (TRUE) {
					char* _tmp7_;
					if (!(!g_str_has_prefix (cname, common_prefix))) {
						break;
					}
					_tmp7_ = NULL;
					common_prefix = (_tmp7_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp7_);
				}
			}
			while (TRUE) {
				gboolean _tmp8_;
				char* _tmp11_;
				_tmp8_ = FALSE;
				if (g_utf8_strlen (common_prefix, -1) > 0) {
					gboolean _tmp9_;
					_tmp9_ = FALSE;
					if (!g_str_has_suffix (common_prefix, "_")) {
						_tmp9_ = TRUE;
					} else {
						gboolean _tmp10_;
						_tmp10_ = FALSE;
						if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (cname, strlen (common_prefix))))) {
							_tmp10_ = (g_utf8_strlen (cname, -1) - g_utf8_strlen (common_prefix, -1)) <= 1;
						} else {
							_tmp10_ = FALSE;
						}
						_tmp9_ = _tmp10_;
					}
					_tmp8_ = _tmp9_;
				} else {
					_tmp8_ = FALSE;
				}
				if (!_tmp8_) {
					break;
				}
				/* enum values may not consist solely of digits*/
				_tmp11_ = NULL;
				common_prefix = (_tmp11_ = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp11_);
			}
			(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			cname = (g_free (cname), NULL);
		} else {
			/* error*/
			break;
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	vala_gir_parser_end_element (self, "enumeration");
	result = en;
	enum_cname = (g_free (enum_cname), NULL);
	common_prefix = (g_free (common_prefix), NULL);
	return result;
}


static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self) {
	ValaEnum* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaEnum* _tmp2_;
	ValaEnum* en;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "bitfield");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	en = (_tmp2_ = vala_enum_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* _tmp3_;
			_tmp3_ = NULL;
			vala_enum_add_value (en, _tmp3_ = vala_gir_parser_parse_enumeration_member (self));
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
		} else {
			/* error*/
			break;
		}
	}
	vala_gir_parser_end_element (self, "bitfield");
	result = en;
	return result;
}


static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self) {
	ValaEnumValue* result;
	char* _tmp4_;
	gint _tmp3__length1;
	char** _tmp3_;
	char** _tmp2_;
	char* _tmp1_;
	char* _tmp0_;
	ValaEnumValue* _tmp5_;
	ValaEnumValue* ev;
	char* _tmp6_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "member");
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp5_ = NULL;
	ev = (_tmp5_ = vala_enum_value_new (_tmp4_ = g_strjoinv ("_", (_tmp3_ = _tmp2_ = g_strsplit (_tmp1_ = g_utf8_strup (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), -1), "-", 0), _tmp3__length1 = _vala_array_length (_tmp2_), _tmp3_)), NULL), _tmp4_ = (g_free (_tmp4_), NULL), _tmp3_ = (_vala_array_free (_tmp3_, _tmp3__length1, (GDestroyNotify) g_free), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp5_);
	_tmp6_ = NULL;
	vala_enum_value_set_cname (ev, _tmp6_ = vala_markup_reader_get_attribute (self->priv->reader, "c:identifier"));
	_tmp6_ = (g_free (_tmp6_), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "member");
	result = ev;
	return result;
}


static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self, char** ctype) {
	ValaDataType* result;
	char* transfer;
	char* allow_none;
	ValaDataType* _tmp0_;
	ValaDataType* _tmp6_;
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	if (ctype != NULL) {
		*ctype = NULL;
	}
	vala_gir_parser_start_element (self, "return-value");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	allow_none = vala_markup_reader_get_attribute (self->priv->reader, "allow-none");
	vala_gir_parser_next (self);
	_tmp0_ = NULL;
	if ((&(*ctype)) != NULL) {
		ValaDataType* _tmp4_;
		char* _tmp3_;
		ValaDataType* _tmp2_;
		char* _tmp1_;
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		_tmp0_ = (_tmp4_ = (_tmp2_ = vala_gir_parser_parse_type (self, &_tmp1_, NULL), (*ctype) = (_tmp3_ = _tmp1_, (*ctype) = (g_free ((*ctype)), NULL), _tmp3_), _tmp2_), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp4_);
	} else {
		ValaDataType* _tmp5_;
		_tmp5_ = NULL;
		_tmp0_ = (_tmp5_ = vala_gir_parser_parse_type (self, NULL, NULL), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp5_);
	}
	_tmp6_ = NULL;
	type = (_tmp6_ = _tmp0_, (_tmp6_ == NULL) ? NULL : vala_code_node_ref (_tmp6_));
	if (_vala_strcmp0 (transfer, "full") == 0) {
		vala_data_type_set_value_owned (type, TRUE);
	}
	if (_vala_strcmp0 (allow_none, "1") == 0) {
		vala_data_type_set_nullable (type, TRUE);
	}
	vala_gir_parser_end_element (self, "return-value");
	result = type;
	transfer = (g_free (transfer), NULL);
	allow_none = (g_free (allow_none), NULL);
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL));
	return result;
}


static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx, gint* closure_idx, gint* destroy_idx) {
	ValaFormalParameter* result;
	ValaFormalParameter* param;
	char* name;
	char* direction;
	char* transfer;
	char* allow_none;
	char* closure;
	char* destroy;
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	param = NULL;
	if ((&(*array_length_idx)) != NULL) {
		(*array_length_idx) = -1;
	}
	if ((&(*closure_idx)) != NULL) {
		(*closure_idx) = -1;
	}
	if ((&(*destroy_idx)) != NULL) {
		(*destroy_idx) = -1;
	}
	vala_gir_parser_start_element (self, "parameter");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	direction = vala_markup_reader_get_attribute (self->priv->reader, "direction");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	allow_none = vala_markup_reader_get_attribute (self->priv->reader, "allow-none");
	closure = vala_markup_reader_get_attribute (self->priv->reader, "closure");
	destroy = vala_markup_reader_get_attribute (self->priv->reader, "destroy");
	_tmp0_ = FALSE;
	if (closure != NULL) {
		_tmp0_ = (&(*closure_idx)) != NULL;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		(*closure_idx) = atoi (closure);
	}
	_tmp1_ = FALSE;
	if (destroy != NULL) {
		_tmp1_ = (&(*destroy_idx)) != NULL;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		(*destroy_idx) = atoi (destroy);
	}
	vala_gir_parser_next (self);
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "varargs") == 0) {
		ValaFormalParameter* _tmp3_;
		ValaSourceReference* _tmp2_;
		vala_gir_parser_start_element (self, "varargs");
		vala_gir_parser_next (self);
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		param = (_tmp3_ = vala_formal_parameter_new_with_ellipsis (_tmp2_ = vala_gir_parser_get_current_src (self)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp3_);
		(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_source_reference_unref (_tmp2_), NULL));
		vala_gir_parser_end_element (self, "varargs");
	} else {
		ValaDataType* type;
		ValaFormalParameter* _tmp5_;
		ValaSourceReference* _tmp4_;
		type = vala_gir_parser_parse_type (self, NULL, &(*array_length_idx));
		if (_vala_strcmp0 (transfer, "full") == 0) {
			vala_data_type_set_value_owned (type, TRUE);
		}
		if (_vala_strcmp0 (allow_none, "1") == 0) {
			vala_data_type_set_nullable (type, TRUE);
		}
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		param = (_tmp5_ = vala_formal_parameter_new (name, type, _tmp4_ = vala_gir_parser_get_current_src (self)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp5_);
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_source_reference_unref (_tmp4_), NULL));
		if (_vala_strcmp0 (direction, "out") == 0) {
			vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_OUT);
		} else {
			if (_vala_strcmp0 (direction, "inout") == 0) {
				vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_REF);
			}
		}
		(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	}
	vala_gir_parser_end_element (self, "parameter");
	result = param;
	name = (g_free (name), NULL);
	direction = (g_free (direction), NULL);
	transfer = (g_free (transfer), NULL);
	allow_none = (g_free (allow_none), NULL);
	closure = (g_free (closure), NULL);
	destroy = (g_free (destroy), NULL);
	return result;
}


static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, char** ctype, gint* array_length_index) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	if (ctype != NULL) {
		*ctype = NULL;
	}
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "array") == 0) {
		gboolean _tmp0_;
		char* _tmp1_;
		gboolean _tmp2_;
		ValaDataType* element_type;
		vala_gir_parser_start_element (self, "array");
		_tmp0_ = FALSE;
		_tmp1_ = NULL;
		if ((_tmp2_ = (_tmp1_ = vala_markup_reader_get_attribute (self->priv->reader, "length")) != NULL, _tmp1_ = (g_free (_tmp1_), NULL), _tmp2_)) {
			_tmp0_ = (&(*array_length_index)) != NULL;
		} else {
			_tmp0_ = FALSE;
		}
		if (_tmp0_) {
			char* _tmp3_;
			_tmp3_ = NULL;
			(*array_length_index) = atoi (_tmp3_ = vala_markup_reader_get_attribute (self->priv->reader, "length"));
			_tmp3_ = (g_free (_tmp3_), NULL);
		}
		vala_gir_parser_next (self);
		element_type = vala_gir_parser_parse_type (self, NULL, NULL);
		vala_gir_parser_end_element (self, "array");
		result = (ValaDataType*) vala_array_type_new (element_type, 1, NULL);
		(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
		return result;
	} else {
		char* _tmp4_;
		ValaDataType* _tmp5_;
		ValaDataType* type;
		vala_gir_parser_start_element (self, "type");
		_tmp4_ = NULL;
		_tmp5_ = NULL;
		type = (_tmp5_ = vala_gir_parser_parse_type_from_name (self, _tmp4_ = vala_markup_reader_get_attribute (self->priv->reader, "name")), _tmp4_ = (g_free (_tmp4_), NULL), _tmp5_);
		if ((&(*ctype)) != NULL) {
			char* _tmp6_;
			_tmp6_ = NULL;
			(*ctype) = (_tmp6_ = vala_markup_reader_get_attribute (self->priv->reader, "c:type"), (*ctype) = (g_free ((*ctype)), NULL), _tmp6_);
		}
		vala_gir_parser_next (self);
		while (TRUE) {
			ValaDataType* _tmp7_;
			if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
				break;
			}
			_tmp7_ = NULL;
			_tmp7_ = vala_gir_parser_parse_type (self, NULL, NULL);
			(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
		}
		vala_gir_parser_end_element (self, "type");
		result = type;
		return result;
	}
}


static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name) {
	ValaDataType* result;
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	type = NULL;
	if (_vala_strcmp0 (type_name, "none") == 0) {
		ValaDataType* _tmp0_;
		_tmp0_ = NULL;
		type = (_tmp0_ = (ValaDataType*) vala_void_type_new (NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0_);
	} else {
		if (_vala_strcmp0 (type_name, "any") == 0) {
			ValaDataType* _tmp2_;
			ValaVoidType* _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			type = (_tmp2_ = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp1_ = vala_void_type_new (NULL)), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2_);
			(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL));
		} else {
			if (_vala_strcmp0 (type_name, "GObject.Strv") == 0) {
				ValaDataType* _tmp5_;
				ValaUnresolvedType* _tmp4_;
				ValaUnresolvedSymbol* _tmp3_;
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				_tmp3_ = NULL;
				type = (_tmp5_ = (ValaDataType*) vala_array_type_new ((ValaDataType*) (_tmp4_ = vala_unresolved_type_new_from_symbol (_tmp3_ = vala_unresolved_symbol_new (NULL, "string", NULL), NULL)), 1, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp5_);
				(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
			} else {
				char** _tmp7_;
				gint type_components_size;
				gint type_components_length1;
				char** _tmp6_;
				char** type_components;
				if (_vala_strcmp0 (type_name, "utf8") == 0) {
					type_name = "string";
				} else {
					if (_vala_strcmp0 (type_name, "boolean") == 0) {
						type_name = "bool";
					} else {
						if (_vala_strcmp0 (type_name, "GLib.offset") == 0) {
							type_name = "int64";
						} else {
							if (_vala_strcmp0 (type_name, "GType") == 0) {
								type_name = "GLib.Type";
							} else {
								if (_vala_strcmp0 (type_name, "GObject.String") == 0) {
									type_name = "GLib.StringBuilder";
								} else {
									if (_vala_strcmp0 (type_name, "GObject.Class") == 0) {
										type_name = "GLib.ObjectClass";
									} else {
										if (_vala_strcmp0 (type_name, "GLib.unichar") == 0) {
											type_name = "unichar";
										} else {
											if (_vala_strcmp0 (type_name, "GLib.Data") == 0) {
												type_name = "GLib.Datalist";
											}
										}
									}
								}
							}
						}
					}
				}
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				type_components = (_tmp7_ = _tmp6_ = g_strsplit (type_name, ".", 0), type_components_length1 = _vala_array_length (_tmp6_), type_components_size = type_components_length1, _tmp7_);
				if (type_components[1] != NULL) {
					char* namespace_name;
					const char* _tmp8_;
					char* transformed_type_name;
					ValaDataType* _tmp11_;
					ValaUnresolvedSymbol* _tmp10_;
					ValaUnresolvedSymbol* _tmp9_;
					/* namespaced name*/
					namespace_name = vala_gir_parser_transform_namespace_name (self, type_components[0]);
					_tmp8_ = NULL;
					transformed_type_name = (_tmp8_ = type_components[1], (_tmp8_ == NULL) ? NULL : g_strdup (_tmp8_));
					_tmp11_ = NULL;
					_tmp10_ = NULL;
					_tmp9_ = NULL;
					type = (_tmp11_ = (ValaDataType*) vala_unresolved_type_new_from_symbol (_tmp10_ = vala_unresolved_symbol_new (_tmp9_ = vala_unresolved_symbol_new (NULL, namespace_name, NULL), transformed_type_name, NULL), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp11_);
					(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
					(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_code_node_unref (_tmp9_), NULL));
					namespace_name = (g_free (namespace_name), NULL);
					transformed_type_name = (g_free (transformed_type_name), NULL);
				} else {
					ValaDataType* _tmp13_;
					ValaUnresolvedSymbol* _tmp12_;
					_tmp13_ = NULL;
					_tmp12_ = NULL;
					type = (_tmp13_ = (ValaDataType*) vala_unresolved_type_new_from_symbol (_tmp12_ = vala_unresolved_symbol_new (NULL, type_name, NULL), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp13_);
					(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL));
				}
				type_components = (_vala_array_free (type_components, type_components_length1, (GDestroyNotify) g_free), NULL);
			}
		}
	}
	result = type;
	return result;
}


static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name) {
	char* result;
	const char* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (gir_module_name != NULL, NULL);
	if (_vala_strcmp0 (gir_module_name, "GObject") == 0) {
		result = g_strdup ("GLib");
		return result;
	} else {
		if (_vala_strcmp0 (gir_module_name, "Gio") == 0) {
			result = g_strdup ("GLib");
			return result;
		} else {
			if (_vala_strcmp0 (gir_module_name, "GModule") == 0) {
				result = g_strdup ("GLib");
				return result;
			}
		}
	}
	_tmp0_ = NULL;
	result = (_tmp0_ = gir_module_name, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	return result;
}


static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self) {
	ValaStruct* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaStruct* _tmp2_;
	ValaStruct* st;
	char* glib_is_gtype_struct_for;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "record");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	st = (_tmp2_ = vala_struct_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	glib_is_gtype_struct_for = vala_markup_reader_get_attribute (self->priv->reader, "glib:is-gtype-struct-for");
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp3_;
			_tmp3_ = NULL;
			vala_struct_add_field (st, _tmp3_ = vala_gir_parser_parse_field (self));
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
				if (glib_is_gtype_struct_for != NULL) {
					GeeArrayList* callbacks;
					ValaMethod* _tmp5_;
					callbacks = (GeeArrayList*) gee_map_get ((GeeMap*) self->priv->gtype_callbacks, glib_is_gtype_struct_for);
					if (callbacks == NULL) {
						GeeArrayList* _tmp4_;
						_tmp4_ = NULL;
						callbacks = (_tmp4_ = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (callbacks == NULL) ? NULL : (callbacks = (gee_collection_object_unref (callbacks), NULL)), _tmp4_);
						gee_map_set ((GeeMap*) self->priv->gtype_callbacks, glib_is_gtype_struct_for, callbacks);
					}
					_tmp5_ = NULL;
					gee_collection_add ((GeeCollection*) callbacks, _tmp5_ = vala_gir_parser_parse_method (self, "callback"));
					(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
					(callbacks == NULL) ? NULL : (callbacks = (gee_collection_object_unref (callbacks), NULL));
				} else {
					ValaDelegate* _tmp6_;
					_tmp6_ = NULL;
					_tmp6_ = vala_gir_parser_parse_callback (self);
					(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
				}
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
					ValaMethod* _tmp7_;
					_tmp7_ = NULL;
					_tmp7_ = vala_gir_parser_parse_constructor (self, NULL);
					(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
						ValaMethod* _tmp8_;
						_tmp8_ = NULL;
						vala_struct_add_method (st, _tmp8_ = vala_gir_parser_parse_method (self, "method"));
						(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
					} else {
						char* _tmp10_;
						ValaSourceReference* _tmp9_;
						/* error*/
						_tmp10_ = NULL;
						_tmp9_ = NULL;
						vala_report_error (_tmp9_ = vala_gir_parser_get_current_src (self), _tmp10_ = g_strdup_printf ("unknown child element `%s' in `record'", vala_markup_reader_get_name (self->priv->reader)));
						_tmp10_ = (g_free (_tmp10_), NULL);
						(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_source_reference_unref (_tmp9_), NULL));
						break;
					}
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "record");
	result = st;
	glib_is_gtype_struct_for = (g_free (glib_is_gtype_struct_for), NULL);
	return result;
}


static void vala_gir_parser_postprocess_gtype_callbacks (ValaGirParser* self, ValaNamespace* ns) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	{
		GeeSet* _tmp0_;
		GeeIterator* _tmp1_;
		GeeIterator* _gtype_name_it;
		_tmp0_ = NULL;
		_tmp1_ = NULL;
		_gtype_name_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_map_get_keys ((GeeMap*) self->priv->gtype_callbacks))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (gee_collection_object_unref (_tmp0_), NULL)), _tmp1_);
		while (TRUE) {
			char* gtype_name;
			ValaSymbol* _tmp2_;
			ValaObjectTypeSymbol* gtype;
			GeeArrayList* callbacks;
			if (!gee_iterator_next (_gtype_name_it)) {
				break;
			}
			gtype_name = (char*) gee_iterator_get (_gtype_name_it);
			_tmp2_ = NULL;
			gtype = (_tmp2_ = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), gtype_name), VALA_IS_OBJECT_TYPE_SYMBOL (_tmp2_) ? ((ValaObjectTypeSymbol*) _tmp2_) : NULL);
			callbacks = (GeeArrayList*) gee_map_get ((GeeMap*) self->priv->gtype_callbacks, gtype_name);
			{
				GeeIterator* _m_it;
				_m_it = gee_iterable_iterator ((GeeIterable*) callbacks);
				while (TRUE) {
					ValaMethod* m;
					ValaSymbol* symbol;
					if (!gee_iterator_next (_m_it)) {
						break;
					}
					m = (ValaMethod*) gee_iterator_get (_m_it);
					symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) gtype), vala_symbol_get_name ((ValaSymbol*) m));
					if (symbol == NULL) {
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
						continue;
					} else {
						if (VALA_IS_METHOD (symbol)) {
							ValaMethod* _tmp3_;
							ValaMethod* meth;
							_tmp3_ = NULL;
							meth = (_tmp3_ = VALA_METHOD (symbol), (_tmp3_ == NULL) ? NULL : vala_code_node_ref (_tmp3_));
							if (VALA_IS_CLASS (gtype)) {
								vala_method_set_is_virtual (meth, TRUE);
							} else {
								if (VALA_IS_INTERFACE (gtype)) {
									vala_method_set_is_abstract (meth, TRUE);
								}
							}
							(meth == NULL) ? NULL : (meth = (vala_code_node_unref (meth), NULL));
						} else {
							if (VALA_IS_SIGNAL (symbol)) {
								ValaSignal* _tmp4_;
								ValaSignal* sig;
								_tmp4_ = NULL;
								sig = (_tmp4_ = VALA_SIGNAL (symbol), (_tmp4_ == NULL) ? NULL : vala_code_node_ref (_tmp4_));
								vala_signal_set_is_virtual (sig, TRUE);
								(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
							} else {
								char* _tmp6_;
								ValaSourceReference* _tmp5_;
								_tmp6_ = NULL;
								_tmp5_ = NULL;
								vala_report_error (_tmp5_ = vala_gir_parser_get_current_src (self), _tmp6_ = g_strdup_printf ("unknown member type `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) gtype)));
								_tmp6_ = (g_free (_tmp6_), NULL);
								(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_source_reference_unref (_tmp5_), NULL));
							}
						}
					}
					(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
				}
				(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
			}
			gtype_name = (g_free (gtype_name), NULL);
			(gtype == NULL) ? NULL : (gtype = (vala_code_node_unref (gtype), NULL));
			(callbacks == NULL) ? NULL : (callbacks = (gee_collection_object_unref (callbacks), NULL));
		}
		(_gtype_name_it == NULL) ? NULL : (_gtype_name_it = (gee_collection_object_unref (_gtype_name_it), NULL));
	}
}


static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self) {
	ValaClass* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaClass* _tmp2_;
	ValaClass* cl;
	char* cname;
	char* parent;
	GeeArrayList* signals;
	GeeArrayList* methods;
	GeeArrayList* vmethods;
	GeeArrayList* fields;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "class");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	cl = (_tmp2_ = vala_class_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_class_set_cname (cl, cname);
	}
	parent = vala_markup_reader_get_attribute (self->priv->reader, "parent");
	if (parent != NULL) {
		ValaDataType* _tmp3_;
		_tmp3_ = NULL;
		vala_class_add_base_type (cl, _tmp3_ = vala_gir_parser_parse_type_from_name (self, parent));
		(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
	}
	vala_gir_parser_next (self);
	signals = gee_array_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	methods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	vmethods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	fields = gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "implements") == 0) {
			ValaDataType* _tmp5_;
			char* _tmp4_;
			vala_gir_parser_start_element (self, "implements");
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			vala_class_add_base_type (cl, _tmp5_ = vala_gir_parser_parse_type_from_name (self, _tmp4_ = vala_markup_reader_get_attribute (self->priv->reader, "name")));
			(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
			_tmp4_ = (g_free (_tmp4_), NULL);
			vala_gir_parser_next (self);
			vala_gir_parser_end_element (self, "implements");
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constant") == 0) {
				ValaConstant* _tmp6_;
				_tmp6_ = NULL;
				vala_class_add_constant (cl, _tmp6_ = vala_gir_parser_parse_constant (self));
				(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
					ValaField* _tmp7_;
					_tmp7_ = NULL;
					gee_collection_add ((GeeCollection*) fields, _tmp7_ = vala_gir_parser_parse_field (self));
					(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
						ValaProperty* _tmp8_;
						_tmp8_ = NULL;
						vala_class_add_property (cl, _tmp8_ = vala_gir_parser_parse_property (self));
						(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
							ValaMethod* _tmp9_;
							_tmp9_ = NULL;
							vala_class_add_method (cl, _tmp9_ = vala_gir_parser_parse_constructor (self, cname));
							(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_code_node_unref (_tmp9_), NULL));
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
								ValaMethod* _tmp10_;
								_tmp10_ = NULL;
								gee_collection_add ((GeeCollection*) methods, _tmp10_ = vala_gir_parser_parse_method (self, "function"));
								(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
									ValaMethod* _tmp11_;
									_tmp11_ = NULL;
									gee_collection_add ((GeeCollection*) methods, _tmp11_ = vala_gir_parser_parse_method (self, "method"));
									(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_code_node_unref (_tmp11_), NULL));
								} else {
									if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "virtual-method") == 0) {
										ValaMethod* _tmp12_;
										_tmp12_ = NULL;
										gee_collection_add ((GeeCollection*) vmethods, _tmp12_ = vala_gir_parser_parse_method (self, "virtual-method"));
										(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL));
									} else {
										if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
											ValaSignal* _tmp13_;
											_tmp13_ = NULL;
											gee_collection_add ((GeeCollection*) signals, _tmp13_ = vala_gir_parser_parse_signal (self));
											(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_code_node_unref (_tmp13_), NULL));
										} else {
											char* _tmp15_;
											ValaSourceReference* _tmp14_;
											/* error*/
											_tmp15_ = NULL;
											_tmp14_ = NULL;
											vala_report_error (_tmp14_ = vala_gir_parser_get_current_src (self), _tmp15_ = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader)));
											_tmp15_ = (g_free (_tmp15_), NULL);
											(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_source_reference_unref (_tmp14_), NULL));
											break;
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	/* signal merging*/
	{
		GeeIterator* _sig_it;
		/* signal merging*/
		_sig_it = gee_iterable_iterator ((GeeIterable*) signals);
		/* signal merging*/
		while (TRUE) {
			ValaSignal* sig;
			ValaSymbol* symbol;
			/* signal merging*/
			if (!gee_iterator_next (_sig_it)) {
				/* signal merging*/
				break;
			}
			/* signal merging*/
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) sig));
			if (symbol == NULL) {
				vala_class_add_signal (cl, sig);
			} else {
				if (VALA_IS_PROPERTY (symbol)) {
				} else {
					char* _tmp17_;
					ValaSourceReference* _tmp16_;
					/* properties take precedence*/
					_tmp17_ = NULL;
					_tmp16_ = NULL;
					vala_report_error (_tmp16_ = vala_gir_parser_get_current_src (self), _tmp17_ = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) sig), vala_symbol_get_name ((ValaSymbol*) cl)));
					_tmp17_ = (g_free (_tmp17_), NULL);
					(_tmp16_ == NULL) ? NULL : (_tmp16_ = (vala_source_reference_unref (_tmp16_), NULL));
				}
			}
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	/* virtual method merging*/
	{
		GeeIterator* _m_it;
		/* virtual method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) vmethods);
		/* virtual method merging*/
		while (TRUE) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* virtual method merging*/
			if (!gee_iterator_next (_m_it)) {
				/* virtual method merging*/
				break;
			}
			/* virtual method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_class_add_method (cl, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp18_;
					ValaSignal* sig;
					_tmp18_ = NULL;
					sig = (_tmp18_ = VALA_SIGNAL (symbol), (_tmp18_ == NULL) ? NULL : vala_code_node_ref (_tmp18_));
					vala_signal_set_is_virtual (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					gboolean _tmp19_;
					_tmp19_ = FALSE;
					if (VALA_IS_PROPERTY (symbol)) {
						_tmp19_ = TRUE;
					} else {
						_tmp19_ = VALA_IS_FIELD (symbol);
					}
					if (_tmp19_) {
					} else {
						char* _tmp21_;
						ValaSourceReference* _tmp20_;
						/* assume method is getter for property/field ignore method*/
						_tmp21_ = NULL;
						_tmp20_ = NULL;
						vala_report_error (_tmp20_ = vala_gir_parser_get_current_src (self), _tmp21_ = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) cl)));
						_tmp21_ = (g_free (_tmp21_), NULL);
						(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_source_reference_unref (_tmp20_), NULL));
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* method merging*/
	{
		GeeIterator* _m_it;
		/* method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) methods);
		/* method merging*/
		while (TRUE) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* method merging*/
			if (!gee_iterator_next (_m_it)) {
				/* method merging*/
				break;
			}
			/* method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_class_add_method (cl, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp22_;
					ValaSignal* sig;
					_tmp22_ = NULL;
					sig = (_tmp22_ = VALA_SIGNAL (symbol), (_tmp22_ == NULL) ? NULL : vala_code_node_ref (_tmp22_));
					vala_signal_set_has_emitter (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					gboolean _tmp23_;
					_tmp23_ = FALSE;
					if (VALA_IS_PROPERTY (symbol)) {
						_tmp23_ = TRUE;
					} else {
						_tmp23_ = VALA_IS_FIELD (symbol);
					}
					if (_tmp23_) {
					} else {
						if (VALA_IS_METHOD (symbol)) {
						} else {
							char* _tmp25_;
							ValaSourceReference* _tmp24_;
							/* assume method is wrapper for virtual method*/
							_tmp25_ = NULL;
							_tmp24_ = NULL;
							vala_report_error (_tmp24_ = vala_gir_parser_get_current_src (self), _tmp25_ = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) cl)));
							_tmp25_ = (g_free (_tmp25_), NULL);
							(_tmp24_ == NULL) ? NULL : (_tmp24_ = (vala_source_reference_unref (_tmp24_), NULL));
						}
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* fields have lowest priority*/
	{
		GeeIterator* _f_it;
		/* fields have lowest priority*/
		_f_it = gee_iterable_iterator ((GeeIterable*) fields);
		/* fields have lowest priority*/
		while (TRUE) {
			ValaField* f;
			ValaSymbol* symbol;
			/* fields have lowest priority*/
			if (!gee_iterator_next (_f_it)) {
				/* fields have lowest priority*/
				break;
			}
			/* fields have lowest priority*/
			f = (ValaField*) gee_iterator_get (_f_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) f));
			if (symbol == NULL) {
				vala_class_add_field (cl, f);
			}
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	vala_gir_parser_end_element (self, "class");
	result = cl;
	cname = (g_free (cname), NULL);
	parent = (g_free (parent), NULL);
	(signals == NULL) ? NULL : (signals = (gee_collection_object_unref (signals), NULL));
	(methods == NULL) ? NULL : (methods = (gee_collection_object_unref (methods), NULL));
	(vmethods == NULL) ? NULL : (vmethods = (gee_collection_object_unref (vmethods), NULL));
	(fields == NULL) ? NULL : (fields = (gee_collection_object_unref (fields), NULL));
	return result;
}


static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self) {
	ValaInterface* result;
	ValaSourceReference* _tmp1_;
	char* _tmp0_;
	ValaInterface* _tmp2_;
	ValaInterface* iface;
	char* cname;
	GeeArrayList* methods;
	GeeArrayList* vmethods;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "interface");
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	iface = (_tmp2_ = vala_interface_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1_ = vala_gir_parser_get_current_src (self)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_reference_unref (_tmp1_), NULL)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
	vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_interface_set_cname (iface, cname);
	}
	vala_gir_parser_next (self);
	methods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	vmethods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "prerequisite") == 0) {
			ValaDataType* _tmp4_;
			char* _tmp3_;
			vala_gir_parser_start_element (self, "prerequisite");
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			vala_interface_add_prerequisite (iface, _tmp4_ = vala_gir_parser_parse_type_from_name (self, _tmp3_ = vala_markup_reader_get_attribute (self->priv->reader, "name")));
			(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
			_tmp3_ = (g_free (_tmp3_), NULL);
			vala_gir_parser_next (self);
			vala_gir_parser_end_element (self, "prerequisite");
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
				ValaField* _tmp5_;
				_tmp5_ = NULL;
				_tmp5_ = vala_gir_parser_parse_field (self);
				(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
					ValaProperty* _tmp6_;
					_tmp6_ = NULL;
					vala_interface_add_property (iface, _tmp6_ = vala_gir_parser_parse_property (self));
					(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "virtual-method") == 0) {
						ValaMethod* _tmp7_;
						_tmp7_ = NULL;
						gee_collection_add ((GeeCollection*) vmethods, _tmp7_ = vala_gir_parser_parse_method (self, "virtual-method"));
						(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
							ValaMethod* _tmp8_;
							_tmp8_ = NULL;
							gee_collection_add ((GeeCollection*) methods, _tmp8_ = vala_gir_parser_parse_method (self, "function"));
							(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
								ValaMethod* _tmp9_;
								_tmp9_ = NULL;
								gee_collection_add ((GeeCollection*) methods, _tmp9_ = vala_gir_parser_parse_method (self, "method"));
								(_tmp9_ == NULL) ? NULL : (_tmp9_ = (vala_code_node_unref (_tmp9_), NULL));
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
									ValaSignal* _tmp10_;
									_tmp10_ = NULL;
									vala_interface_add_signal (iface, _tmp10_ = vala_gir_parser_parse_signal (self));
									(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
								} else {
									char* _tmp12_;
									ValaSourceReference* _tmp11_;
									/* error*/
									_tmp12_ = NULL;
									_tmp11_ = NULL;
									vala_report_error (_tmp11_ = vala_gir_parser_get_current_src (self), _tmp12_ = g_strdup_printf ("unknown child element `%s' in `interface'", vala_markup_reader_get_name (self->priv->reader)));
									_tmp12_ = (g_free (_tmp12_), NULL);
									(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_source_reference_unref (_tmp11_), NULL));
									break;
								}
							}
						}
					}
				}
			}
		}
	}
	/* virtual method merging*/
	{
		GeeIterator* _m_it;
		/* virtual method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) vmethods);
		/* virtual method merging*/
		while (TRUE) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* virtual method merging*/
			if (!gee_iterator_next (_m_it)) {
				/* virtual method merging*/
				break;
			}
			/* virtual method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) iface), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_interface_add_method (iface, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp13_;
					ValaSignal* sig;
					_tmp13_ = NULL;
					sig = (_tmp13_ = VALA_SIGNAL (symbol), (_tmp13_ == NULL) ? NULL : vala_code_node_ref (_tmp13_));
					vala_signal_set_is_virtual (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					char* _tmp15_;
					ValaSourceReference* _tmp14_;
					_tmp15_ = NULL;
					_tmp14_ = NULL;
					vala_report_error (_tmp14_ = vala_gir_parser_get_current_src (self), _tmp15_ = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) iface)));
					_tmp15_ = (g_free (_tmp15_), NULL);
					(_tmp14_ == NULL) ? NULL : (_tmp14_ = (vala_source_reference_unref (_tmp14_), NULL));
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* method merging*/
	{
		GeeIterator* _m_it;
		/* method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) methods);
		/* method merging*/
		while (TRUE) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* method merging*/
			if (!gee_iterator_next (_m_it)) {
				/* method merging*/
				break;
			}
			/* method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) iface), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_interface_add_method (iface, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp16_;
					ValaSignal* sig;
					_tmp16_ = NULL;
					sig = (_tmp16_ = VALA_SIGNAL (symbol), (_tmp16_ == NULL) ? NULL : vala_code_node_ref (_tmp16_));
					vala_signal_set_has_emitter (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					if (VALA_IS_METHOD (symbol)) {
					} else {
						char* _tmp18_;
						ValaSourceReference* _tmp17_;
						/* assume method is wrapper for virtual method*/
						_tmp18_ = NULL;
						_tmp17_ = NULL;
						vala_report_error (_tmp17_ = vala_gir_parser_get_current_src (self), _tmp18_ = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) iface)));
						_tmp18_ = (g_free (_tmp18_), NULL);
						(_tmp17_ == NULL) ? NULL : (_tmp17_ = (vala_source_reference_unref (_tmp17_), NULL));
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	vala_gir_parser_end_element (self, "interface");
	result = iface;
	cname = (g_free (cname), NULL);
	(methods == NULL) ? NULL : (methods = (gee_collection_object_unref (methods), NULL));
	(vmethods == NULL) ? NULL : (vmethods = (gee_collection_object_unref (vmethods), NULL));
	return result;
}


static ValaField* vala_gir_parser_parse_field (ValaGirParser* self) {
	ValaField* result;
	char* name;
	char* allow_none;
	ValaDataType* type;
	ValaSourceReference* _tmp0_;
	ValaField* _tmp1_;
	ValaField* field;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "field");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	allow_none = vala_markup_reader_get_attribute (self->priv->reader, "allow-none");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	field = (_tmp1_ = vala_field_new (name, type, NULL, _tmp0_ = vala_gir_parser_get_current_src (self)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_source_reference_unref (_tmp0_), NULL)), _tmp1_);
	vala_symbol_set_access ((ValaSymbol*) field, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (allow_none, "1") == 0) {
		vala_data_type_set_nullable (type, TRUE);
	}
	vala_gir_parser_end_element (self, "field");
	result = field;
	name = (g_free (name), NULL);
	allow_none = (g_free (allow_none), NULL);
	(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	return result;
}


static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self) {
	ValaProperty* result;
	gint _tmp2__length1;
	char** _tmp2_;
	char** _tmp1_;
	char* _tmp0_;
	char* _tmp3_;
	char* name;
	char* readable;
	char* writable;
	char* construct_;
	char* construct_only;
	ValaDataType* type;
	ValaSourceReference* _tmp4_;
	ValaProperty* _tmp5_;
	ValaProperty* prop;
	gboolean _tmp8_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "property");
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp3_ = NULL;
	name = (_tmp3_ = g_strjoinv ("_", (_tmp2_ = _tmp1_ = g_strsplit (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), "-", 0), _tmp2__length1 = _vala_array_length (_tmp1_), _tmp2_)), _tmp2_ = (_vala_array_free (_tmp2_, _tmp2__length1, (GDestroyNotify) g_free), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp3_);
	readable = vala_markup_reader_get_attribute (self->priv->reader, "readable");
	writable = vala_markup_reader_get_attribute (self->priv->reader, "writable");
	construct_ = vala_markup_reader_get_attribute (self->priv->reader, "construct");
	construct_only = vala_markup_reader_get_attribute (self->priv->reader, "construct-only");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL, NULL);
	_tmp4_ = NULL;
	_tmp5_ = NULL;
	prop = (_tmp5_ = vala_property_new (name, type, NULL, NULL, _tmp4_ = vala_gir_parser_get_current_src (self)), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_source_reference_unref (_tmp4_), NULL)), _tmp5_);
	vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (readable, "0") != 0) {
		ValaPropertyAccessor* _tmp7_;
		ValaDataType* _tmp6_;
		_tmp7_ = NULL;
		_tmp6_ = NULL;
		vala_property_set_get_accessor (prop, _tmp7_ = vala_property_accessor_new (TRUE, FALSE, FALSE, _tmp6_ = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp7_ == NULL) ? NULL : (_tmp7_ = (vala_code_node_unref (_tmp7_), NULL));
		(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
	}
	_tmp8_ = FALSE;
	if (_vala_strcmp0 (writable, "1") == 0) {
		_tmp8_ = TRUE;
	} else {
		_tmp8_ = _vala_strcmp0 (construct_only, "1") == 0;
	}
	if (_tmp8_) {
		gboolean _tmp9_;
		gboolean _tmp10_;
		ValaPropertyAccessor* _tmp12_;
		ValaDataType* _tmp11_;
		_tmp9_ = FALSE;
		if (_vala_strcmp0 (construct_only, "1") != 0) {
			_tmp9_ = _vala_strcmp0 (writable, "1") == 0;
		} else {
			_tmp9_ = FALSE;
		}
		_tmp10_ = FALSE;
		if (_vala_strcmp0 (construct_only, "1") == 0) {
			_tmp10_ = TRUE;
		} else {
			_tmp10_ = _vala_strcmp0 (construct_, "1") == 0;
		}
		_tmp12_ = NULL;
		_tmp11_ = NULL;
		vala_property_set_set_accessor (prop, _tmp12_ = vala_property_accessor_new (FALSE, _tmp9_, _tmp10_, _tmp11_ = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp12_ == NULL) ? NULL : (_tmp12_ = (vala_code_node_unref (_tmp12_), NULL));
		(_tmp11_ == NULL) ? NULL : (_tmp11_ = (vala_code_node_unref (_tmp11_), NULL));
	}
	vala_gir_parser_end_element (self, "property");
	result = prop;
	name = (g_free (name), NULL);
	readable = (g_free (readable), NULL);
	writable = (g_free (writable), NULL);
	construct_ = (g_free (construct_), NULL);
	construct_only = (g_free (construct_only), NULL);
	(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	return result;
}


static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self) {
	ValaDelegate* result;
	char* name;
	ValaDataType* return_type;
	gboolean _tmp0_;
	ValaSourceReference* _tmp3_;
	ValaDelegate* _tmp4_;
	ValaDelegate* d;
	gboolean _tmp5_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "callback");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp0_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		ValaDataType* _tmp1_;
		_tmp1_ = NULL;
		return_type = (_tmp1_ = vala_gir_parser_parse_return_value (self, NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp1_);
	} else {
		ValaDataType* _tmp2_;
		_tmp2_ = NULL;
		return_type = (_tmp2_ = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp2_);
	}
	_tmp3_ = NULL;
	_tmp4_ = NULL;
	d = (_tmp4_ = vala_delegate_new (name, return_type, _tmp3_ = vala_gir_parser_get_current_src (self)), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_source_reference_unref (_tmp3_), NULL)), _tmp4_);
	vala_symbol_set_access ((ValaSymbol*) d, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp5_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp5_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp5_ = FALSE;
	}
	if (_tmp5_) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (TRUE) {
			ValaFormalParameter* _tmp6_;
			if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
				break;
			}
			_tmp6_ = NULL;
			vala_delegate_add_parameter (d, _tmp6_ = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "callback");
	result = d;
	name = (g_free (name), NULL);
	(return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL));
	return result;
}


static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self, const char* parent_ctype) {
	ValaMethod* result;
	char* name;
	char* ctype;
	ValaDataType* _tmp3_;
	char* _tmp2_;
	ValaDataType* _tmp1_;
	char* _tmp0_;
	ValaSourceReference* _tmp4_;
	ValaCreationMethod* _tmp5_;
	ValaCreationMethod* m;
	gboolean _tmp6_;
	gboolean _tmp9_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constructor");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	ctype = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp3_ = (_tmp1_ = vala_gir_parser_parse_return_value (self, &_tmp0_), ctype = (_tmp2_ = _tmp0_, ctype = (g_free (ctype), NULL), _tmp2_), _tmp1_);
	(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
	_tmp4_ = NULL;
	_tmp5_ = NULL;
	m = (_tmp5_ = vala_creation_method_new (NULL, name, _tmp4_ = vala_gir_parser_get_current_src (self)), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_source_reference_unref (_tmp4_), NULL)), _tmp5_);
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_has_construct_function ((ValaMethod*) m, FALSE);
	_tmp6_ = FALSE;
	if (ctype != NULL) {
		gboolean _tmp7_;
		_tmp7_ = FALSE;
		if (parent_ctype == NULL) {
			_tmp7_ = TRUE;
		} else {
			char* _tmp8_;
			_tmp8_ = NULL;
			_tmp7_ = _vala_strcmp0 (ctype, _tmp8_ = g_strconcat (parent_ctype, "*", NULL)) != 0;
			_tmp8_ = (g_free (_tmp8_), NULL);
		}
		_tmp6_ = _tmp7_;
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		vala_creation_method_set_custom_return_type_cname (m, ctype);
	}
	if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "new_")) {
		vala_symbol_set_name ((ValaSymbol*) m, g_utf8_offset_to_pointer (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("new_", -1)));
	}
	_tmp9_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp9_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp9_ = FALSE;
	}
	if (_tmp9_) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (TRUE) {
			ValaFormalParameter* _tmp10_;
			if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
				break;
			}
			_tmp10_ = NULL;
			vala_method_add_parameter ((ValaMethod*) m, _tmp10_ = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "constructor");
	result = (ValaMethod*) m;
	name = (g_free (name), NULL);
	ctype = (g_free (ctype), NULL);
	return result;
}


static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self, const char* element_name) {
	ValaMethod* result;
	char* name;
	char* throws_string;
	char* invoker;
	ValaDataType* return_type;
	gboolean _tmp0_;
	ValaSourceReference* _tmp3_;
	ValaMethod* _tmp4_;
	ValaMethod* m;
	gboolean _tmp5_;
	GeeArrayList* parameters;
	GeeArrayList* array_length_parameters;
	GeeArrayList* closure_parameters;
	GeeArrayList* destroy_parameters;
	gboolean _tmp6_;
	gint i;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (element_name != NULL, NULL);
	vala_gir_parser_start_element (self, element_name);
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	throws_string = vala_markup_reader_get_attribute (self->priv->reader, "throws");
	invoker = vala_markup_reader_get_attribute (self->priv->reader, "invoker");
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp0_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		ValaDataType* _tmp1_;
		_tmp1_ = NULL;
		return_type = (_tmp1_ = vala_gir_parser_parse_return_value (self, NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp1_);
	} else {
		ValaDataType* _tmp2_;
		_tmp2_ = NULL;
		return_type = (_tmp2_ = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp2_);
	}
	_tmp3_ = NULL;
	_tmp4_ = NULL;
	m = (_tmp4_ = vala_method_new (name, return_type, _tmp3_ = vala_gir_parser_get_current_src (self)), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_source_reference_unref (_tmp3_), NULL)), _tmp4_);
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp5_ = FALSE;
	if (_vala_strcmp0 (element_name, "virtual-method") == 0) {
		_tmp5_ = TRUE;
	} else {
		_tmp5_ = _vala_strcmp0 (element_name, "callback") == 0;
	}
	if (_tmp5_) {
		vala_method_set_is_virtual (m, TRUE);
		if (invoker != NULL) {
			vala_symbol_set_name ((ValaSymbol*) m, invoker);
		}
	} else {
		if (_vala_strcmp0 (element_name, "function") == 0) {
			vala_method_set_binding (m, MEMBER_BINDING_STATIC);
		}
	}
	parameters = gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	array_length_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	closure_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	destroy_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	_tmp6_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp6_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		gboolean first;
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		first = TRUE;
		while (TRUE) {
			gint array_length_idx;
			gint closure_idx;
			gint destroy_idx;
			ValaFormalParameter* param;
			gboolean _tmp7_;
			if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
				break;
			}
			array_length_idx = 0;
			closure_idx = 0;
			destroy_idx = 0;
			param = vala_gir_parser_parse_parameter (self, &array_length_idx, &closure_idx, &destroy_idx);
			if (array_length_idx != (-1)) {
				gee_collection_add ((GeeCollection*) array_length_parameters, GINT_TO_POINTER (array_length_idx));
			}
			if (closure_idx != (-1)) {
				gee_collection_add ((GeeCollection*) closure_parameters, GINT_TO_POINTER (closure_idx));
			}
			if (destroy_idx != (-1)) {
				gee_collection_add ((GeeCollection*) destroy_parameters, GINT_TO_POINTER (destroy_idx));
			}
			_tmp7_ = FALSE;
			if (_vala_strcmp0 (element_name, "callback") != 0) {
				_tmp7_ = TRUE;
			} else {
				_tmp7_ = !first;
			}
			/* first parameter is instance pointer in virtual methods, ignore*/
			if (_tmp7_) {
				gee_collection_add ((GeeCollection*) parameters, param);
			} else {
				first = FALSE;
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	i = 0;
	if (_vala_strcmp0 (element_name, "method") == 0) {
		/* implicit instance parameter*/
		i++;
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) parameters);
		while (TRUE) {
			ValaFormalParameter* param;
			gboolean _tmp8_;
			gboolean _tmp9_;
			if (!gee_iterator_next (_param_it)) {
				break;
			}
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp8_ = FALSE;
			_tmp9_ = FALSE;
			if (!gee_collection_contains ((GeeCollection*) array_length_parameters, GINT_TO_POINTER (i))) {
				_tmp9_ = !gee_collection_contains ((GeeCollection*) closure_parameters, GINT_TO_POINTER (i));
			} else {
				_tmp9_ = FALSE;
			}
			if (_tmp9_) {
				_tmp8_ = !gee_collection_contains ((GeeCollection*) destroy_parameters, GINT_TO_POINTER (i));
			} else {
				_tmp8_ = FALSE;
			}
			if (_tmp8_) {
				vala_method_add_parameter (m, param);
			}
			i++;
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	if (_vala_strcmp0 (throws_string, "1") == 0) {
		ValaErrorType* _tmp10_;
		_tmp10_ = NULL;
		vala_code_node_add_error_type ((ValaCodeNode*) m, (ValaDataType*) (_tmp10_ = vala_error_type_new (NULL, NULL, NULL)));
		(_tmp10_ == NULL) ? NULL : (_tmp10_ = (vala_code_node_unref (_tmp10_), NULL));
	}
	vala_gir_parser_end_element (self, element_name);
	result = m;
	name = (g_free (name), NULL);
	throws_string = (g_free (throws_string), NULL);
	invoker = (g_free (invoker), NULL);
	(return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL));
	(parameters == NULL) ? NULL : (parameters = (gee_collection_object_unref (parameters), NULL));
	(array_length_parameters == NULL) ? NULL : (array_length_parameters = (gee_collection_object_unref (array_length_parameters), NULL));
	(closure_parameters == NULL) ? NULL : (closure_parameters = (gee_collection_object_unref (closure_parameters), NULL));
	(destroy_parameters == NULL) ? NULL : (destroy_parameters = (gee_collection_object_unref (destroy_parameters), NULL));
	return result;
}


static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self) {
	ValaSignal* result;
	gint _tmp2__length1;
	char** _tmp2_;
	char** _tmp1_;
	char* _tmp0_;
	char* _tmp3_;
	char* name;
	ValaDataType* return_type;
	gboolean _tmp4_;
	ValaSignal* sig;
	gboolean _tmp7_;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:signal");
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp3_ = NULL;
	name = (_tmp3_ = g_strjoinv ("_", (_tmp2_ = _tmp1_ = g_strsplit (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), "-", 0), _tmp2__length1 = _vala_array_length (_tmp1_), _tmp2_)), _tmp2_ = (_vala_array_free (_tmp2_, _tmp2__length1, (GDestroyNotify) g_free), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp3_);
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp4_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp4_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		ValaDataType* _tmp5_;
		_tmp5_ = NULL;
		return_type = (_tmp5_ = vala_gir_parser_parse_return_value (self, NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp5_);
	} else {
		ValaDataType* _tmp6_;
		_tmp6_ = NULL;
		return_type = (_tmp6_ = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp6_);
	}
	sig = vala_signal_new (name, return_type, NULL);
	vala_symbol_set_access ((ValaSymbol*) sig, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp7_ = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp7_ = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp7_ = FALSE;
	}
	if (_tmp7_) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (TRUE) {
			ValaFormalParameter* _tmp8_;
			if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
				break;
			}
			_tmp8_ = NULL;
			vala_signal_add_parameter (sig, _tmp8_ = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp8_ == NULL) ? NULL : (_tmp8_ = (vala_code_node_unref (_tmp8_), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "glib:signal");
	result = sig;
	name = (g_free (name), NULL);
	(return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL));
	return result;
}


static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self) {
	ValaStruct* result;
	char* _tmp0_;
	ValaStruct* _tmp1_;
	ValaStruct* st;
	char* cname;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:boxed");
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	st = (_tmp1_ = vala_struct_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "glib:name"), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_struct_set_cname (st, cname);
	}
	vala_gir_parser_next (self);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2_;
			_tmp2_ = NULL;
			vala_struct_add_field (st, _tmp2_ = vala_gir_parser_parse_field (self));
			(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_code_node_unref (_tmp2_), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3_;
				_tmp3_ = NULL;
				_tmp3_ = vala_gir_parser_parse_constructor (self, NULL);
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4_;
					_tmp4_ = NULL;
					vala_struct_add_method (st, _tmp4_ = vala_gir_parser_parse_method (self, "method"));
					(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
				} else {
					char* _tmp6_;
					ValaSourceReference* _tmp5_;
					/* error*/
					_tmp6_ = NULL;
					_tmp5_ = NULL;
					vala_report_error (_tmp5_ = vala_gir_parser_get_current_src (self), _tmp6_ = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader)));
					_tmp6_ = (g_free (_tmp6_), NULL);
					(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_source_reference_unref (_tmp5_), NULL));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "glib:boxed");
	result = st;
	cname = (g_free (cname), NULL);
	return result;
}


static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self) {
	ValaStruct* result;
	char* _tmp0_;
	ValaStruct* _tmp1_;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "union");
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	st = (_tmp1_ = vala_struct_new (_tmp0_ = vala_markup_reader_get_attribute (self->priv->reader, "name"), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (TRUE) {
		if (!(self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT)) {
			break;
		}
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2_;
			_tmp2_ = NULL;
			vala_struct_add_field (st, _tmp2_ = vala_gir_parser_parse_field (self));
			(_tmp2_ == NULL) ? NULL : (_tmp2_ = (vala_code_node_unref (_tmp2_), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3_;
				_tmp3_ = NULL;
				_tmp3_ = vala_gir_parser_parse_constructor (self, NULL);
				(_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_node_unref (_tmp3_), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4_;
					_tmp4_ = NULL;
					vala_struct_add_method (st, _tmp4_ = vala_gir_parser_parse_method (self, "method"));
					(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_code_node_unref (_tmp4_), NULL));
				} else {
					char* _tmp6_;
					ValaSourceReference* _tmp5_;
					/* error*/
					_tmp6_ = NULL;
					_tmp5_ = NULL;
					vala_report_error (_tmp5_ = vala_gir_parser_get_current_src (self), _tmp6_ = g_strdup_printf ("unknown child element `%s' in `union'", vala_markup_reader_get_name (self->priv->reader)));
					_tmp6_ = (g_free (_tmp6_), NULL);
					(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_source_reference_unref (_tmp5_), NULL));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "union");
	result = st;
	return result;
}


static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self) {
	ValaConstant* result;
	char* name;
	ValaDataType* type;
	ValaSourceReference* _tmp0_;
	ValaConstant* _tmp1_;
	ValaConstant* c;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constant");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL, NULL);
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	c = (_tmp1_ = vala_constant_new (name, type, NULL, _tmp0_ = vala_gir_parser_get_current_src (self)), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_source_reference_unref (_tmp0_), NULL)), _tmp1_);
	vala_symbol_set_access ((ValaSymbol*) c, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_symbol_set_external ((ValaSymbol*) c, TRUE);
	vala_gir_parser_end_element (self, "constant");
	result = c;
	name = (g_free (name), NULL);
	(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	return result;
}


static char* string_substring (const char* self, glong offset, glong len) {
	char* result;
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	result = g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
	return result;
}


static glong string_get_length (const char* self) {
	glong result;
	g_return_val_if_fail (self != NULL, 0L);
	result = g_utf8_strlen (self, -1);
	return result;
}


void vala_gir_parser_parse_metadata (ValaGirParser* self, const char* metadata_filename) {
	GError * _inner_error_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (metadata_filename != NULL);
	_inner_error_ = NULL;
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			char* _tmp2_;
			gboolean _tmp1_;
			char* _tmp0_;
			metadata = NULL;
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			_tmp1_ = g_file_get_contents (metadata_filename, &_tmp0_, NULL, &_inner_error_);
			metadata = (_tmp2_ = _tmp0_, metadata = (g_free (metadata), NULL), _tmp2_);
			_tmp1_;
			if (_inner_error_ != NULL) {
				metadata = (g_free (metadata), NULL);
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch2_g_file_error;
				}
				goto __finally2;
			}
			{
				char** _tmp3_;
				char** line_collection;
				int line_collection_length1;
				int line_it;
				_tmp3_ = NULL;
				line_collection = _tmp3_ = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = _vala_array_length (_tmp3_);
				for (line_it = 0; line_it < _vala_array_length (_tmp3_); line_it = line_it + 1) {
					const char* _tmp12_;
					char* line;
					_tmp12_ = NULL;
					line = (_tmp12_ = line_collection[line_it], (_tmp12_ == NULL) ? NULL : g_strdup (_tmp12_));
					{
						char** _tmp5_;
						gint tokens_size;
						gint tokens_length1;
						char** _tmp4_;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							/* ignore comment lines*/
							line = (g_free (line), NULL);
							continue;
						}
						_tmp5_ = NULL;
						_tmp4_ = NULL;
						tokens = (_tmp5_ = _tmp4_ = g_strsplit (line, " ", 2), tokens_length1 = _vala_array_length (_tmp4_), tokens_size = tokens_length1, _tmp5_);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
							continue;
						}
						{
							char** _tmp6_;
							char** attribute_collection;
							int attribute_collection_length1;
							int attribute_it;
							_tmp6_ = NULL;
							attribute_collection = _tmp6_ = g_strsplit (tokens[1], " ", 0);
							attribute_collection_length1 = _vala_array_length (_tmp6_);
							for (attribute_it = 0; attribute_it < _vala_array_length (_tmp6_); attribute_it = attribute_it + 1) {
								const char* _tmp11_;
								char* attribute;
								_tmp11_ = NULL;
								attribute = (_tmp11_ = attribute_collection[attribute_it], (_tmp11_ == NULL) ? NULL : g_strdup (_tmp11_));
								{
									char** _tmp8_;
									gint pair_size;
									gint pair_length1;
									char** _tmp7_;
									char** pair;
									gboolean _tmp9_;
									char* key;
									char* _tmp10_;
									_tmp8_ = NULL;
									_tmp7_ = NULL;
									pair = (_tmp8_ = _tmp7_ = g_strsplit (attribute, "=", 2), pair_length1 = _vala_array_length (_tmp7_), pair_size = pair_length1, _tmp8_);
									_tmp9_ = FALSE;
									if (pair[0] == NULL) {
										_tmp9_ = TRUE;
									} else {
										_tmp9_ = pair[1] == NULL;
									}
									if (_tmp9_) {
										attribute = (g_free (attribute), NULL);
										pair = (_vala_array_free (pair, pair_length1, (GDestroyNotify) g_free), NULL);
										continue;
									}
									key = g_strdup_printf ("%s/@%s", tokens[0], pair[0]);
									_tmp10_ = NULL;
									gee_map_set ((GeeMap*) self->priv->attributes_map, key, _tmp10_ = string_substring (pair[1], (glong) 1, string_get_length (pair[1]) - 2));
									_tmp10_ = (g_free (_tmp10_), NULL);
									attribute = (g_free (attribute), NULL);
									pair = (_vala_array_free (pair, pair_length1, (GDestroyNotify) g_free), NULL);
									key = (g_free (key), NULL);
								}
							}
							attribute_collection = (_vala_array_free (attribute_collection, attribute_collection_length1, (GDestroyNotify) g_free), NULL);
						}
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally2;
		__catch2_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp13_;
				_tmp13_ = NULL;
				vala_report_error (NULL, _tmp13_ = g_strdup_printf ("Unable to read metadata file: %s", e->message));
				_tmp13_ = (g_free (_tmp13_), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally2:
		if (_inner_error_ != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return;
		}
	} else {
		char* _tmp14_;
		_tmp14_ = NULL;
		vala_report_error (NULL, _tmp14_ = g_strdup_printf ("Metadata file `%s' not found", metadata_filename));
		_tmp14_ = (g_free (_tmp14_), NULL);
	}
}


/**
 * Code visitor parsing all Vala source files.
 */
ValaGirParser* vala_gir_parser_construct (GType object_type) {
	ValaGirParser* self;
	self = (ValaGirParser*) g_type_create_instance (object_type);
	return self;
}


ValaGirParser* vala_gir_parser_new (void) {
	return vala_gir_parser_construct (VALA_TYPE_GIR_PARSER);
}


const char* vala_gir_parser_get_package_name (ValaGirParser* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_package_name;
	return result;
}


static void vala_gir_parser_set_package_name (ValaGirParser* self, const char* value) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_package_name = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->_package_name = (g_free (self->priv->_package_name), NULL), _tmp1_);
}


static void vala_gir_parser_class_init (ValaGirParserClass * klass) {
	vala_gir_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gir_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaGirParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gir_parser_real_visit_source_file;
}


static void vala_gir_parser_instance_init (ValaGirParser * self) {
	self->priv = VALA_GIR_PARSER_GET_PRIVATE (self);
	self->priv->attributes_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	self->priv->gtype_callbacks = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, GEE_TYPE_ARRAY_LIST, (GBoxedCopyFunc) gee_collection_object_ref, gee_collection_object_unref, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_gir_parser_finalize (ValaCodeVisitor* obj) {
	ValaGirParser * self;
	self = VALA_GIR_PARSER (obj);
	self->priv->_package_name = (g_free (self->priv->_package_name), NULL);
	(self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->glib_ns == NULL) ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL));
	(self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL));
	self->priv->cheader_filenames = (_vala_array_free (self->priv->cheader_filenames, self->priv->cheader_filenames_length1, (GDestroyNotify) g_free), NULL);
	(self->priv->attributes_map == NULL) ? NULL : (self->priv->attributes_map = (gee_collection_object_unref (self->priv->attributes_map), NULL));
	(self->priv->gtype_callbacks == NULL) ? NULL : (self->priv->gtype_callbacks = (gee_collection_object_unref (self->priv->gtype_callbacks), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gir_parser_parent_class)->finalize (obj);
}


GType vala_gir_parser_get_type (void) {
	static GType vala_gir_parser_type_id = 0;
	if (vala_gir_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGirParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gir_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGirParser), 0, (GInstanceInitFunc) vala_gir_parser_instance_init, NULL };
		vala_gir_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGirParser", &g_define_type_info, 0);
	}
	return vala_gir_parser_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




