/* valaenum.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;
typedef struct _ValaTypeSymbolPrivate ValaTypeSymbolPrivate;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;
typedef struct _ValaEnumPrivate ValaEnumPrivate;

#define VALA_TYPE_ENUM_VALUE (vala_enum_value_get_type ())
#define VALA_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM_VALUE, ValaEnumValue))
#define VALA_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM_VALUE, ValaEnumValueClass))
#define VALA_IS_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM_VALUE))
#define VALA_IS_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM_VALUE))
#define VALA_ENUM_VALUE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM_VALUE, ValaEnumValueClass))

typedef struct _ValaEnumValue ValaEnumValue;
typedef struct _ValaEnumValueClass ValaEnumValueClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_VALUE_TYPE (vala_value_type_get_type ())
#define VALA_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VALUE_TYPE, ValaValueType))
#define VALA_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VALUE_TYPE, ValaValueTypeClass))
#define VALA_IS_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VALUE_TYPE))
#define VALA_IS_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VALUE_TYPE))
#define VALA_VALUE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VALUE_TYPE, ValaValueTypeClass))

typedef struct _ValaValueType ValaValueType;
typedef struct _ValaValueTypeClass ValaValueTypeClass;

#define VALA_TYPE_ENUM_VALUE_TYPE (vala_enum_value_type_get_type ())
#define VALA_ENUM_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueType))
#define VALA_ENUM_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueTypeClass))
#define VALA_IS_ENUM_VALUE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM_VALUE_TYPE))
#define VALA_IS_ENUM_VALUE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM_VALUE_TYPE))
#define VALA_ENUM_VALUE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM_VALUE_TYPE, ValaEnumValueTypeClass))

typedef struct _ValaEnumValueType ValaEnumValueType;
typedef struct _ValaEnumValueTypeClass ValaEnumValueTypeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a runtime data type. This data type may be defined in Vala source
 * code or imported from an external library with a Vala API file.
 */
struct _ValaTypeSymbol {
	ValaSymbol parent_instance;
	ValaTypeSymbolPrivate * priv;
};

struct _ValaTypeSymbolClass {
	ValaSymbolClass parent_class;
	char* (*get_cname) (ValaTypeSymbol* self, gboolean const_type);
	gboolean (*is_reference_type) (ValaTypeSymbol* self);
	char* (*get_dup_function) (ValaTypeSymbol* self);
	char* (*get_free_function) (ValaTypeSymbol* self);
	char* (*get_copy_function) (ValaTypeSymbol* self);
	char* (*get_destroy_function) (ValaTypeSymbol* self);
	gboolean (*is_reference_counting) (ValaTypeSymbol* self);
	char* (*get_ref_function) (ValaTypeSymbol* self);
	char* (*get_unref_function) (ValaTypeSymbol* self);
	char* (*get_ref_sink_function) (ValaTypeSymbol* self);
	char* (*get_type_id) (ValaTypeSymbol* self);
	char* (*get_marshaller_type_name) (ValaTypeSymbol* self);
	char* (*get_param_spec_function) (ValaTypeSymbol* self);
	char* (*get_get_value_function) (ValaTypeSymbol* self);
	char* (*get_set_value_function) (ValaTypeSymbol* self);
	char* (*get_upper_case_cname) (ValaTypeSymbol* self, const char* infix);
	char* (*get_default_value) (ValaTypeSymbol* self);
	gboolean (*is_subtype_of) (ValaTypeSymbol* self, ValaTypeSymbol* t);
	gint (*get_type_parameter_index) (ValaTypeSymbol* self, const char* name);
	char* (*get_type_signature) (ValaTypeSymbol* self);
};

/**
 * Represents an enum declaration in the source code.
 */
struct _ValaEnum {
	ValaTypeSymbol parent_instance;
	ValaEnumPrivate * priv;
};

struct _ValaEnumClass {
	ValaTypeSymbolClass parent_class;
};

struct _ValaEnumPrivate {
	gboolean _is_flags;
	gboolean _has_type_id;
	GeeList* values;
	GeeList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	char* type_id;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_enum_get_type (void);
GType vala_enum_value_get_type (void);
GType vala_member_get_type (void);
GType vala_method_get_type (void);
#define VALA_ENUM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM, ValaEnumPrivate))
enum  {
	VALA_ENUM_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaTypeSymbol* vala_typesymbol_new (const char* name, ValaSourceReference* source_reference);
ValaTypeSymbol* vala_typesymbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference);
ValaEnum* vala_enum_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
const char* vala_symbol_get_name (ValaSymbol* self);
void vala_enum_add_value (ValaEnum* self, ValaEnumValue* value);
GType vala_creation_method_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType member_binding_get_type (void);
MemberBinding vala_method_get_binding (ValaMethod* self);
ValaEnumValueType* vala_enum_value_type_new (ValaEnum* type_symbol);
ValaEnumValueType* vala_enum_value_type_construct (GType object_type, ValaEnum* type_symbol);
GType vala_value_type_get_type (void);
GType vala_enum_value_type_get_type (void);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
GType vala_formal_parameter_get_type (void);
void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value);
ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self);
void vala_enum_add_method (ValaEnum* self, ValaMethod* m);
GeeList* vala_enum_get_values (ValaEnum* self);
GeeList* vala_enum_get_methods (ValaEnum* self);
void vala_code_visitor_visit_enum (ValaCodeVisitor* self, ValaEnum* en);
static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaAttribute* vala_code_node_get_attribute (ValaCodeNode* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_enum_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
void vala_enum_set_cname (ValaEnum* self, const char* cname);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
static char* vala_enum_get_lower_case_csuffix (ValaEnum* self);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_enum_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static gboolean vala_enum_real_is_reference_type (ValaTypeSymbol* base);
char* vala_typesymbol_get_upper_case_cname (ValaTypeSymbol* self, const char* infix);
static char* vala_enum_real_get_cprefix (ValaSymbol* base);
void vala_enum_set_cprefix (ValaEnum* self, const char* cprefix);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
void vala_typesymbol_add_cheader_filename (ValaTypeSymbol* self, const char* filename);
gboolean vala_attribute_get_bool (ValaAttribute* self, const char* name);
void vala_enum_set_has_type_id (ValaEnum* self, gboolean value);
static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_enum_set_is_flags (ValaEnum* self, gboolean value);
void vala_enum_process_attributes (ValaEnum* self);
void vala_enum_set_type_id (ValaEnum* self, const char* type_id);
gboolean vala_enum_get_has_type_id (ValaEnum* self);
gboolean vala_enum_get_is_flags (ValaEnum* self);
static char* vala_enum_real_get_type_id (ValaTypeSymbol* base);
static char* vala_enum_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_enum_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_enum_real_get_set_value_function (ValaTypeSymbol* base);
static char* vala_enum_real_get_default_value (ValaTypeSymbol* base);
static char* vala_enum_real_get_type_signature (ValaTypeSymbol* base);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_semantic_analyzer_get_current_source_file (ValaSemanticAnalyzer* self);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
void vala_semantic_analyzer_set_current_source_file (ValaSemanticAnalyzer* self, ValaSourceFile* value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_enum_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_enum_parent_class = NULL;
static void vala_enum_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new enum.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created enum
 */
ValaEnum* vala_enum_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaEnum* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaEnum*) vala_typesymbol_construct (object_type, name, source_reference);
	return self;
}


ValaEnum* vala_enum_new (const char* name, ValaSourceReference* source_reference) {
	return vala_enum_construct (VALA_TYPE_ENUM, name, source_reference);
}


/**
 * Appends the specified enum value to the list of values.
 *
 * @param value an enum value
 */
void vala_enum_add_value (ValaEnum* self, ValaEnumValue* value) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (value != NULL);
	gee_collection_add ((GeeCollection*) self->priv->values, value);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) value), (ValaSymbol*) value);
}


/**
 * Adds the specified method as a member to this enum.
 *
 * @param m a method
 */
void vala_enum_add_method (ValaEnum* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1_;
		ValaEnumValueType* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		vala_method_set_this_parameter (m, _tmp1_ = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0_ = vala_enum_value_type_new (self)), NULL));
		(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_code_node_unref (_tmp1_), NULL));
		(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of enum values.
 *
 * @return list of enum values
 */
GeeList* vala_enum_get_values (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM_VALUE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->values);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeList* vala_enum_get_methods (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


static void vala_enum_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_enum (visitor, self);
}


static void vala_enum_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _value_it;
		_value_it = gee_iterable_iterator ((GeeIterable*) self->priv->values);
		while (gee_iterator_next (_value_it)) {
			ValaEnumValue* value;
			value = (ValaEnumValue*) gee_iterator_get (_value_it);
			vala_code_node_accept ((ValaCodeNode*) value, visitor);
			(value == NULL) ? NULL : (value = (vala_code_node_unref (value), NULL));
		}
		(_value_it == NULL) ? NULL : (_value_it = (gee_collection_object_unref (_value_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
}


static char* vala_enum_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaEnum * self;
	const char* _tmp3_;
	self = (ValaEnum*) base;
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
		if (attr != NULL) {
			char* _tmp0_;
			_tmp0_ = NULL;
			self->priv->cname = (_tmp0_ = vala_attribute_get_string (attr, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0_);
		}
		if (self->priv->cname == NULL) {
			char* _tmp2_;
			char* _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			self->priv->cname = (_tmp2_ = g_strdup_printf ("%s%s", _tmp1_ = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp2_);
			_tmp1_ = (g_free (_tmp1_), NULL);
		}
		(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
	}
	_tmp3_ = NULL;
	return (_tmp3_ = self->priv->cname, (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
}


void vala_enum_set_cname (ValaEnum* self, const char* cname) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->cname = (_tmp1_ = (_tmp0_ = cname, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1_);
}


static char* vala_enum_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaEnum * self;
	const char* _tmp2_;
	self = (ValaEnum*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		self->priv->lower_case_cprefix = (_tmp1_ = g_strdup_printf ("%s_", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1_);
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	_tmp2_ = NULL;
	return (_tmp2_ = self->priv->lower_case_cprefix, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
}


static char* vala_enum_get_lower_case_csuffix (ValaEnum* self) {
	const char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->lower_case_csuffix = (_tmp0_ = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0_);
	}
	_tmp1_ = NULL;
	return (_tmp1_ = self->priv->lower_case_csuffix, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
}


static char* vala_enum_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp1_;
	char* _tmp0_;
	char* _tmp2_;
	self = (ValaEnum*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = NULL;
	return (_tmp2_ = g_strdup_printf ("%s%s%s", _tmp0_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1_ = vala_enum_get_lower_case_csuffix (self)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
}


static char* vala_enum_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaEnum * self;
	char* _tmp0_;
	char* _tmp1_;
	self = (ValaEnum*) base;
	_tmp0_ = NULL;
	_tmp1_ = NULL;
	return (_tmp1_ = g_utf8_strup (_tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
}


static gboolean vala_enum_real_is_reference_type (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	return FALSE;
}


static char* vala_enum_real_get_cprefix (ValaSymbol* base) {
	ValaEnum * self;
	const char* _tmp2_;
	self = (ValaEnum*) base;
	if (self->priv->cprefix == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		self->priv->cprefix = (_tmp1_ = g_strdup_printf ("%s_", _tmp0_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, NULL)), self->priv->cprefix = (g_free (self->priv->cprefix), NULL), _tmp1_);
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	_tmp2_ = NULL;
	return (_tmp2_ = self->priv->cprefix, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
}


/**
 * Sets the string to be prepended to the name of members of this enum
 * when used in C code.
 *
 * @param cprefix the prefix to be used in C code
 */
void vala_enum_set_cprefix (ValaEnum* self, const char* cprefix) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->cprefix = (_tmp1_ = (_tmp0_ = cprefix, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->cprefix = (g_free (self->priv->cprefix), NULL), _tmp1_);
}


static void vala_enum_process_ccode_attribute (ValaEnum* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp0_;
		_tmp0_ = NULL;
		vala_enum_set_cprefix (self, _tmp0_ = vala_attribute_get_string (a, "cprefix"));
		_tmp0_ = (g_free (_tmp0_), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp1_;
		_tmp1_ = NULL;
		self->priv->lower_case_csuffix = (_tmp1_ = vala_attribute_get_string (a, "lower_case_csuffix"), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp1_);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp2_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp2_ = NULL;
			filename_collection = _tmp2_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp2_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp2_); filename_it = filename_it + 1) {
				const char* _tmp3_;
				char* filename;
				_tmp3_ = NULL;
				filename = (_tmp3_ = filename_collection[filename_it], (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "has_type_id")) {
		vala_enum_set_has_type_id (self, vala_attribute_get_bool (a, "has_type_id"));
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp4_;
		_tmp4_ = NULL;
		self->priv->type_id = (_tmp4_ = vala_attribute_get_string (a, "type_id"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp4_);
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_process_attributes (ValaEnum* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0_;
			ValaAttribute* a;
			_tmp0_ = NULL;
			a = (_tmp0_ = (ValaAttribute*) a_it->data, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_enum_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "Flags") == 0) {
						vala_enum_set_is_flags (self, TRUE);
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


void vala_enum_set_type_id (ValaEnum* self, const char* type_id) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->type_id = (_tmp1_ = (_tmp0_ = type_id, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp1_);
}


static char* vala_enum_real_get_type_id (ValaTypeSymbol* base) {
	ValaEnum * self;
	const char* _tmp4_;
	self = (ValaEnum*) base;
	if (self->priv->type_id == NULL) {
		if (self->priv->_has_type_id) {
			char* _tmp0_;
			_tmp0_ = NULL;
			self->priv->type_id = (_tmp0_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, "TYPE_"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp0_);
		} else {
			const char* _tmp1_;
			char* _tmp3_;
			const char* _tmp2_;
			_tmp1_ = NULL;
			if (self->priv->_is_flags) {
				_tmp1_ = "G_TYPE_UINT";
			} else {
				_tmp1_ = "G_TYPE_INT";
			}
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			self->priv->type_id = (_tmp3_ = (_tmp2_ = _tmp1_, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp3_);
		}
	}
	_tmp4_ = NULL;
	return (_tmp4_ = self->priv->type_id, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_));
}


static char* vala_enum_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("FLAGS");
		} else {
			return g_strdup ("ENUM");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("UINT");
		} else {
			return g_strdup ("INT");
		}
	}
}


static char* vala_enum_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_get_flags");
		} else {
			return g_strdup ("g_value_get_enum");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_get_uint");
		} else {
			return g_strdup ("g_value_get_int");
		}
	}
}


static char* vala_enum_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_has_type_id) {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_set_flags");
		} else {
			return g_strdup ("g_value_set_enum");
		}
	} else {
		if (self->priv->_is_flags) {
			return g_strdup ("g_value_set_uint");
		} else {
			return g_strdup ("g_value_set_int");
		}
	}
}


static char* vala_enum_real_get_default_value (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	return g_strdup ("0");
}


static char* vala_enum_real_get_type_signature (ValaTypeSymbol* base) {
	ValaEnum * self;
	self = (ValaEnum*) base;
	if (self->priv->_is_flags) {
		return g_strdup ("u");
	} else {
		return g_strdup ("i");
	}
}


static gboolean vala_enum_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaEnum * self;
	ValaSourceFile* _tmp1_;
	ValaSourceFile* old_source_file;
	gboolean _tmp2_;
	self = (ValaEnum*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_enum_process_attributes (self);
	_tmp1_ = NULL;
	old_source_file = (_tmp1_ = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1_ == NULL) ? NULL : vala_source_file_ref (_tmp1_));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	{
		GeeIterator* _value_it;
		_value_it = gee_iterable_iterator ((GeeIterable*) self->priv->values);
		while (gee_iterator_next (_value_it)) {
			ValaEnumValue* value;
			value = (ValaEnumValue*) gee_iterator_get (_value_it);
			vala_code_node_check ((ValaCodeNode*) value, analyzer);
			(value == NULL) ? NULL : (value = (vala_code_node_unref (value), NULL));
		}
		(_value_it == NULL) ? NULL : (_value_it = (gee_collection_object_unref (_value_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	return (_tmp2_ = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), _tmp2_);
}


gboolean vala_enum_get_is_flags (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_flags;
}


void vala_enum_set_is_flags (ValaEnum* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_flags = value;
}


gboolean vala_enum_get_has_type_id (ValaEnum* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_type_id;
}


void vala_enum_set_has_type_id (ValaEnum* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_type_id = value;
}


static void vala_enum_class_init (ValaEnumClass * klass) {
	vala_enum_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_enum_finalize;
	g_type_class_add_private (klass, sizeof (ValaEnumPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_enum_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_enum_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_enum_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_enum_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_enum_real_is_reference_type;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_enum_real_get_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_enum_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_enum_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_enum_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_enum_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_default_value = vala_enum_real_get_default_value;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_signature = vala_enum_real_get_type_signature;
	VALA_CODE_NODE_CLASS (klass)->check = vala_enum_real_check;
}


static void vala_enum_instance_init (ValaEnum * self) {
	self->priv = VALA_ENUM_GET_PRIVATE (self);
	self->priv->_has_type_id = TRUE;
	self->priv->values = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM_VALUE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_enum_finalize (ValaCodeNode* obj) {
	ValaEnum * self;
	self = VALA_ENUM (obj);
	(self->priv->values == NULL) ? NULL : (self->priv->values = (gee_collection_object_unref (self->priv->values), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->cprefix = (g_free (self->priv->cprefix), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	VALA_CODE_NODE_CLASS (vala_enum_parent_class)->finalize (obj);
}


GType vala_enum_get_type (void) {
	static GType vala_enum_type_id = 0;
	if (vala_enum_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnum), 0, (GInstanceInitFunc) vala_enum_instance_init, NULL };
		vala_enum_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaEnum", &g_define_type_info, 0);
	}
	return vala_enum_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




