/* valaccodefunctioncall.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_EXPRESSION (vala_ccode_expression_get_type ())
#define VALA_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpression))
#define VALA_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))
#define VALA_IS_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_IS_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_CCODE_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))

typedef struct _ValaCCodeExpression ValaCCodeExpression;
typedef struct _ValaCCodeExpressionClass ValaCCodeExpressionClass;
typedef struct _ValaCCodeExpressionPrivate ValaCCodeExpressionPrivate;

#define VALA_TYPE_CCODE_FUNCTION_CALL (vala_ccode_function_call_get_type ())
#define VALA_CCODE_FUNCTION_CALL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_FUNCTION_CALL, ValaCCodeFunctionCall))
#define VALA_CCODE_FUNCTION_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_FUNCTION_CALL, ValaCCodeFunctionCallClass))
#define VALA_IS_CCODE_FUNCTION_CALL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_FUNCTION_CALL))
#define VALA_IS_CCODE_FUNCTION_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_FUNCTION_CALL))
#define VALA_CCODE_FUNCTION_CALL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_FUNCTION_CALL, ValaCCodeFunctionCallClass))

typedef struct _ValaCCodeFunctionCall ValaCCodeFunctionCall;
typedef struct _ValaCCodeFunctionCallClass ValaCCodeFunctionCallClass;
typedef struct _ValaCCodeFunctionCallPrivate ValaCCodeFunctionCallPrivate;

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents an expression node in the C code tree.
 */
struct _ValaCCodeExpression {
	ValaCCodeNode parent_instance;
	ValaCCodeExpressionPrivate * priv;
};

struct _ValaCCodeExpressionClass {
	ValaCCodeNodeClass parent_class;
	void (*write_inner) (ValaCCodeExpression* self, ValaCCodeWriter* writer);
};

/**
 * Represents a function call in the C code.
 */
struct _ValaCCodeFunctionCall {
	ValaCCodeExpression parent_instance;
	ValaCCodeFunctionCallPrivate * priv;
};

struct _ValaCCodeFunctionCallClass {
	ValaCCodeExpressionClass parent_class;
};

struct _ValaCCodeFunctionCallPrivate {
	ValaCCodeExpression* _call;
	GeeList* arguments;
};



gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_expression_get_type (void);
GType vala_ccode_function_call_get_type (void);
#define VALA_CCODE_FUNCTION_CALL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_FUNCTION_CALL, ValaCCodeFunctionCallPrivate))
enum  {
	VALA_CCODE_FUNCTION_CALL_DUMMY_PROPERTY
};
void vala_ccode_function_call_set_call (ValaCCodeFunctionCall* self, ValaCCodeExpression* value);
ValaCCodeFunctionCall* vala_ccode_function_call_new (ValaCCodeExpression* call);
ValaCCodeFunctionCall* vala_ccode_function_call_construct (GType object_type, ValaCCodeExpression* call);
ValaCCodeFunctionCall* vala_ccode_function_call_new (ValaCCodeExpression* call);
void vala_ccode_function_call_add_argument (ValaCCodeFunctionCall* self, ValaCCodeExpression* expr);
void vala_ccode_function_call_insert_argument (ValaCCodeFunctionCall* self, gint index, ValaCCodeExpression* expr);
GeeList* vala_ccode_function_call_get_arguments (ValaCCodeFunctionCall* self);
ValaCCodeExpression* vala_ccode_function_call_get_call (ValaCCodeFunctionCall* self);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
static void vala_ccode_function_call_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_function_call_parent_class = NULL;
static void vala_ccode_function_call_finalize (ValaCCodeNode* obj);



ValaCCodeFunctionCall* vala_ccode_function_call_construct (GType object_type, ValaCCodeExpression* call) {
	ValaCCodeFunctionCall* self;
	self = (ValaCCodeFunctionCall*) g_type_create_instance (object_type);
	vala_ccode_function_call_set_call (self, call);
	return self;
}


ValaCCodeFunctionCall* vala_ccode_function_call_new (ValaCCodeExpression* call) {
	return vala_ccode_function_call_construct (VALA_TYPE_CCODE_FUNCTION_CALL, call);
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param expr a C code expression
 */
void vala_ccode_function_call_add_argument (ValaCCodeFunctionCall* self, ValaCCodeExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->arguments, expr);
}


void vala_ccode_function_call_insert_argument (ValaCCodeFunctionCall* self, gint index, ValaCCodeExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_list_insert (self->priv->arguments, index, expr);
}


/**
 * Returns a copy of the list of arguments.
 *
 * @return list of arguments
 */
GeeList* vala_ccode_function_call_get_arguments (ValaCCodeFunctionCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, self->priv->arguments);
}


static void vala_ccode_function_call_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeFunctionCall * self;
	gboolean first;
	self = (ValaCCodeFunctionCall*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_node_write ((ValaCCodeNode*) self->priv->_call, writer);
	vala_ccode_writer_write_string (writer, " (");
	first = TRUE;
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->arguments);
		while (gee_iterator_next (_expr_it)) {
			ValaCCodeExpression* expr;
			expr = (ValaCCodeExpression*) gee_iterator_get (_expr_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			if (expr != NULL) {
				vala_ccode_node_write ((ValaCCodeNode*) expr, writer);
			}
			(expr == NULL) ? NULL : (expr = (vala_ccode_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
	vala_ccode_writer_write_string (writer, ")");
}


ValaCCodeExpression* vala_ccode_function_call_get_call (ValaCCodeFunctionCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_call;
}


void vala_ccode_function_call_set_call (ValaCCodeFunctionCall* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2_;
	ValaCCodeExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_call = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_call == NULL) ? NULL : (self->priv->_call = (vala_ccode_node_unref (self->priv->_call), NULL)), _tmp2_);
}


static void vala_ccode_function_call_class_init (ValaCCodeFunctionCallClass * klass) {
	vala_ccode_function_call_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_function_call_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeFunctionCallPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_function_call_real_write;
}


static void vala_ccode_function_call_instance_init (ValaCCodeFunctionCall * self) {
	self->priv = VALA_CCODE_FUNCTION_CALL_GET_PRIVATE (self);
	self->priv->arguments = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_function_call_finalize (ValaCCodeNode* obj) {
	ValaCCodeFunctionCall * self;
	self = VALA_CCODE_FUNCTION_CALL (obj);
	(self->priv->_call == NULL) ? NULL : (self->priv->_call = (vala_ccode_node_unref (self->priv->_call), NULL));
	(self->priv->arguments == NULL) ? NULL : (self->priv->arguments = (gee_collection_object_unref (self->priv->arguments), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_function_call_parent_class)->finalize (obj);
}


GType vala_ccode_function_call_get_type (void) {
	static GType vala_ccode_function_call_type_id = 0;
	if (vala_ccode_function_call_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeFunctionCallClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_function_call_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeFunctionCall), 0, (GInstanceInitFunc) vala_ccode_function_call_instance_init, NULL };
		vala_ccode_function_call_type_id = g_type_register_static (VALA_TYPE_CCODE_EXPRESSION, "ValaCCodeFunctionCall", &g_define_type_info, 0);
	}
	return vala_ccode_function_call_type_id;
}




