/* 
Copyright 2008 by Danny Milosavjevic <danny.milo@scratchpost.org>.
All rights reserved.
   
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
   
1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
            
This software is provided by the copyright holders and contributors "as is" and any express of implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed.
In no event shall the copyright owner or contributors be liable for any direct, indirect, incidental, special, exemplary or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
*/
/* TODO use unzip library */
/* TODO make perror() actually include the file name. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>

#include <libosso.h>
#include <hildon/hildon-program.h>

#ifndef PREFIX
#define PREFIX "/usr/" /* sigh... */
#endif

static osso_context_t* osso_context;
static gboolean cmdline = TRUE;

static void error (const char *format, ...)
{
	va_list arg;
	char text [200];

	va_start (arg, format);
	if (cmdline)
	{
		fprintf (stderr, "error: ");
		vfprintf (stderr, format, arg);
		fprintf (stderr, "\n");
	}
	else
		vsnprintf (text, sizeof (text), format, arg);
	va_end (arg);

	if (!cmdline)
		osso_system_note_dialog (osso_context, text, OSSO_GN_ERROR, NULL);
}

void perror (const char *syscall)
{
	error ("%s failed: %s", syscall, strerror (errno));
}

/** wait_until_child_terminated: waits until a process terminated.
  *  returns:
  *    -1   on error
  *     0   terminated normally and with exit code 0
  *    >0   terminated "normally" with that exit code
  *    
  * Side effects:
  *   - prints to stderr
  */
static int wait_until_child_terminated(const char* zip_file_name, pid_t child_pid)
{
	int status;
	int waitpid_result;
	do {
		waitpid_result = waitpid(child_pid, &status, 0);
		if (waitpid_result == -1) {
			break;
		}
		
		/* WIFSTOPPED WIFCONTINUED */
	} while (!WIFEXITED(status) && !WIFSIGNALED(status));
	
	if (waitpid_result == -1) { /* error */
		perror("waitpid");
		return -1;
	}
	
	if (WIFEXITED(status)) {
		if (WEXITSTATUS(status) != 0) {
			error ("could not unzip \"%s\"", zip_file_name);
			return WEXITSTATUS(status);
		}
	} else if (WIFSIGNALED(status)) {
		error("could not unzip \"%s\": unzip received signal %d", zip_file_name, WTERMSIG(status));
		return -1;
	}
	
	/* here, unzip finished successfully */
	return 0;
}

/**
  * Side effects:
  *   - prints to stderr
  */
static char** get_zip_content_file_names(const char* zip_file_name)
{
	const char* argv[] = {
		PREFIX "bin/unzip",
		"-Z",
		"-1",
		"--",
		zip_file_name,
		NULL
	};
	gint exit_code;
	gchar* standard_output;
	gchar** file_names;

	standard_output = NULL;
	
	
	if (!g_spawn_sync(NULL, (char**) argv, NULL, G_SPAWN_STDERR_TO_DEV_NULL, NULL, NULL, &standard_output, NULL, &exit_code, NULL)) {
		/* unzip printed reason to stderr */
		error("could not read zip file \"%s\"", zip_file_name);
		return NULL;
	}

	if (standard_output == NULL || 	exit_code != 0) {
		g_free(standard_output);
		error ("could not read zip file \"%s\"", zip_file_name);
		return NULL;
	}

	/* here, unzip (-l) finished successfully */

	assert(standard_output != NULL);
	
	file_names = g_strsplit(standard_output, "\n", 0);
	assert(file_names != NULL);
	
	g_free(standard_output);

	return file_names;
}

/** prepare_unzip: prepares to unzip (maybe create subfolder, change there, ...)
 * returns:
 *     NULL        error
 *     otherwise   the (new) directory path where it will be extracted to
 *
 * Side effects:
 *   - creates directory
 *   - changes current directory
 *   - prints to stderr
 */
static char* prepare_unzip(const char* zip_file_name)
{
	const char* file_name;
	char* suggested_folder_name;
	char* suggested_folder_name_and_slash;
	int i;
	gboolean contains_only_obvious_subfolder;
	char* zip_base_name;
	char** file_names;
	char* result;
	char* source_directory;
	
	file_names = get_zip_content_file_names(zip_file_name);
	if (file_names == NULL) {
		return NULL;
	}
		
	zip_base_name = g_path_get_basename(zip_file_name);
	assert(zip_base_name != NULL);

	if (g_str_has_suffix(zip_base_name, ".zip")) {
		suggested_folder_name = g_strdup(zip_base_name);
		suggested_folder_name[strlen(suggested_folder_name) - 4] = 0;
	} else {
		suggested_folder_name = g_strdup_printf("%s.d", zip_base_name);
	}

	g_free(zip_base_name);
	zip_base_name = NULL;
	
	suggested_folder_name_and_slash = g_strdup_printf("%s/", suggested_folder_name);

	/* check whether the zip file only contains files in subfolders */
	
	contains_only_obvious_subfolder = TRUE;
	
	for (i = 0; file_names[i] != NULL; ++i) {
		file_name = file_names[i];
		
		if (!g_str_has_prefix(file_name, suggested_folder_name_and_slash) && !g_str_equal(file_name, suggested_folder_name)) {
			contains_only_obvious_subfolder = FALSE;
			break;
		}
	}
	
	g_strfreev(file_names);
	
	/*previous_directory = g_get_current_dir();*/
	
	source_directory = g_path_get_dirname(zip_file_name);

	result = g_build_filename(source_directory, suggested_folder_name, NULL);
	g_free(suggested_folder_name);
	suggested_folder_name = NULL;
	
	if (!contains_only_obvious_subfolder) {
		/* since it is a messy file, specially create a confinement directory for it */
		
		if (mkdir(result, 0777) == -1) {
			g_free(result);
			g_free(source_directory);
			perror("mkdir");
			return NULL;
		}
		
	} else {
		/* it is an orderly file, so just check whether folder doesn't exist */
		
		if (mkdir(result, 0777) == -1) { /* oops. */
			g_free(result);
			g_free(source_directory);
			perror("mkdir");
			return NULL;
		}
		
		/* orderly file, so return source directory as extraction dir */
		
		g_free(result);
		result = g_strdup(source_directory);
	}
	
	g_free(source_directory);

	if (chdir(result) == -1) {
		g_free(result);
		perror("chdir");
		return NULL;
	}
	
	return result;
}

/* Cleans up a bit */
static void undo_unzip(const char* zip_extraction_directory_name)
{
	const char* argv[] = {
		"/bin/rm",
		"-rf",
		"--",
		zip_extraction_directory_name,
		NULL
	};

	gint exit_code;
	
	if (zip_extraction_directory_name[0] == '/' && zip_extraction_directory_name[1] == 0)
	{
		error("tried to remove the root directory \"/\". This is a serious program bug. Please report to <danny.milo@scratchpost.org>");
		return; /* chicken */
	}
	
	if (!g_spawn_sync(NULL, (char**) argv, NULL, 0, NULL, NULL, NULL, NULL, &exit_code, NULL) || exit_code != 0) {
		/* unzip printed reason to stderr */
		error("could not delete directory \"%s\"", zip_extraction_directory_name);
		/* ignore error... */
	}
	
	/*
	if (remove(zip_extraction_directory_name) == -1) {
		perror("remove");
		ignore error...
	}*/
}

/** unzip_file: unzips a file
 * Preconditions:
 *   - prepare_unzip()
 * Side effects:
 *   - prints to stderr
 */
static int unzip_file(const char* zip_file_name, const char* zip_extraction_directory_name)
{
	const char* argv[] = {
		PREFIX "bin/unzip",
		"-o",
		"-q",
		"-d",
		zip_extraction_directory_name,
		"--",
		zip_file_name,
		NULL
	};
	GPid child_pid;
	char text[2049];
	
	/* Display a HildonBanner to the user */
	if (snprintf(text, sizeof(text), "Unzipping \"%s\"", g_basename (zip_file_name)) != -1) {
		gdk_notify_startup_complete();
		osso_system_note_infoprint(osso_context, text, NULL);
	}
	
	if (!g_spawn_async(NULL, (char**) argv, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &child_pid, NULL)) {
		/* unzip printed reason to stderr */
		error ("could not unzip \"%s\"", zip_file_name);
		return -1;
	}

	if (wait_until_child_terminated(zip_file_name, child_pid) != 0) { /* bad */
		return -1;
	}
	
	/* here, unzip finished successfully */
	return 0;
}

/** unzip_frontend: the main unzip entry point
 * Side effects:
 *   - prints to stderr
 * Limitations:
 *   - only works with absolute paths most of the time
 */
static int unzip_frontend(const char* zip_file_name)
{
	char* previous_directory;
	char* source_directory;
	char* destination_new_directory;
	int   status = 0;
	
	previous_directory = g_get_current_dir();
	
	source_directory = g_path_get_dirname(zip_file_name);
	assert(source_directory != NULL);
	
	if (source_directory[0] != 0) {
		if (chdir(source_directory) == -1) {
			perror("chdir");
			return -1; /* whoops. this. is. bad. */
		}
	}
	
	g_free(source_directory);
	
	destination_new_directory = prepare_unzip(zip_file_name);
	if (destination_new_directory != NULL) {
		if (unzip_file(zip_file_name, destination_new_directory) != 0)
			status = -1;
	} else {
		status = -1;
	}
	
	
	if (chdir(previous_directory) == -1) {
		g_free(previous_directory);
		perror("chdir");
		chdir("/"); /* whoops. */
		return -1;
	}
	g_free(previous_directory);


	if (status != 0) {	
		if (destination_new_directory != NULL) {
			undo_unzip(destination_new_directory);
		}
	}

	g_free(destination_new_directory);
	destination_new_directory = NULL;


	return status;
}


static void document_open_callback(gpointer data, int argc, char* argv[]) 
{ 
	char* document_path; 
	char const* document_url; 
	int i;
	
	for (i = 0; i < argc; ++i) {
		document_url = argv[i]; 
	
		if (document_url == NULL) { 
			g_warning("internal error: document_url[%d] = NULL", i);
			break;
		} 

		document_path = g_filename_from_uri(document_url, NULL /*&document_host*/, NULL); 
		if (document_path == NULL) { 
			g_warning("could not find path to document. Maybe the document is not local?"); 
			continue;
		} 

		if (!g_file_test(document_path, G_FILE_TEST_IS_REGULAR)) {
			continue;
		}

		unzip_frontend(document_path);

		g_free(document_path);
	}
	
	exit(0); /* Commit suicide so that we don't consume RAM doing nothing.
	            DBus should be launching us again when we are needed. */
}

static char* expand_name(const char* name)
{
	char* result;
	char* current_directory;
	
	if (name[0] == '/') {
		return g_strdup(name);
	} else {
		current_directory = g_get_current_dir();
		assert(current_directory != NULL);
		
		result = g_strdup_printf("%s/%s", current_directory, name);
		g_free(current_directory);
		return result;
	}
}


int main(int argc, char* argv[])
{
	const char** argp;
	char* absolute_path;
	
	gtk_init(&argc, &argv);
	
	osso_context = osso_initialize("unzip_fm", "1.0", FALSE, NULL);
	if (osso_context == NULL) {
		g_error("could not register document handler service");
		exit(1);
	}

	/* if we got a parameter, that probably is a file to be batch-unzipped. */
	if (argc > 1) {
		/* skip options until the end-of-option marker.
		   For example:
		     ./batch-unzip -f -b -k -- hello.zip
		   Moves argp over there:
					       ^^^^^^^^^
		   For example:
		     ./batch-unzip -f -b -k -- --hello.zip
		   Moves argp over there:
					       ^^^^^^^^^^^
		 */
		argp = (const char**) &argv[1];
		while (*argp && (*argp)[0] == '-') {
			if ((*argp)[0] == '-' && (*argp)[1] == '-' && (*argp)[2] == 0) {
				++argp;
				break;
			}
			++argp;
		}
		
		if (*argp) {
			absolute_path = expand_name(*argp);
			if (unzip_frontend(absolute_path) != 0) {
				g_free(absolute_path);
				return 1;
			}
			g_free(absolute_path);
		}
	} else {
		/* process dbus stuff */
		cmdline = FALSE;
		
		osso_mime_set_cb(osso_context, document_open_callback, NULL);

		/* hang around until a request comes....
		   Note that this could just make us hang around doing nothing,
		   should we have been started from the application menu */
		gtk_main();
	}
	
	return 0;
}
