#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QMessageBox>
#include <QProcess>
#include <QTimer>
#include <QDebug>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    plugin(PLUGINPATH)
{
    qDebug() << "TweakFlashVer v0.3";

    settings = new QSettings("tweakflashver", "main");
    ui->setupUi(this);

    qDebug() << "Loading... ";
    loadFlashVers();
    qDebug() << "Loaded";
    if (ui->comboBox->count() == 0) // no version loaded from config
    {
        qDebug() << "Initializing settings";
        settings->beginGroup("global");
        settings->setValue("pluginpath", PLUGINPATH);
        settings->setValue("pluginsize", PLUGINSIZE);
        settings->endGroup();
#ifdef Q_WS_MAEMO5
        settings->beginGroup("Android, 10.1.61 (FroYo)");
        settings->setValue("part1","10.1 r61");
        settings->setValue("part2","AND 10,1,61,0");
        settings->setValue("index", 4);
        settings->endGroup();
        settings->beginGroup("Linux, 10.1.53 (Desktop)");
        settings->setValue("part1","10.1 r53");
        settings->setValue("part2","LNX 10,1,53,0");
        settings->setValue("index", 3);
        settings->endGroup();
        settings->beginGroup("Linux, 10.0.45 (Desktop) - recommended");
        settings->setValue("part1","10.0 r45");
        settings->setValue("part2","LNX 10,0,45,0");
        settings->setValue("index", 2);
        settings->endGroup();
        settings->beginGroup("Linux, 9.0.246 (PR1.1)");
        settings->setValue("part1","9.0 r246");
        settings->setValue("part2","LNX 9,0,246,0");
        settings->setValue("index", 1);
        settings->endGroup();
        settings->beginGroup("Linux, 9.0.260 (PR1.2)");
        settings->setValue("part1","9.0 r260");
        settings->setValue("part2","LNX 9,0,260,0");
        settings->setValue("index", 0);
        settings->endGroup();
#else
        settings->beginGroup("Android, 10.1.9");
        settings->setValue("part1","10.1 r9");
        settings->setValue("part2","AND 10,1,9,0");
        settings->setValue("index", 4);
        settings->endGroup();
        settings->beginGroup("Linux, 10.1.9 (Desktop)");
        settings->setValue("part1","10.1 r9");
        settings->setValue("part2","LNX 10,1,9,0");
        settings->setValue("index", 3);
        settings->endGroup();
        settings->beginGroup("Linux, 10.0.9 (Desktop) - recommended");
        settings->setValue("part1","10.0 r9");
        settings->setValue("part2","LNX 10,0,9,0");
        settings->setValue("index", 2);
        settings->endGroup();
        settings->beginGroup("Linux, 9.0.99 (Desktop)");
        settings->setValue("part1","9.0 r99");
        settings->setValue("part2","LNX 9,0,99,0");
        settings->setValue("index", 1);
        settings->endGroup();
        settings->beginGroup("Linux, 9.0.48 (N810 Diablo)");
        settings->setValue("part1","9.0 r48");
        settings->setValue("part2","LNX 9,0,48,0");
        settings->setValue("index", 0);
        settings->endGroup();
#endif
        settings->sync();
        qDebug() << "Default settings saved";
        loadFlashVers();
        qDebug() << "Settings re-loaded";
    }

    qDebug() << "Checking plugin file";
    settings->beginGroup("global");
    if (!QFile::exists(settings->value("pluginpath", PLUGINPATH).toString()))
    {
        QMessageBox::critical(this, tr("Error"), tr("Flash plugin file not found"));
        qApp->quit();
    }
    else if (plugin.size() != settings->value("pluginsize", PLUGINSIZE).toUInt())
    {
        QMessageBox::critical(this, tr("Error"), tr("Flash plugin file size mismatch"));
        qApp->quit();
    }
    settings->endGroup();
    qDebug() << "Checks out OK";

    connect(ui->pushButton_2, SIGNAL(clicked()), this, SLOT(changeVer()));
    connect(ui->checkBox, SIGNAL(clicked(bool)), this, SLOT(setManualMode(bool)));
    connect(ui->comboBox, SIGNAL(currentIndexChanged(QString)), this, SLOT(updateFields()));

    readFlashVer();
    setManualMode(ui->checkBox->isChecked());
}

void MainWindow::readFlashVer()
{
    if ( plugin.open(QIODevice::ReadOnly) && plugin.seek(OFFSET2)  )
    {
        ui->lineEdit->setText(plugin.read(PART2LEN));
    }
    plugin.close();
}


void MainWindow::setManualMode(bool b)
{
    ui->comboBox->setEnabled(!b);
    ui->lineEdit_2->setEnabled(b);
    ui->lineEdit_3->setEnabled(b);
    ui->lineEdit_4->setEnabled(b);
    if (!b)
        updateFields();
}

void MainWindow::updateFields()
{
    if (ui->comboBox->count() == 0)
        return;

    QString str = settings->value(ui->comboBox->currentText()+"/part2", "UNK   0,0,0,0").toString().right(PART2LEN-4); // -4 skips the platform part
    QStringList fields = str.split(",");
    ui->lineEdit_2->setText(fields[0]);
    ui->lineEdit_3->setText(fields[1]);
    ui->lineEdit_4->setText(fields[2]);
}

void MainWindow::changeVer()
{
    if (!QFile::exists(PLUGINBACKPUPPATH)) // create backup
        if (!QFile::copy(PLUGINPATH, PLUGINBACKPUPPATH))
        {
            QMessageBox::critical(this, tr("Error"), tr("Unable to create backup !"));
            return;
        }
    QString part1 = settings->value(ui->comboBox->currentText()+"/part1").toString();
    QString part2 = settings->value(ui->comboBox->currentText()+"/part2").toString();

    if (ui->checkBox->isChecked()) // manual
    {
        if (ui->lineEdit_2->text().length() + ui->lineEdit_3->text().length() + ui->lineEdit_4->text().length() != DIGITS ||
            ui->lineEdit_2->text().isEmpty() ||
            ui->lineEdit_3->text().isEmpty() ||
            ui->lineEdit_4->text().isEmpty() )
        {
            QMessageBox::warning(this, tr("Incorrect version string"), tr("The version string MUST contain %0 numbers in total, and every field has to have at least a single digit. That's the way it is. Sorry.").arg(DIGITS));
            return;
        }
        part1 = QString("%0.%1 r%2").arg(ui->lineEdit_2->text()).arg(ui->lineEdit_3->text()).arg(ui->lineEdit_4->text());
        part2 = QString("LNX %0,%1,%2,0").arg(ui->lineEdit_2->text()).arg(ui->lineEdit_3->text()).arg(ui->lineEdit_4->text());
    }

    if (part1.length() != PART1LEN || part2.length() != PART2LEN)
    {
        QMessageBox::critical(this, tr("Error"), tr("Version string length wrong, would break binary !"));
        return;
    }
    if (
        plugin.open(QIODevice::ReadWrite) &&
        plugin.seek(OFFSET1) &&
        plugin.write(part1.toAscii()) &&
        plugin.seek(OFFSET2) &&
        plugin.write(part2.toAscii()))
    {
        QMessageBox::information(this, tr("Success"), tr("New version info successfully written, restarting browser daemon."));
        QStringList param;
#ifdef Q_WS_MAEMO_5
        param << "tablet-browser-daemon";
        QProcess::execute("/usr/sbin/stop", param);
        QProcess::execute("/usr/sbin/start", param);
#else
        param << "restart";
        QProcess::execute("/etc/init.d/tablet-browser-daemon", param);
#endif
    }
    else
    {
        QMessageBox::critical(this, tr("Error"), tr("Uh-oh. Error while opening/writing Flash plugin binary. I hope I didn't break it. Fear not, even if I did, I made a backup to %0, just copy it back to %1 as root").arg(PLUGINBACKPUPPATH).arg(PLUGINPATH));
    }

    plugin.close();
    readFlashVer();
}


void MainWindow::enableAutoRotate()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

}

void MainWindow::loadFlashVers()
{
    if (settings) {
        foreach (QString group, settings->childGroups())
        {
            if (group != "global")
                ui->comboBox->addItem(group);
        }
        updateFields();
    }
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::on_actionAbout_Qt_triggered()
{
    qApp->aboutQt();
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, tr("Tweak Flash Plugin Version"), tr("This applications allows you to painlessly change the version of the Flash plugin reported to sites from a GUI. This will, however, NOT make Flash faster or sites genuinely requiring Flash 10 work. It is not a Flash 10.1 upgrade replacement. What it does is getting you around pesky Flash version checks. Enjoy. <BR><BR>Copyright (C) 2010 by Attila Csipa, licensed under the GNU GPL v2"));
}

void MainWindow::on_actionRestore_plugin_to_factory_state_triggered()
{
    QString ppath;
    if (QFile::exists(PLUGINBACKPUPPATH))
        ppath = PLUGINBACKPUPPATH;
    else if (QFile::exists(PLUGINBACKPUPPATH_LEGACY))
        ppath = PLUGINBACKPUPPATH_LEGACY;
    else {
            QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to find backup ! If you have removed it, please restore the file to %0").arg( PLUGINBACKPUPPATH));
            return;
    }

    if (!QFile(ppath).exists() || QFile(ppath).size() != PLUGINSIZE)
    {
            QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to find backup ! If you have removed it, please restore the file to %0").arg( PLUGINBACKPUPPATH));
            return;
    }

    if (!(QFile::remove(PLUGINPATH) && QFile::copy(ppath, PLUGINPATH)))
    {
        QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to restore backup !"));
        return;
    }
    readFlashVer();
    QMessageBox::information(this, tr("Success"), tr("Flash plugin restored to original state"));

}
