/*
 * Tomiku
 *
 * Project homepage: http://tomiku.garage.maemo.org
 * Author homepage: http://www.aivokyyti.net
 *
 * Copyright (C) 2007 Tomi Pihlainen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA. 
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <math.h>

#include <gtk/gtkmain.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef CHINOOK
#include <hildon/hildon-program.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-help.h>
#else
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-banner.h>
#include <osso-helplib.h>
#endif

#include "main_window.h"
#include "appdata.h"
#include "board.h"
#include "game.h"
#include "layout_theme1.h"

#include <libintl.h>
#include <locale.h>
#include <libosso.h>

#define _(String) gettext(String)
#define HELP_TOPIC_ABOUT "Example_tomiku_Content"
#define GAME_DURATION_TIMER 500

#include <config.h>

void free_pixmaps(MainWindow* main_window)
{
    g_object_unref(main_window->background_pixbuf);
    g_object_unref(main_window->cell_predefined_pixbuf);
    g_object_unref(main_window->cell_correct_pixbuf[0]);
    g_object_unref(main_window->cell_correct_pixbuf[1]);
    g_object_unref(main_window->cell_correct_pixbuf[2]);
    g_object_unref(main_window->cell_correct_pixbuf[3]);
    g_object_unref(main_window->cell_correct_pixbuf[4]);
    g_object_unref(main_window->cell_correct_pixbuf[5]);
    g_object_unref(main_window->cell_correct_pixbuf[6]);
    g_object_unref(main_window->cell_correct_pixbuf[7]);
    g_object_unref(main_window->cell_correct_pixbuf[8]);
    g_object_unref(main_window->cell_incorrect_pixbuf[0]);
    g_object_unref(main_window->cell_incorrect_pixbuf[1]);
    g_object_unref(main_window->cell_incorrect_pixbuf[2]);
    g_object_unref(main_window->cell_incorrect_pixbuf[3]);
    g_object_unref(main_window->cell_incorrect_pixbuf[4]);
    g_object_unref(main_window->cell_incorrect_pixbuf[5]);
    g_object_unref(main_window->cell_incorrect_pixbuf[6]);
    g_object_unref(main_window->cell_incorrect_pixbuf[7]);
    g_object_unref(main_window->cell_incorrect_pixbuf[8]);
    g_object_unref(main_window->subcell_pixbuf[0]);
    g_object_unref(main_window->subcell_pixbuf[1]);
    g_object_unref(main_window->subcell_pixbuf[2]);
    g_object_unref(main_window->subcell_pixbuf[3]);
    g_object_unref(main_window->subcell_pixbuf[4]);
    g_object_unref(main_window->subcell_pixbuf[5]);
    g_object_unref(main_window->subcell_pixbuf[6]);
    g_object_unref(main_window->subcell_pixbuf[7]);
    g_object_unref(main_window->subcell_pixbuf[8]);
    g_object_unref(main_window->subcell_dots_pixbuf);
    g_object_unref(main_window->marker_button_selected);
    g_object_unref(main_window->marker_button_unselected);
    g_object_unref(main_window->submarker_button_selected);
    g_object_unref(main_window->submarker_button_unselected);
}

void load_pixmaps(MainWindow* main_window)
{
    free_pixmaps(main_window);
    
    GError* error = NULL;
    
    if (main_window->full_screen)
    {
        main_window->background_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_BACKGROUND_FULL, &error);
        main_window->cell_predefined_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_CELL_PREDEFINED_FULL, &error);
        main_window->cell_correct_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT1_FULL, &error);
        main_window->cell_correct_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT2_FULL, &error);
        main_window->cell_correct_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT3_FULL, &error);
        main_window->cell_correct_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT4_FULL, &error);
        main_window->cell_correct_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT5_FULL, &error);
        main_window->cell_correct_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT6_FULL, &error);
        main_window->cell_correct_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT7_FULL, &error);
        main_window->cell_correct_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT8_FULL, &error);
        main_window->cell_correct_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT9_FULL, &error);
        main_window->cell_incorrect_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT1_FULL, &error);
        main_window->cell_incorrect_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT2_FULL, &error);
        main_window->cell_incorrect_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT3_FULL, &error);
        main_window->cell_incorrect_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT4_FULL, &error);
        main_window->cell_incorrect_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT5_FULL, &error);
        main_window->cell_incorrect_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT6_FULL, &error);
        main_window->cell_incorrect_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT7_FULL, &error);
        main_window->cell_incorrect_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT8_FULL, &error);
        main_window->cell_incorrect_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT9_FULL, &error);
        main_window->subcell_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL1_FULL, &error);
        main_window->subcell_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL2_FULL, &error);
        main_window->subcell_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL3_FULL, &error);
        main_window->subcell_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL4_FULL, &error);
        main_window->subcell_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL5_FULL, &error);
        main_window->subcell_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL6_FULL, &error);
        main_window->subcell_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL7_FULL, &error);
        main_window->subcell_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL8_FULL, &error);
        main_window->subcell_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL9_FULL, &error);
        main_window->subcell_dots_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL_DOTS_FULL, &error);
        main_window->marker_button_selected = gdk_pixbuf_new_from_file(PIXMAP_MARKER_BUTTON_SELECTED_FULL, &error);
        main_window->marker_button_unselected = gdk_pixbuf_new_from_file(PIXMAP_MARKER_BUTTON_UNSELECTED_FULL, &error);
        main_window->submarker_button_selected = gdk_pixbuf_new_from_file(PIXMAP_SUBMARKER_BUTTON_SELECTED_FULL, &error);
        main_window->submarker_button_unselected = gdk_pixbuf_new_from_file(PIXMAP_SUBMARKER_BUTTON_UNSELECTED_FULL, &error);
    }
    else
    {
        main_window->background_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_BACKGROUND, &error);
        main_window->cell_predefined_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_CELL_PREDEFINED, &error);
        main_window->cell_correct_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT1, &error);
        main_window->cell_correct_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT2, &error);
        main_window->cell_correct_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT3, &error);
        main_window->cell_correct_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT4, &error);
        main_window->cell_correct_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT5, &error);
        main_window->cell_correct_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT6, &error);
        main_window->cell_correct_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT7, &error);
        main_window->cell_correct_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT8, &error);
        main_window->cell_correct_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_CELL_CORRECT9, &error);
        main_window->cell_incorrect_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT1, &error);
        main_window->cell_incorrect_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT2, &error);
        main_window->cell_incorrect_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT3, &error);
        main_window->cell_incorrect_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT4, &error);
        main_window->cell_incorrect_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT5, &error);
        main_window->cell_incorrect_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT6, &error);
        main_window->cell_incorrect_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT7, &error);
        main_window->cell_incorrect_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT8, &error);
        main_window->cell_incorrect_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_CELL_INCORRECT9, &error);
        main_window->subcell_pixbuf[0] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL1, &error);
        main_window->subcell_pixbuf[1] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL2, &error);
        main_window->subcell_pixbuf[2] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL3, &error);
        main_window->subcell_pixbuf[3] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL4, &error);
        main_window->subcell_pixbuf[4] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL5, &error);
        main_window->subcell_pixbuf[5] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL6, &error);
        main_window->subcell_pixbuf[6] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL7, &error);
        main_window->subcell_pixbuf[7] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL8, &error);
        main_window->subcell_pixbuf[8] = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL9, &error);
        main_window->subcell_dots_pixbuf = gdk_pixbuf_new_from_file(PIXMAP_SUBCELL_DOTS, &error);
        main_window->marker_button_selected = gdk_pixbuf_new_from_file(PIXMAP_MARKER_BUTTON_SELECTED, &error);
        main_window->marker_button_unselected = gdk_pixbuf_new_from_file(PIXMAP_MARKER_BUTTON_UNSELECTED, &error);
        main_window->submarker_button_selected = gdk_pixbuf_new_from_file(PIXMAP_SUBMARKER_BUTTON_SELECTED, &error);
        main_window->submarker_button_unselected = gdk_pixbuf_new_from_file(PIXMAP_SUBMARKER_BUTTON_UNSELECTED, &error);
    }
}

void main_view_redraw(MainWindow* main_window)
{
    gtk_widget_queue_draw(main_window->drawing_area);
}

static gint game_duration_timer(gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_view_redraw(main_window);
    main_window->game_duration_timer = gtk_timeout_add(GAME_DURATION_TIMER, (GtkFunction)game_duration_timer, main_window);
    return FALSE;
}

void toggle_full_screen(MainWindow* main_window)
{
    main_window->full_screen = !main_window->full_screen;
        
    load_pixmaps(main_window);

    if (main_window->full_screen)
    {
        gtk_widget_set_size_request(main_window->drawing_area, MAIN_VIEW_WIDTH_FULL, MAIN_VIEW_HEIGHT_FULL);
        gtk_window_fullscreen(GTK_WINDOW(main_window->appdata->main_window));
    }
    else
    {
        gtk_widget_set_size_request(main_window->drawing_area, MAIN_VIEW_WIDTH, MAIN_VIEW_HEIGHT);
        gtk_window_unfullscreen(GTK_WINDOW(main_window->appdata->main_window));
    }

    main_view_redraw(main_window);
}

void main_view_new_game(MainWindow* main_window)
{
    create_board();
    
    start_new_game();

    gtk_widget_set_sensitive(main_window->item_game_pause, TRUE);
    gtk_widget_set_sensitive(main_window->item_game_continue, FALSE);

    if (main_window->game_duration_timer)
        gtk_timeout_remove(main_window->game_duration_timer);
    main_window->game_duration_timer = gtk_timeout_add(GAME_DURATION_TIMER, (GtkFunction)game_duration_timer, main_window);
    
    main_view_redraw(main_window);
}

void main_view_pause_game(MainWindow* main_window)
{
    pause_game();
    
    gtk_widget_set_sensitive(main_window->item_game_pause, FALSE);
    gtk_widget_set_sensitive(main_window->item_game_continue, TRUE);

    if (main_window->game_duration_timer)
        gtk_timeout_remove(main_window->game_duration_timer);
        
    main_view_redraw(main_window);
}

void main_view_continue_game(MainWindow* main_window)
{
    continue_game();
    
    gtk_widget_set_sensitive(main_window->item_game_pause, TRUE);
    gtk_widget_set_sensitive(main_window->item_game_continue, FALSE);

    if (main_window->game_duration_timer)
        gtk_timeout_remove(main_window->game_duration_timer);
    main_window->game_duration_timer = gtk_timeout_add(GAME_DURATION_TIMER, (GtkFunction)game_duration_timer, main_window);

    main_view_redraw(main_window);
}

void main_view_game_solved(MainWindow* main_window)
{
    end_game();
    
    gtk_widget_set_sensitive(main_window->item_game_pause, FALSE);
    gtk_widget_set_sensitive(main_window->item_game_continue, FALSE);

    if (main_window->game_duration_timer)
        gtk_timeout_remove(main_window->game_duration_timer);

    main_view_redraw(main_window);

    /* Notify user for solving the problem */
    GtkWidget *dialog = NULL;
    dialog = GTK_WIDGET(hildon_note_new_information(
        GTK_WINDOW(main_window->appdata->main_window),
        _("Game solved!")));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}

/* Callback for "Game->Pause" menu entry */
void item_game_pause_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_view_pause_game(main_window);
}

/* Callback for "Game->Continue" menu entry */
void item_game_continue_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_view_continue_game(main_window);
}

/* Callback for "Game->New" menu entry */
void item_game_new_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_view_new_game(main_window);
}

/* Callback for "Show->Full screen" menu entry */
void item_show_full_screen_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    toggle_full_screen(main_window);
}

/* Callback for "Show->Cell dots" menu entry */
void item_show_subcell_dots_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_window->subcell_dots = !main_window->subcell_dots;
    main_view_redraw(main_window);
}

/* Callback for "Show->Incorrect cells" menu entry */
void item_show_incorrect_cells_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
    main_window->show_incorrect_cells = !main_window->show_incorrect_cells;
    main_view_redraw(main_window);
}

/* Callback for "Help" menu entry */
void item_help_cb(GtkAction* action, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;
#ifdef CHINOOK
    hildon_help_show(main_window->appdata->osso_context, HELP_TOPIC_ABOUT, HILDON_HELP_SHOW_DIALOG);
#else
    ossohelp_show(main_window->appdata->osso_context, HELP_TOPIC_ABOUT, OSSO_HELP_SHOW_DIALOG);
#endif
}

/* Callback for "Close" menu entry */
void item_close_cb()
{
    gtk_main_quit();
}

/* Create the menu items needed for the main view */
static void create_menu(MainWindow* main_window, AppData* appdata) 
{
    /* Create needed variables */
    GtkWidget *main_menu;
    GtkWidget *menu_game;
    GtkWidget *item_game;
    GtkWidget *item_game_separator;
    GtkWidget *item_game_new;
    GtkWidget *menu_show;
    GtkWidget *item_show;
    GtkWidget *item_help_separator;
    GtkWidget *item_help;
    GtkWidget *item_close;

    /* Create menus */
    main_menu = gtk_menu_new();
    menu_game = gtk_menu_new();
    menu_show = gtk_menu_new();

    /* Create menu items */
    item_game = gtk_menu_item_new_with_label(_("Game"));
    main_window->item_game_pause = gtk_menu_item_new_with_label(_("Pause"));
    main_window->item_game_continue = gtk_menu_item_new_with_label(_("Continue"));
    item_game_separator = gtk_separator_menu_item_new();
    item_game_new = gtk_menu_item_new_with_label(_("New"));
    item_show = gtk_menu_item_new_with_label(_("Show"));
    main_window->item_show_full_screen = gtk_check_menu_item_new_with_label(_("Full screen"));
    main_window->item_show_subcell_dots = gtk_check_menu_item_new_with_label(_("Helper dots in cells"));
    main_window->item_show_incorrect_cells = gtk_check_menu_item_new_with_label(_("Highlight incorrect cells"));
    item_help_separator = gtk_separator_menu_item_new();
    item_help = gtk_menu_item_new_with_label(_("About"));
    item_close = gtk_menu_item_new_with_label(_("Close"));

    /* Add menu items to right menus */
    gtk_menu_append(main_menu, item_game);
    gtk_menu_append(menu_game, main_window->item_game_pause);
    gtk_menu_append(menu_game, main_window->item_game_continue);
    gtk_menu_append(menu_game, item_game_separator);
    gtk_menu_append(menu_game, item_game_new);
    gtk_menu_append(main_menu, item_show);
    gtk_menu_append(menu_show, main_window->item_show_full_screen);
    gtk_menu_append(menu_show, main_window->item_show_subcell_dots);
    gtk_menu_append(menu_show, main_window->item_show_incorrect_cells);
    gtk_menu_append(main_menu, item_help_separator);
    gtk_menu_append(main_menu, item_help);
    gtk_menu_append(main_menu, item_close);

    /* Set main menu and submenus */
    hildon_window_set_menu(HILDON_WINDOW(appdata->main_window), GTK_MENU(main_menu));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_game), menu_game);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_show), menu_show);

    /* Attach the callback functions to the activate signal */
    g_signal_connect(G_OBJECT(main_window->item_game_pause), "activate", GTK_SIGNAL_FUNC(item_game_pause_cb), main_window);
    g_signal_connect(G_OBJECT(main_window->item_game_continue), "activate", GTK_SIGNAL_FUNC(item_game_continue_cb), main_window);
    g_signal_connect(G_OBJECT(item_game_new), "activate", GTK_SIGNAL_FUNC(item_game_new_cb), main_window);
    g_signal_connect(G_OBJECT(main_window->item_show_full_screen), "activate", GTK_SIGNAL_FUNC(item_show_full_screen_cb), main_window);
    g_signal_connect(G_OBJECT(main_window->item_show_subcell_dots), "activate", GTK_SIGNAL_FUNC(item_show_subcell_dots_cb), main_window);
    g_signal_connect(G_OBJECT(main_window->item_show_incorrect_cells), "activate", GTK_SIGNAL_FUNC(item_show_incorrect_cells_cb), main_window);
    g_signal_connect(G_OBJECT(item_help), "activate", GTK_SIGNAL_FUNC(item_help_cb), main_window);
    g_signal_connect(G_OBJECT(item_close), "activate", GTK_SIGNAL_FUNC(item_close_cb), NULL);

    /* Make all menu widgets visible */
    gtk_widget_show_all(GTK_WIDGET(main_menu));
}

static void configure_layoyt_params(gboolean full_screen,
    int* b_top_left_x, int* b_top_left_y,
    double* c_width, double* c_height,
    int* c_3x3_padding_x, int* c_3x3_padding_y,
    double* b_width_with_padding, double* b_height_with_padding,
    double* c_width_with_padding, double* c_height_with_padding,
    int* marker_button_x, int* marker_button_y,
    int* marker_button_width, int* marker_button_height,
    int* submarker_button_x, int* submarker_button_y,
    int* submarker_button_width, int* submarker_button_height,
    int* game_state_label_x, int* game_state_label_y,
    int* game_state_area_x, int* game_state_area_y,
    int* game_state_area_width, int* game_state_area_height)
{
    if (full_screen)
    {
        *marker_button_x = MARKER_BUTTON_X_FULL;
        *marker_button_y = MARKER_BUTTON_Y_FULL;
        *marker_button_width = MARKER_BUTTON_WIDTH_FULL;
        *marker_button_height = MARKER_BUTTON_HEIGHT_FULL;
        *submarker_button_x = SUBMARKER_BUTTON_X_FULL;
        *submarker_button_y = SUBMARKER_BUTTON_Y_FULL;
        *submarker_button_width = SUBMARKER_BUTTON_WIDTH_FULL;
        *submarker_button_height = SUBMARKER_BUTTON_HEIGHT_FULL;

        *b_top_left_x = BOARD_TOP_LEFT_X_FULL;
        *b_top_left_y = BOARD_TOP_LEFT_Y_FULL;
        *c_width = CELL_WIDTH_FULL;
        *c_height = CELL_WIDTH_FULL; /* no separate define for cell height */
        *c_3x3_padding_x = CELL_3X3_PADDING_FULL;
        *c_3x3_padding_y = CELL_3X3_PADDING_FULL; /* no separate define for y padding */
        *b_width_with_padding = CELL_WIDTH_FULL*BOARD_DIM + ((BOARD_DIM/3-1)*CELL_3X3_PADDING_FULL);
        *b_height_with_padding = CELL_WIDTH_FULL*BOARD_DIM + ((BOARD_DIM/3-1)*CELL_3X3_PADDING_FULL);
        *c_width_with_padding = *b_width_with_padding/BOARD_DIM;
        *c_height_with_padding = *b_height_with_padding/BOARD_DIM;

        *game_state_label_x = GAME_STATE_LABEL_X_FULL;
        *game_state_label_y = GAME_STATE_LABEL_Y_FULL;
        *game_state_area_x = GAME_STATE_AREA_X_FULL;
        *game_state_area_y = GAME_STATE_AREA_Y_FULL;
        *game_state_area_width = GAME_STATE_AREA_WIDTH_FULL;
        *game_state_area_height = GAME_STATE_AREA_HEIGHT_FULL;
    }
    else
    {
        *marker_button_x = MARKER_BUTTON_X;
        *marker_button_y = MARKER_BUTTON_Y;
        *marker_button_width = MARKER_BUTTON_WIDTH;
        *marker_button_height = MARKER_BUTTON_HEIGHT;
        *submarker_button_x = SUBMARKER_BUTTON_X;
        *submarker_button_y = SUBMARKER_BUTTON_Y;
        *submarker_button_width = SUBMARKER_BUTTON_WIDTH;
        *submarker_button_height = SUBMARKER_BUTTON_HEIGHT;

        *b_top_left_x = BOARD_TOP_LEFT_X;
        *b_top_left_y = BOARD_TOP_LEFT_Y;
        *c_width = CELL_WIDTH;
        *c_height = CELL_WIDTH; /* no separate define for cell height */
        *c_3x3_padding_x = CELL_3X3_PADDING;
        *c_3x3_padding_y = CELL_3X3_PADDING; /* no separate define for y padding */
        *b_width_with_padding = CELL_WIDTH*BOARD_DIM + ((BOARD_DIM/3-1)*CELL_3X3_PADDING);
        *b_height_with_padding = CELL_WIDTH*BOARD_DIM + ((BOARD_DIM/3-1)*CELL_3X3_PADDING);
        *c_width_with_padding = *b_width_with_padding/BOARD_DIM;
        *c_height_with_padding = *b_height_with_padding/BOARD_DIM;

        *game_state_label_x = GAME_STATE_LABEL_X;
        *game_state_label_y = GAME_STATE_LABEL_Y;
        *game_state_area_x = GAME_STATE_AREA_X;
        *game_state_area_y = GAME_STATE_AREA_Y;
        *game_state_area_width = GAME_STATE_AREA_WIDTH;
        *game_state_area_height = GAME_STATE_AREA_HEIGHT;
    }
}

static void configure_layoyt_params_number_pad(gboolean full_screen,
    int* np1_x, int* np1_y,
    int* np2_x, int* np2_y,
    int* np3_x, int* np3_y,
    int* np4_x, int* np4_y,
    int* np5_x, int* np5_y,
    int* np6_x, int* np6_y,
    int* np7_x, int* np7_y,
    int* np8_x, int* np8_y,
    int* np9_x, int* np9_y,
    int* np_width, int* np_height
    )
{
    if (full_screen)
    {
        *np1_x = NUMBER_PAD_1_X_FULL;
        *np1_y = NUMBER_PAD_1_Y_FULL;
        *np2_x = NUMBER_PAD_2_X_FULL;
        *np2_y = NUMBER_PAD_2_Y_FULL;
        *np3_x = NUMBER_PAD_3_X_FULL;
        *np3_y = NUMBER_PAD_3_Y_FULL;
        *np4_x = NUMBER_PAD_4_X_FULL;
        *np4_y = NUMBER_PAD_4_Y_FULL;
        *np5_x = NUMBER_PAD_5_X_FULL;
        *np5_y = NUMBER_PAD_5_Y_FULL;
        *np6_x = NUMBER_PAD_6_X_FULL;
        *np6_y = NUMBER_PAD_6_Y_FULL;
        *np7_x = NUMBER_PAD_7_X_FULL;
        *np7_y = NUMBER_PAD_7_Y_FULL;
        *np8_x = NUMBER_PAD_8_X_FULL;
        *np8_y = NUMBER_PAD_8_Y_FULL;
        *np9_x = NUMBER_PAD_9_X_FULL;
        *np9_y = NUMBER_PAD_9_Y_FULL;
        *np_width = NUMBER_PAD_BUTTON_WIDTH_FULL;
        *np_height = NUMBER_PAD_BUTTON_HEIGHT_FULL;
    }
    else
    {
        *np1_x = NUMBER_PAD_1_X;
        *np1_y = NUMBER_PAD_1_Y;
        *np2_x = NUMBER_PAD_2_X;
        *np2_y = NUMBER_PAD_2_Y;
        *np3_x = NUMBER_PAD_3_X;
        *np3_y = NUMBER_PAD_3_Y;
        *np4_x = NUMBER_PAD_4_X;
        *np4_y = NUMBER_PAD_4_Y;
        *np5_x = NUMBER_PAD_5_X;
        *np5_y = NUMBER_PAD_5_Y;
        *np6_x = NUMBER_PAD_6_X;
        *np6_y = NUMBER_PAD_6_Y;
        *np7_x = NUMBER_PAD_7_X;
        *np7_y = NUMBER_PAD_7_Y;
        *np8_x = NUMBER_PAD_8_X;
        *np8_y = NUMBER_PAD_8_Y;
        *np9_x = NUMBER_PAD_9_X;
        *np9_y = NUMBER_PAD_9_Y;
        *np_width = NUMBER_PAD_BUTTON_WIDTH;
        *np_height = NUMBER_PAD_BUTTON_HEIGHT;
    }
}

static int convert_subcell_index_to_marker(int subcell_index_x, int subcell_index_y)
{
    return subcell_index_x + subcell_index_y*3;
}

static gboolean main_window_button_press_callback(GtkWidget* event_box, GdkEventButton* event, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;

    /* Layout */
    int b_top_left_x, b_top_left_y;
    double c_width, c_height;
    int c_3x3_padding_x, c_3x3_padding_y;
    double b_width_with_padding, b_height_with_padding;
    double c_width_with_padding, c_height_with_padding;
    int marker_button_x, marker_button_y;
    int marker_button_width, marker_button_height;
    int submarker_button_x, submarker_button_y;
    int submarker_button_width, submarker_button_height;
    int game_state_label_x, game_state_label_y;
    int game_state_area_x, game_state_area_y;
    int game_state_area_width, game_state_area_height;
    
    configure_layoyt_params(main_window->full_screen,
        &b_top_left_x, &b_top_left_y,
        &c_width, &c_height,
        &c_3x3_padding_x, &c_3x3_padding_y,
        &b_width_with_padding, &b_height_with_padding,
        &c_width_with_padding, &c_height_with_padding,
        &marker_button_x, &marker_button_y,
        &marker_button_width, &marker_button_height,
        &submarker_button_x, &submarker_button_y,
        &submarker_button_width, &submarker_button_height,
        &game_state_label_x, &game_state_label_y,
        &game_state_area_x, &game_state_area_y,
        &game_state_area_width, &game_state_area_height);

    int np1_x, np1_y, np2_x, np2_y, np3_x, np3_y, np4_x, np4_y, np5_x, np5_y,
        np6_x, np6_y, np7_x, np7_y, np8_x, np8_y, np9_x, np9_y;
    int np_width, np_height;
    
    configure_layoyt_params_number_pad(main_window->full_screen,
        &np1_x, &np1_y, &np2_x, &np2_y, &np3_x, &np3_y, &np4_x, &np4_y, &np5_x, &np5_y,
        &np6_x, &np6_y, &np7_x, &np7_y, &np8_x, &np8_y, &np9_x, &np9_y,
        &np_width, &np_height);

    /* Check if clicked game state area */
    if (event->x >= game_state_area_x && event->x < game_state_area_x+game_state_area_width &&
        event->y >= game_state_area_y && event->y < game_state_area_y+game_state_area_height)
    {
        if (get_game_state() == GAME_STATE_PAUSED)
            main_view_continue_game(main_window);
        else if (get_game_state() == GAME_STATE_PLAY)
            main_view_pause_game(main_window);
    }

    /* Allow playing only if game is in playing state */
    if (get_game_state() == GAME_STATE_PLAY)
    {
        /* Check if clicked the board */
        if (event->x >= b_top_left_x && event->x < b_top_left_x+b_width_with_padding &&
            event->y >= b_top_left_y && event->y < b_top_left_y+b_height_with_padding)
        {
            double cell_x = (event->x-b_top_left_x) / c_width_with_padding;
            double cell_y = (event->y-b_top_left_y) / c_height_with_padding;
            
            int cell_index_x = floor(cell_x);
            int cell_index_y = floor(cell_y);
            
            int subcell_index_x = floor((cell_x-cell_index_x)*3);
            int subcell_index_y = floor((cell_y-cell_index_y)*3);
            
            int marker = convert_subcell_index_to_marker(subcell_index_x, subcell_index_y);
            
            /* If played using subcell marking */
            if (get_mark_mode() == MARK_MODE_SUBCELL)
            {
                if (get_play_mode() == PLAY_MODE_MARK_CELL)
                    set_marker(cell_index_x, cell_index_y, marker);
                else
                    toggle_submarker(cell_index_x, cell_index_y, marker);
            }
            else /* Mark cells using number pad */
            {
                if (get_play_mode() == PLAY_MODE_MARK_CELL)
                    set_marker(cell_index_x, cell_index_y, get_mark_mode()-1);
                else
                    toggle_submarker(cell_index_x, cell_index_y, get_mark_mode()-1);
            }

            main_view_redraw(main_window);
        }
        /* If clicked marker button */
        else if (event->x >= marker_button_x && event->x < marker_button_x+marker_button_width &&
            event->y >= marker_button_y && event->y < marker_button_y+marker_button_height)
        {
            if (get_play_mode() != PLAY_MODE_MARK_CELL)
            {
                set_play_mode(PLAY_MODE_MARK_CELL);
                main_view_redraw(main_window);
            }
        }
        /* If clicked submarker button */
        else if (event->x >= submarker_button_x && event->x < submarker_button_x+submarker_button_width &&
            event->y >= submarker_button_y && event->y < submarker_button_y+submarker_button_height)
        {
            if (get_play_mode() != PLAY_MODE_MARK_SUBCELL)
            {
                set_play_mode(PLAY_MODE_MARK_SUBCELL);
                main_view_redraw(main_window);
            }
        }
        /* If clicked number 1 in the number pad */
        else if (event->x >= np1_x && event->x < np1_x+np_width &&
                 event->y >= np1_y && event->y < np1_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_1);
            main_view_redraw(main_window);
        }
        /* If clicked number 2 in the number pad */
        else if (event->x >= np2_x && event->x < np2_x+np_width &&
                 event->y >= np2_y && event->y < np2_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_2);
            main_view_redraw(main_window);
        }
        /* If clicked number 3 in the number pad */
        else if (event->x >= np3_x && event->x < np3_x+np_width &&
                 event->y >= np3_y && event->y < np3_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_3);
            main_view_redraw(main_window);
        }
        /* If clicked number 4 in the number pad */
        else if (event->x >= np4_x && event->x < np4_x+np_width &&
                 event->y >= np4_y && event->y < np4_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_4);
            main_view_redraw(main_window);
        }
        /* If clicked number 5 in the number pad */
        else if (event->x >= np5_x && event->x < np5_x+np_width &&
                 event->y >= np5_y && event->y < np5_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_5);
            main_view_redraw(main_window);
        }
        /* If clicked number 6 in the number pad */
        else if (event->x >= np6_x && event->x < np6_x+np_width &&
                 event->y >= np6_y && event->y < np6_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_6);
            main_view_redraw(main_window);
        }
        /* If clicked number 7 in the number pad */
        else if (event->x >= np7_x && event->x < np7_x+np_width &&
                 event->y >= np7_y && event->y < np7_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_7);
            main_view_redraw(main_window);
        }
        /* If clicked number 8 in the number pad */
        else if (event->x >= np8_x && event->x < np8_x+np_width &&
                 event->y >= np8_y && event->y < np8_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_8);
            main_view_redraw(main_window);
        }
        /* If clicked number 9 in the number pad */
        else if (event->x >= np9_x && event->x < np9_x+np_width &&
                 event->y >= np9_y && event->y < np9_y+np_height)
        {
            toggle_mark_mode(MARK_MODE_NUMBER_PAD_9);
            main_view_redraw(main_window);
        }
        
        /* Check if problem is solved */
        if (is_solved())
        {
            main_view_game_solved(main_window);
        }
    }
    
    /* Returning TRUE means we handled the event, so the signal 
        * emission should be stopped (don't call any further 
        * callbacks that may be connected). Return FALSE
        * to continue invoking callbacks.
        */
    return TRUE;
}

void calculate_cell_padding(int x, int y,
    int c_3x3_padding_x, int c_3x3_padding_y,
    int* padding_x, int* padding_y)
{
    if (x>5) *padding_x = *padding_x + c_3x3_padding_x;
    if (x>2) *padding_x = *padding_x + c_3x3_padding_x;
    if (y>5) *padding_y = *padding_y + c_3x3_padding_y;
    if (y>2) *padding_y = *padding_y + c_3x3_padding_y;
}

gboolean main_window_expose_event_callback(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
    MainWindow *main_window = (MainWindow*)data;

    /* Layout */
    int b_top_left_x, b_top_left_y;
    double c_width, c_height;
    int c_3x3_padding_x, c_3x3_padding_y;
    double b_width_with_padding, b_height_with_padding;
    double c_width_with_padding, c_height_with_padding;
    int marker_button_x, marker_button_y;
    int marker_button_width, marker_button_height;
    int submarker_button_x, submarker_button_y;
    int submarker_button_width, submarker_button_height;
    int game_state_label_x, game_state_label_y;
    int game_state_area_x, game_state_area_y;
    int game_state_area_width, game_state_area_height;
    
    configure_layoyt_params(main_window->full_screen,
        &b_top_left_x, &b_top_left_y,
        &c_width, &c_height,
        &c_3x3_padding_x, &c_3x3_padding_y,
        &b_width_with_padding, &b_height_with_padding,
        &c_width_with_padding, &c_height_with_padding,
        &marker_button_x, &marker_button_y,
        &marker_button_width, &marker_button_height,
        &submarker_button_x, &submarker_button_y,
        &submarker_button_width, &submarker_button_height,
        &game_state_label_x, &game_state_label_y,
        &game_state_area_x, &game_state_area_y,
        &game_state_area_width, &game_state_area_height);

    /* Background */
    gdk_draw_pixbuf(widget->window,
        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
        main_window->background_pixbuf,
        0, 0,
        0, 0,
        gdk_pixbuf_get_width(main_window->background_pixbuf), gdk_pixbuf_get_height(main_window->background_pixbuf),
        GDK_RGB_DITHER_NORMAL, 0, 0);
    
    /* Game state */
    PangoFontDescription* font;
    PangoContext* context;
    PangoLayout* layout;

    font =  pango_font_description_from_string(GAME_STATE_FONT);
    context = gtk_widget_create_pango_context(widget);
    layout  = pango_layout_new(context);
    g_object_unref(context);
    
    pango_layout_set_font_description(layout, font);
    char game_state_text[50];
    struct tm *l_time;
    time_t duration = get_game_duration();
    l_time = gmtime(&duration);
    switch (get_game_state())
    {
        case GAME_STATE_PLAY:
            strftime(game_state_text, sizeof(game_state_text), _("%H:%M:%S"), l_time);            
            break;
        case GAME_STATE_PAUSED:
            strftime(game_state_text, sizeof(game_state_text), _("Paused (%H:%M:%S)"), l_time);            
            break;
        case GAME_STATE_ENDED:
            strftime(game_state_text, sizeof(game_state_text), _("Solved (%H:%M:%S)"), l_time);            
            break;
    }
    pango_layout_set_text(layout, game_state_text, -1);
    gdk_draw_layout(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
        game_state_label_x, game_state_label_y,
        layout);
    g_object_unref(layout);

    /* If game is paused, don't draw any game information on the board */        
    if (get_game_state() != GAME_STATE_PAUSED)
    {
        /* Buttons */
        if (get_play_mode() == PLAY_MODE_MARK_CELL)
        {
            gdk_draw_pixbuf(widget->window,
                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                main_window->marker_button_selected,
                0, 0,
                marker_button_x, marker_button_y,
                gdk_pixbuf_get_width(main_window->marker_button_selected), gdk_pixbuf_get_height(main_window->marker_button_selected),
                GDK_RGB_DITHER_NORMAL, 0, 0);
        }
        else
        {
            gdk_draw_pixbuf(widget->window,
                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                main_window->marker_button_unselected,
                0, 0,
                marker_button_x, marker_button_y,
                gdk_pixbuf_get_width(main_window->marker_button_unselected), gdk_pixbuf_get_height(main_window->marker_button_unselected),
                GDK_RGB_DITHER_NORMAL, 0, 0);
        }
        if (get_play_mode() == PLAY_MODE_MARK_SUBCELL)
        {
            gdk_draw_pixbuf(widget->window,
                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                main_window->submarker_button_selected,
                0, 0,
                submarker_button_x, submarker_button_y,
                gdk_pixbuf_get_width(main_window->submarker_button_selected), gdk_pixbuf_get_height(main_window->submarker_button_selected),
                GDK_RGB_DITHER_NORMAL, 0, 0);
        }
        else
        {
            gdk_draw_pixbuf(widget->window,
                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                main_window->submarker_button_unselected,
                0, 0,
                submarker_button_x, submarker_button_y,
                gdk_pixbuf_get_width(main_window->submarker_button_unselected), gdk_pixbuf_get_height(main_window->submarker_button_unselected),
                GDK_RGB_DITHER_NORMAL, 0, 0);
        }
        
        /* Number pad */
        if (get_mark_mode() != MARK_MODE_SUBCELL)
        {
            int np1_x, np1_y, np2_x, np2_y, np3_x, np3_y, np4_x, np4_y, np5_x, np5_y,
                np6_x, np6_y, np7_x, np7_y, np8_x, np8_y, np9_x, np9_y;
            int np_width, np_height;
            configure_layoyt_params_number_pad(main_window->full_screen,
                &np1_x, &np1_y, &np2_x, &np2_y, &np3_x, &np3_y, &np4_x, &np4_y, &np5_x, &np5_y,
                &np6_x, &np6_y, &np7_x, &np7_y, &np8_x, &np8_y, &np9_x, &np9_y,
                &np_width, &np_height);

            int np_x = 0;
            int np_y = 0;
            switch (get_mark_mode())
            {
                case MARK_MODE_NUMBER_PAD_1: np_x = np1_x; np_y = np1_y; break;
                case MARK_MODE_NUMBER_PAD_2: np_x = np2_x; np_y = np2_y; break;
                case MARK_MODE_NUMBER_PAD_3: np_x = np3_x; np_y = np3_y; break;
                case MARK_MODE_NUMBER_PAD_4: np_x = np4_x; np_y = np4_y; break;
                case MARK_MODE_NUMBER_PAD_5: np_x = np5_x; np_y = np5_y; break;
                case MARK_MODE_NUMBER_PAD_6: np_x = np6_x; np_y = np6_y; break;
                case MARK_MODE_NUMBER_PAD_7: np_x = np7_x; np_y = np7_y; break;
                case MARK_MODE_NUMBER_PAD_8: np_x = np8_x; np_y = np8_y; break;
                case MARK_MODE_NUMBER_PAD_9: np_x = np9_x; np_y = np9_y; break;
            }
            
            int ofs_x, ofs_y;
            ofs_x = floor(np_width*0.5 - gdk_pixbuf_get_width(main_window->cell_correct_pixbuf[get_mark_mode()-1])*0.5);
            ofs_y = floor(np_height*0.5 - gdk_pixbuf_get_height(main_window->cell_correct_pixbuf[get_mark_mode()-1])*0.5);
            
            gdk_draw_pixbuf(widget->window,
                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                main_window->cell_correct_pixbuf[get_mark_mode()-1],
                0, 0,
                np_x+ofs_x, np_y+ofs_y,
                gdk_pixbuf_get_width(main_window->cell_correct_pixbuf[get_mark_mode()-1]),
                gdk_pixbuf_get_height(main_window->cell_correct_pixbuf[get_mark_mode()-1]),
                GDK_RGB_DITHER_NORMAL, 0, 0);
        }

        /* Cell data */
        int x, y;
        for (y=0; y<BOARD_DIM; y++)
            for (x=0; x<BOARD_DIM; x++)
            {
                int padding_x = 0, padding_y = 0;
                calculate_cell_padding(x, y, c_3x3_padding_x, c_3x3_padding_y, &padding_x, &padding_y);

                t_cell cell = get_cell(x, y);
                if (cell.marker != CELL_NOT_MARKED && (cell.correct || !main_window->show_incorrect_cells))
                {
                    gdk_draw_pixbuf(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        main_window->cell_correct_pixbuf[cell.marker],
                        0, 0,
                        floor(b_top_left_x + x*c_width + padding_x),
                        floor(b_top_left_y + y*c_height + padding_y),
                        gdk_pixbuf_get_width(main_window->cell_correct_pixbuf[cell.marker]),
                        gdk_pixbuf_get_height(main_window->cell_correct_pixbuf[cell.marker]),
                        GDK_RGB_DITHER_NORMAL, 0, 0);
                }
                else if (cell.marker != CELL_NOT_MARKED && !cell.correct)
                {
                    gdk_draw_pixbuf(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        main_window->cell_incorrect_pixbuf[cell.marker],
                        0, 0,
                        floor(b_top_left_x + x*c_width + padding_x),
                        floor(b_top_left_y + y*c_height + padding_y),
                        gdk_pixbuf_get_width(main_window->cell_incorrect_pixbuf[cell.marker]),
                        gdk_pixbuf_get_height(main_window->cell_incorrect_pixbuf[cell.marker]),
                        GDK_RGB_DITHER_NORMAL, 0, 0);
                }
                else /* submarkers if any */
                {
                    int subcell_index;
                    for (subcell_index=0; subcell_index<BOARD_DIM; subcell_index++)
                    {
                        int subcell = get_subcell_marker(x, y, subcell_index);
                        if (subcell == CELL_MARKED)
                        {
                            int padding_x = 0, padding_y = 0;
                            calculate_cell_padding(x, y, c_3x3_padding_x, c_3x3_padding_y, &padding_x, &padding_y);
                            
                            gdk_draw_pixbuf(widget->window,
                                widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                                main_window->subcell_pixbuf[subcell_index],
                                0, 0,
                                floor(b_top_left_x + x*c_width + padding_x),
                                floor(b_top_left_y + y*c_height + padding_y),
                                gdk_pixbuf_get_width(main_window->subcell_pixbuf[subcell_index]),
                                gdk_pixbuf_get_height(main_window->subcell_pixbuf[subcell_index]),
                                GDK_RGB_DITHER_NORMAL, 0, 0);
                        }
                    }
                }
                
                /* Predefine tag */
                if (cell.predefined)
                {
                    gdk_draw_pixbuf(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        main_window->cell_predefined_pixbuf,
                        0, 0,
                        floor(b_top_left_x + x*c_width + padding_x),
                        floor(b_top_left_y + y*c_height + padding_y),
                        gdk_pixbuf_get_width(main_window->cell_predefined_pixbuf),
                        gdk_pixbuf_get_height(main_window->cell_predefined_pixbuf),
                        GDK_RGB_DITHER_NORMAL, 0, 0);
                }
                
                /* Subcell dots only for not predefined cells and the cells that are
                   not yet solved */
                if (main_window->subcell_dots && !cell.predefined && !cell.correct)
                {
                    gdk_draw_pixbuf(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        main_window->subcell_dots_pixbuf,
                        0, 0,
                        floor(b_top_left_x + x*c_width + padding_x),
                        floor(b_top_left_y + y*c_height + padding_y),
                        gdk_pixbuf_get_width(main_window->subcell_dots_pixbuf),
                        gdk_pixbuf_get_height(main_window->subcell_dots_pixbuf),
                        GDK_RGB_DITHER_NORMAL, 0, 0);
                }
            }
    }
    
    return TRUE;
}

/* Callback for hardware keys */
gboolean main_window_key_press_callback(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    MainWindow* main_window = (MainWindow*)data;

    switch (event->keyval)
    {
        case GDK_F6: /* Full screen key */
            if (main_window->full_screen)
                gtk_check_menu_item_set_active(GTK_MENU_ITEM(main_window->item_show_full_screen), FALSE);
            else
                gtk_check_menu_item_set_active(GTK_MENU_ITEM(main_window->item_show_full_screen), TRUE);
            return TRUE;
    }

    return FALSE;
}

MainWindow* main_window_new(AppData* appdata)
{
    /* Initialize MainWindow */
    MainWindow* result = g_new0(MainWindow, 1);
    result->appdata = appdata;
    result->full_screen = FALSE;
    result->subcell_dots = FALSE;
    
    /* Create the main window */
    HildonWindow* main_window = HILDON_WINDOW(hildon_window_new());
    appdata->main_window = main_window;
    hildon_program_add_window(appdata->program, main_window);

    /* Event receiving */
    result->main_view_event_box = gtk_event_box_new();
    gtk_container_add(GTK_CONTAINER(main_window), GTK_WIDGET(result->main_view_event_box));
    g_signal_connect(G_OBJECT(result->main_view_event_box), "button_press_event",
        G_CALLBACK(main_window_button_press_callback), result);
    g_signal_connect(G_OBJECT(main_window), "key_press_event",
        G_CALLBACK(main_window_key_press_callback), result);
        
    /* Pixbufs for drawing */
    load_pixmaps(result);
    
    /* Buffer for drawing */
    GtkWidget* drawing_area = gtk_drawing_area_new();
    result->drawing_area = drawing_area;
    gtk_widget_set_size_request(drawing_area, MAIN_VIEW_WIDTH, MAIN_VIEW_HEIGHT);
    g_signal_connect(G_OBJECT(drawing_area), "expose_event",
        G_CALLBACK(main_window_expose_event_callback), result);
    gtk_container_add(GTK_CONTAINER(result->main_view_event_box), GTK_WIDGET(drawing_area));

    /* Create menu for HildonWindow */
    create_menu(result, appdata);
    
    return result;
}     

void main_window_destroy(MainWindow* main_window)
{
    g_free(main_window);
}
