/*
 * Tomiku
 *
 * Project homepage: http://tomiku.garage.maemo.org
 * Author homepage: http://www.aivokyyti.net
 *
 * Copyright (C) 2007 Tomi Pihlainen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA. 
 *
 */

#include <stdlib.h>
#include <time.h>
#include "board.h"
#include "sudoku-sensei/sudoku-pub.h"

#define _(String) gettext(String)

#include <config.h>

/* Playing mode */
int play_mode;

/* Marking mode */
int mark_mode;

/* Cell data */
t_cell cell_data[BOARD_DIM][BOARD_DIM];

/* Board data */
int problem_data[BOARD_DIM][BOARD_DIM];
int solution_data[BOARD_DIM][BOARD_DIM];

/* Forward declarations */
int my_random (int num);

/* Interface functions */

void init_cell_data()
{
    int x, y;
    for (y=0; y<BOARD_DIM; y++)
        for (x=0; x<BOARD_DIM; x++)
        {
            if (problem_data[x][y] != CELL_NOT_MARKED)
            {
                cell_data[x][y].predefined = TRUE;
                cell_data[x][y].marker = problem_data[x][y];
                cell_data[x][y].correct = TRUE;
            }
            else
            {
                cell_data[x][y].predefined = FALSE;
                cell_data[x][y].marker = CELL_NOT_MARKED;
                cell_data[x][y].correct = FALSE;
            }
            
            int sm;
            for (sm=0; sm<BOARD_DIM; sm++)
                cell_data[x][y].submarker[sm] = CELL_NOT_MARKED;
        }
}

void set_play_mode(int mode)
{
    play_mode = mode;
}

int get_play_mode()
{
    return play_mode;
}

void toggle_mark_mode(int mode)
{
    if (mark_mode == mode)
        mark_mode = MARK_MODE_SUBCELL;
    else
        mark_mode = mode;
}

int get_mark_mode()
{
    return mark_mode;
}

t_cell get_cell(int x, int y)
{
    return cell_data[x][y];
}

int get_subcell_marker(int x, int y, int subcell_index)
{
    return cell_data[x][y].submarker[subcell_index];
}

int set_marker(int x, int y, int cell_value)
{
    if (cell_data[x][y].predefined)
    {
        return MARK_FAILED_PREDEFINED;
    }
    
    if (cell_data[x][y].marker == cell_value)
    {
        cell_data[x][y].marker = CELL_NOT_MARKED;
        cell_data[x][y].correct = FALSE;
    }
    else
    {
        cell_data[x][y].marker = cell_value;
        cell_data[x][y].correct = (cell_value == solution_data[x][y]);
    }
        
    return MARK_OK;
}

void toggle_submarker(int x, int y, int subcell_index)
{
    if (cell_data[x][y].submarker[subcell_index] == CELL_NOT_MARKED)
        cell_data[x][y].submarker[subcell_index] = CELL_MARKED;
    else
        cell_data[x][y].submarker[subcell_index] = CELL_NOT_MARKED;
}

int is_solved()
{
    int x, y;
    for (y=0; y<BOARD_DIM; y++)
	    for (x=0; x<BOARD_DIM; x++)
	        if (!cell_data[x][y].correct)
	            return 0;
    return 1;
}

int create_board()
{
    t_board_p b;
    e_state state;
    int i, j, c, s, n, m;
    int cells[BOARD_DIM*BOARD_DIM], symbols[BOARD_DIM*BOARD_DIM];

    b = ConstructBasicBoard (BOARD_DIM/3,       /* (3*3)*(3*3) == 81 cells */
                             FALSE);  /* No diagonals */

    if (b)
    {
        srand ((unsigned)time(NULL));  /* Use clock as random seed */

        n = 0;   /* n: number of symbols fixed */

        do
        {
            do                                      /* Choose a cell */
            c = my_random (BOARD_DIM*BOARD_DIM);  /* at random */
            while (GetNumPossValuesOfCell(b,c)==1); /* c: cell number */

            s = my_random (                       /* Choose one of */
                GetNumPossValuesOfCell (b, c)); /* the possible */
                                            /* values */

            m = GetPossValuesOfCell (b, c);  /* Mask with possible */
                                       /* values */
            for (i=j=0; i<BOARD_DIM; i++)
                if ((1<<i) & m)        /* Search for the chosen */
                {                      /* possible value in the mask */
                    if (j==s)
                    {
                        s = i;             /* s: chosen symbol number */
                        break;
                    }
                    j++;
                }

            SetSymbolInCell (b, s, c);  /* Set it, and then */
                                      /* try to solve */
            Solve (b, 0, FALSE);

            state = GetState (b);

            if (state==impossible)   /* This seldom happens, but.. */
            {
                CleanBoard (b);        /* Start from scratch */
                n = 0;
                continue;
            }

            symbols[n] = s;   /* The choice was right */
            cells[n] = c;     /* Store cell-symbol pair */
            n++;
        }
        while (state!=solved);   /* Repeat until solved */

        /* solution */
        for (j=0; j<BOARD_DIM; j++)
        {
    	    for (i=0; i<BOARD_DIM; i++)
            {
                int cell_no = GetCellNumber(b, i, j);
                solution_data[i][j] = GetSymbolOfCell(b, cell_no);
            }
        }

        /* problem */
        CleanBoard(b); /* Clean the board and set all the n symbols again */
        for (i=0; i<n; i++)
        {
            SetSymbolInCell(b, symbols[i], cells[i]);
        }
        for (j=0; j<BOARD_DIM; j++)
        {
    	    for (i=0; i<BOARD_DIM; i++)
            {
                int cell_no = GetCellNumber(b, i, j);
                problem_data[i][j] = GetSymbolOfCell(b, cell_no);
            }
        }

        DestroyBoard(b);
    
        init_cell_data();

        /* For debugging purposes.
        
        int x, y;
        g_print("board.c: Problem:\n");
        for (y=0; y<BOARD_DIM; y++)
        {
        	for (x=0; x<BOARD_DIM; x++)
            {
                g_print("%d ", problem_data[x][y]);
        	}
            g_print("\n");
        }

        g_print("board.c: Solution:\n");
        for (y=0; y<BOARD_DIM; y++)
        {
            for (x=0; x<BOARD_DIM; x++)
            {
                g_print("%d ", solution_data[x][y]);
            }
            g_print("\n");
        }
        */
    }

    return 0;
}

/* Private functions */

int my_random (int num)
{
  double d;
  int r;

  d = rand () / (double) RAND_MAX;
  r = (int) (d*num);

  return r<0 ? 0 : r>=num ? num-1 : r;
}
