/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/* 
 * File:   utils.hpp
 * Author: ivan
 *
 * Created on April 22, 2009, 2:31 PM
 */

#pragma once


#include <map>
#include <deque>
#include <string>

#include <glibmm/dispatcher.h>

#include <ting/debug.hpp>
#include <ting/math.hpp>
#include <ting/types.hpp>
#include <ting/Singleton.hpp>
#include <ting/Signal.hpp>
#include <ting/Ptr.hpp>
#include <ting/Thread.hpp>
#include <ting/Ref.hpp>

#include <aumiks/synth/synth.hpp>

#include "stdafx.hpp"


enum E_Timbre{
	SINE_PLUS_ADSR,
	PLUCKED_STRING,
	TROMBONE,
	CLARINET,
	VIOLIN,
	PIANO,
	SAWTOOTH,

	NUM_TIMBRES
};



std::string TimbreTypeToName(E_Timbre timbre);



E_Timbre TimbreNameToType(std::string name);



template <typename T> inline float CalcFreq(T baseFreq, T numOctavesToShift){
	//formula: freq = baseFreq * (2 ^ numOctavesToShift);
	return baseFreq * ting::Exp(numOctavesToShift * ting::DLnOf2<T>());
}



inline float PitchToFreq(ting::uint pitch){
	return CalcFreq(27.5f, float(pitch) / 12);
}



class GlibDispatcher : public ting::Singleton<GlibDispatcher>{
	typedef std::map<ting::uint, ting::Ptr<ting::Signal0> > T_SigMap;
	typedef T_SigMap::iterator T_SigIter;
	T_SigMap sigMap;

	ting::Mutex mutex;

	ting::uint nextId;

	Glib::Dispatcher glibDispatcher;
	std::deque<ting::uint> glibDispatcherArg;

	void GlibDispatcherHandler();
	
public:

	GlibDispatcher();

	~GlibDispatcher(){
		ASSERT(this->sigMap.size() == 0)
	}



	ting::uint Connect(ting::Ptr<ting::Signal0> signal);

	void Disconnect(ting::uint id);

	void Emit(ting::uint id);
};



class MonotonicInstrumentHolder{
	ting::Ref<aumiks::Instrument> instr;//current instrument
	ting::Ref<aumiks::Instrument::Channel> ch;
public:

	void ApplyTimbre(E_Timbre timbre);
	E_Timbre GetCurrentTimbreType()const;

	void Stop();

	bool IsChannelValid()const{
		return this->ch.IsValid();
	}

	void RenewChannel(){
		ASSERT(this->instr)
		this->ch = this->instr->CreateChannel();
	}

	void SetVolume(ting::u8 vol){
		ASSERT(this->ch)
		this->ch->SetVolume(vol);
	}

	void SetFrequency(float freq){
		ASSERT(this->ch)
		this->ch->SetFrequency(freq);
	}

	void Play(){
		ASSERT(this->ch)
		this->ch->Play();
	}
};
