/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */


#include <gtkmm/box.h>
#include <gtkmm/separator.h>
#include <gtkmm/button.h>
#include <gtkmm/label.h>
#include <gtkmm/frame.h>
#include <gtkmm/main.h>

#include <ting/Ptr.hpp>
#include <ting/Thread.hpp>

#include "stdafx.hpp"
#include "NetworkSettingsDialog.hpp"
#include "RegisterDialog.hpp"
#include "ting/Singleton.hpp"
#include "Preferences.hpp"
#include "utils.hpp"



using namespace ting;



NetworkSettingsDialog::NetworkSettingsDialog(Gtk::Window &parent) :
		Gtk::Dialog("Network settings", parent, true),
		broadcastMusicCheckbox("Broadcast music you play"),
		autologinCheckbox("Log in automatically"),
		loginButton("Log in"),
		logoutButton("Log out")
{
	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): enter" << std::endl)

	Gtk::VBox *vbox = this->get_vbox();
	vbox->set_spacing(5);

	this->broadcastMusicCheckbox.set_size_request(-1, DFingerFriendlyMinSize());
	this->broadcastMusicCheckbox.set_active(Preferences::Inst().GetBroadcast());
	vbox->add(this->broadcastMusicCheckbox);

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): broadcast checkbox added" << std::endl)

	//Add account settings controls
	{
		Gtk::Frame *f = new Gtk::Frame("Account settings");
		vbox->add(*Gtk::manage(f));

		Gtk::VBox *vb = new Gtk::VBox();
		f->add(*Gtk::manage(vb));

		//add login/password fields
		{
			Gtk::HBox *hbox = new Gtk::HBox();
			vb->add(*Gtk::manage(hbox));
			{
				Gtk::VBox *v = new Gtk::VBox();
				hbox->add(*Gtk::manage(v));

				v->add(*Gtk::manage(new Gtk::Label("Username:", Gtk::ALIGN_LEFT)));

				this->usernameEntryField.set_text(Preferences::Inst().GetUsername());
				v->add(this->usernameEntryField);
			}
			{
				Gtk::VBox *v = new Gtk::VBox();
				hbox->add(*Gtk::manage(v));

				v->add(*Gtk::manage(new Gtk::Label("Password:", Gtk::ALIGN_LEFT)));

				//NOTE: property accessors are disabled on maemo, so use
				//set_property() on maemo
#ifdef GLIBMM_PROPERTIES_ENABLED
				this->passwordEntryField.property_visibility().set_value(false);//set password mode
#else
				this->passwordEntryField.set_property("visibility", false);//set password mode
#endif

				this->passwordEntryField.set_text(Preferences::Inst().GetPassword());
				v->add(this->passwordEntryField);
			}
			{
				Gtk::VBox *v = new Gtk::VBox();
				hbox->add(*Gtk::manage(v));

				v->add(*Gtk::manage(new Gtk::Label("Current status:", Gtk::ALIGN_LEFT)));

				v->add(this->statusLabel);
			}
		}

		{
			Gtk::HBox *h = new Gtk::HBox();
			vb->add(*Gtk::manage(h));

			this->autologinCheckbox.set_size_request(-1, DFingerFriendlyMinSize());
			this->autologinCheckbox.set_active(Preferences::Inst().GetAutologin());
			h->add(this->autologinCheckbox);

			h->add(this->loginButton);
			this->loginButton.signal_clicked().connect(
					sigc::mem_fun(*this, &NetworkSettingsDialog::OnLoginPressed)
				);

			h->add(this->logoutButton);
			this->logoutButton.signal_clicked().connect(
					sigc::mem_fun(*this, &NetworkSettingsDialog::OnLogoutPressed)
				);
		}
	}//~account settings

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): account settings controls added" << std::endl)

	{
		Gtk::Frame *f = new Gtk::Frame("Create new account");
		vbox->add(*Gtk::manage(f));

		Gtk::VBox *vb = new Gtk::VBox();
		f->add(*Gtk::manage(vb));

		vb->add(*Gtk::manage(new Gtk::Label(
				"If you don't have an account you can create a new one:"
			)));

		{
			Gtk::Button *b = new Gtk::Button("Create new account");
			vb->add(*Gtk::manage(b));

			b->signal_clicked().connect(
					sigc::mem_fun(*this, &NetworkSettingsDialog::RegisterNewAccount)
				);
		}
	}

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): register account controls added" << std::endl)

	//Add Ok and Cancel buttons
	{
		Gtk::Button *b = new Gtk::Button("Ok");
		TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): OK button created" << std::endl)

		Gtk::ButtonBox* bb = this->get_action_area();

		TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): action area got" << std::endl)

		ASSERT(bb)

		bb->add(*Gtk::manage(b));

		TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): OK button added" << std::endl)

		b->signal_clicked().connect(
				sigc::mem_fun(*this, &NetworkSettingsDialog::OnOkPressed)
			);
	}
	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): OK button connected" << std::endl)

	this->add_button("Cancel", Gtk::RESPONSE_CANCEL);

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): Cancel button added" << std::endl)
	
	this->show_all_children();//Show all window child widgets

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): all childrens shown" << std::endl)

	this->SetButtonState(MainThread::OFFLINE);//set offline initially

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): connecting dispatcher" << std::endl)

	//connect dispatcher
	{
		Ptr<Signal0> sig(new Signal0());
		sig->Connect(this, &NetworkSettingsDialog::DispatcherHandler);
		this->dispatcherConnection = GlibDispatcher::Inst().Connect(sig);
	}

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): adding network state listener" << std::endl)

	//start listening network state
	MainThread::Inst().AddNetworkStateListener_ts(this);

	TRACE(<< "NetworkSettingsDialog::NetworkSettingsDialog(): exit" << std::endl)
}



NetworkSettingsDialog::~NetworkSettingsDialog(){
	//After removing the listener we can be sure
	//that no notification will occur.
	MainThread::Inst().RemoveNetworkStateListener_ts(this);

	//Disconnecting the dispatcher.
	//As soon as Disconnect() function returns it guarantees that
	//dispatcher handler will not be called. This is the thing
	//that could not be acheived when using standard Glib::Dispatcher!
	GlibDispatcher::Inst().Disconnect(this->dispatcherConnection);

	//NOTE: it is possible that some arguments was not handled, so,
	//do not assert for == 0:
	//ASSERT(this->dispatcherArg.size() == 0)
}



void NetworkSettingsDialog::RegisterNewAccount(){
	RegisterDialog d(*this);
	d.set_deletable(false);

	switch(d.run()){
		case Gtk::RESPONSE_OK:
			this->usernameEntryField.set_text(d.username);
			this->passwordEntryField.set_text(d.password);
			break;
		default:
			//do nothing
			break;
	}

	d.hide();
}



void NetworkSettingsDialog::OnOkPressed(){
	//save settings
	Preferences::Inst().SetUsername(this->usernameEntryField.get_text());
	Preferences::Inst().SetPassword(this->passwordEntryField.get_text());
	Preferences::Inst().SetAutologin(this->autologinCheckbox.get_active());
	Preferences::Inst().SetBroadcast(this->broadcastMusicCheckbox.get_active());

	//check if autologin is now on and send connect request
	if(Preferences::Inst().GetAutologin()){
		//It is ok to send connect request if already connected,
		//the request will be ignored.
		MainThread::Inst().Connect_ts();
	}

	this->response(Gtk::RESPONSE_OK);
}



void NetworkSettingsDialog::OnLoginPressed(){
	//save username and password
	Preferences::Inst().SetUsername(this->usernameEntryField.get_text());
	Preferences::Inst().SetPassword(this->passwordEntryField.get_text());
	
	MainThread::Inst().Connect_ts();
}



void NetworkSettingsDialog::OnLogoutPressed(){
	MainThread::Inst().Disconnect_ts();
}



//override
void NetworkSettingsDialog::OnStateChange(MainThread::ENetworkState newState){
	{
		ting::Mutex::Guard mutexGuard(this->dispatcherArgMutex);
		this->dispatcherArg.push_back(newState);
	}
	GlibDispatcher::Inst().Emit(this->dispatcherConnection);
	return;
}



void NetworkSettingsDialog::DispatcherHandler(){
	TRACE(<< "NetworkSettingsDialog::DispatcherHandler(): invoked" << std::endl)
	MainThread::ENetworkState st;
	{
		ting::Mutex::Guard mutexGuard(this->dispatcherArgMutex);
		ASSERT(this->dispatcherArg.size() > 0)
		st = this->dispatcherArg.front();
		this->dispatcherArg.pop_front();
	}
	TRACE(<< "NetworkSettingsDialog::DispatcherHandler(): st = " << st << std::endl)
	this->SetButtonState(st);
}



void NetworkSettingsDialog::SetButtonState(MainThread::ENetworkState st){
	switch(st){
		default:
		case MainThread::OFFLINE_LOGIN_FAILED:
		case MainThread::OFFLINE:
			this->statusLabel.set_text(DStrOffline());
			this->loginButton.set_sensitive(true);
			this->loginButton.show();
			this->logoutButton.hide();
			break;
		case MainThread::CONNECTING:
			this->statusLabel.set_text(DStrConnecting());
			this->loginButton.set_sensitive(false);
			this->logoutButton.set_sensitive(false);
			break;
		case MainThread::CONNECTED:
			this->statusLabel.set_text(DStrConnected());
			this->loginButton.set_sensitive(false);
			this->logoutButton.set_sensitive(false);
			break;
		case MainThread::LOGGING_IN:
			this->statusLabel.set_text(DStrLoggingIn());
			this->loginButton.set_sensitive(false);
			this->logoutButton.set_sensitive(false);
			break;
		case MainThread::LOGGED_IN:
			this->statusLabel.set_text(DStrLoggedIn());
			this->logoutButton.set_sensitive(true);
			this->logoutButton.show();
			this->loginButton.hide();
			break;
	}
}



